AJS.test.require("com.atlassian.jira.jira-projects-issue-navigator:application-test", function() {
    "use strict";

    require([
        "jira/projectissuenavigator/services/filters",
        "jira/projectissuenavigator/libs/underscore",
        "jira/util/data/meta",
        "wrm/data"
    ], function(
        Filters,
        _,
        Meta,
        WrmData
    ) {
        var WRM_PROJECT_FILTERS_KEY = "com.atlassian.jira.jira-projects-issue-navigator:project-filters";
        var WRM_CAN_MANAGE_FILTERS_KEY = "com.atlassian.jira.jira-projects-issue-navigator:can-manage-filters";
        var WRM_GENERIC_FILTERS_KEY = "com.atlassian.jira.jira-projects-issue-navigator:generic-filters";
        var WRM_FILTER_PRIORITY_KEY = "com.atlassian.jira.jira-projects-issue-navigator:default-filter-priority";
        var ALL_FILTERS = ['allissues', 'myopenissues', 'allopenissues', 'reportedbyme', 'addedrecently', 'resolvedrecently', 'updatedrecently', 'recentlyviewed', 'doneissues'];
        var JQLs = {
            allissues: 'project = "DEMO" ORDER BY priority DESC, updated DESC',
            myopenissues: 'project = "DEMO" AND assignee = currentUser() AND resolution = Unresolved ORDER BY priority DESC, updated DESC',
            allopenissues: 'project = "DEMO" AND resolution = Unresolved ORDER BY priority DESC, updated DESC',
            reportedbyme: 'project = "DEMO" AND reporter = currentUser() ORDER BY created DESC',
            addedrecently: 'project = "DEMO" AND created >= -1w ORDER BY created DESC',
            resolvedrecently: 'project = "DEMO" AND resolutiondate >= -1w ORDER BY updated DESC',
            updatedrecently: 'project = "DEMO" AND updated >= -1w ORDER BY updated DESC',
            recentlyviewed: 'project = "DEMO" AND issuekey in issueHistory() ORDER BY lastViewed DESC',
            doneissues: 'project = "DEMO" AND statusCategory = Done ORDER BY updated DESC'
        };

        var GENERIC_FILTERS = [
            {
                id: 'allissues',
                jql: 'project = "{0}" ORDER BY {1}',
                defaultOrderby: 'priority DESC, updated DESC',
                label: AJS.I18n.getText("common.filters.all"),
                requiresUser: false,
                fields: [],
                supportsInlineIssueCreate: true
            },
            {
                id: 'myopenissues',
                jql: 'project = "{0}" AND assignee = currentUser() AND resolution = Unresolved ORDER BY {1}',
                defaultOrderby: 'priority DESC, updated DESC',
                label: AJS.I18n.getText("issue.nav.filters.my.open.issues"),
                requiresUser: true,
                fields: ['assignee', 'resolution'],
                supportsInlineIssueCreate: true
            },
            {
                id: 'allopenissues',
                jql: 'project = "{0}" AND resolution = Unresolved ORDER BY {1}',
                defaultOrderby: 'priority DESC, updated DESC',
                label: AJS.I18n.getText("issue.nav.filters.open.issues"),
                fields: ['resolution'],
                supportsInlineIssueCreate: true
            },
            {
                id: 'reportedbyme',
                jql: 'project = "{0}" AND reporter = currentUser() ORDER BY {1}',
                defaultOrderby: 'created DESC',
                label: AJS.I18n.getText("common.filters.reportedbyme"),
                requiresUser: true,
                fields: ['reporter'],
                supportsInlineIssueCreate: true
            },
            {
                id: 'addedrecently',
                jql: 'project = "{0}" AND created >= -1w ORDER BY {1}',
                defaultOrderby: 'created DESC',
                label: AJS.I18n.getText("issue.nav.filters.created.recently"),
                fields: ['created'],
                supportsInlineIssueCreate: true
            },
            {
                id: 'resolvedrecently',
                jql: 'project = "{0}" AND resolutiondate >= -1w ORDER BY {1}',
                defaultOrderby: 'updated DESC',
                label: AJS.I18n.getText("common.filters.resolvedrecently"),
                fields: ['resolutiondate']
            },
            {
                id: 'updatedrecently',
                jql: 'project = "{0}" AND updated >= -1w ORDER BY {1}',
                defaultOrderby: 'updated DESC',
                label: AJS.I18n.getText("common.filters.updatedrecently"),
                fields: ['updated'],
                supportsInlineIssueCreate: true
            },
            {
                id: 'recentlyviewed',
                jql: 'project = "{0}" AND issuekey in issueHistory() ORDER BY {1}',
                defaultOrderby: 'lastViewed DESC',
                label: AJS.I18n.getText("issue.nav.filters.recently.viewed"),
                fields: ['issuekey'],
                supportsInlineIssueCreate: true
            },
            {
                id: 'doneissues',
                jql: 'project = "{0}" AND statusCategory = Done ORDER BY {1}',
                defaultOrderby: 'updated DESC',
                label: AJS.I18n.getText("issue.nav.filters.done.issues"),
                fields: ['statusCategory'],
                supportsInlineIssueCreate: false
            }
        ];
        var PROJECT_FILTERS = [
            {
                id: "filter1",
                jql: 'project = "{0}" ORDER BY {1}',
                defaultOrderby: "priority DESC, updated DESC",
                label: 'Filter 1',
                requiresUser: true
            },
            {
                id: "filter2",
                jql: 'project = "{0}" ORDER BY {1}',
                defaultOrderby: "priority DESC, updated DESC",
                label: 'Filter 2',
                requiresUser: false
            }
        ];

        module("jira/projectissuenavigator/services/filters", {
            setup: function() {
                this.sandbox = sinon.sandbox.create();
                this.sandbox.stub(Meta, "get");
                this.sandbox.stub(WrmData, "claim");
                WrmData.claim.withArgs(WRM_FILTER_PRIORITY_KEY).returns(['allopenissues', 'allissues']);
                WrmData.claim.withArgs(WRM_GENERIC_FILTERS_KEY).returns(GENERIC_FILTERS);
            },

            teardown: function() {
                this.sandbox.restore();
            },

            withFilters: function(canManage) {
                var claimStub = WrmData.claim;
                claimStub.withArgs(WRM_CAN_MANAGE_FILTERS_KEY).returns(canManage);
                claimStub.withArgs(WRM_PROJECT_FILTERS_KEY).returns(PROJECT_FILTERS);
            }

        });

        test("It provides the list of default filters", function() {
            Filters.initialize();

            deepEqual(_.pluck(Filters.getAvailableFilters()[0], "id"), ALL_FILTERS);
        });

        test("It provides a link to Manage Filters if there are project filters", function() {
            this.withFilters(true);
            Filters.initialize();

            deepEqual(_.pluck(Filters.getAvailableFilters(true)[2], "id"), ["manage"]);
        });

        test("It doesn't provide a link to Manage Filters if the current user can't manage them", function() {
            this.withFilters(false);
            Filters.initialize();

            deepEqual(_.pluck(Filters.getAvailableFilters(true)[2], "id"), []);
        });

        test("It does not provides a link to Manage Filters if there are not project filters", function() {
            WrmData.claim.withArgs(WRM_PROJECT_FILTERS_KEY).returns(undefined);
            Filters.initialize();

            equal(Filters.getAvailableFilters(true).length, 1);
        });

        test("It provides the 'all' filter", function() {
            Filters.initialize();

            equal(Filters.getAllIssuesFilterId(), "allissues");
        });

        test("It provides the JQL for each filter with the default order by", function() {
            Filters.initialize();

            _.each(ALL_FILTERS, function(filter) {
                var jql = Filters.getJQL(filter, "DEMO");
                equal(jql, JQLs[filter], "JQL for " + filter);
            });
        });

        test("It provides the JQL for each filter with a custom order by", function() {
            Filters.initialize();

            _.each(ALL_FILTERS, function(filter) {
                var jql = Filters.getJQL(filter, "DEMO", "key DESC");

                var customOrderBy = {
                    allissues: 'project = "DEMO" ORDER BY key DESC',
                    myopenissues: 'project = "DEMO" AND assignee = currentUser() AND resolution = Unresolved ORDER BY key DESC',
                    allopenissues: 'project = "DEMO" AND resolution = Unresolved ORDER BY key DESC',
                    reportedbyme: 'project = "DEMO" AND reporter = currentUser() ORDER BY key DESC',
                    addedrecently: 'project = "DEMO" AND created >= -1w ORDER BY key DESC',
                    resolvedrecently: 'project = "DEMO" AND resolutiondate >= -1w ORDER BY key DESC',
                    updatedrecently: 'project = "DEMO" AND updated >= -1w ORDER BY key DESC',
                    recentlyviewed: 'project = "DEMO" AND issuekey in issueHistory() ORDER BY key DESC',
                    doneissues: 'project = "DEMO" AND statusCategory = Done ORDER BY key DESC'
                };
                equal(jql, customOrderBy[filter], "JQL for " + filter);
            });
        });

        test("It escapes the project key by using quotes in the JQL", function() {
            Filters.initialize();

            var jql = Filters.getJQL('allissues', 'IN');

            equal(jql, 'project = "IN" ORDER BY priority DESC, updated DESC');
        });

        test("It checks if a filter can be refreshed when a new issue is created using Inline Issue Create", function() {
            this.withFilters(false);
            Filters.initialize();

            equal(Filters.filterSupportsInlineIssueCreate('allissues'), true);
            equal(Filters.filterSupportsInlineIssueCreate('myopenissues'), true);
            equal(Filters.filterSupportsInlineIssueCreate('allopenissues'), true);
            equal(Filters.filterSupportsInlineIssueCreate('reportedbyme'), true);
            equal(Filters.filterSupportsInlineIssueCreate('addedrecently'), true);
            equal(Filters.filterSupportsInlineIssueCreate('resolvedrecently'), false);
            equal(Filters.filterSupportsInlineIssueCreate('updatedrecently'), true);
            equal(Filters.filterSupportsInlineIssueCreate('recentlyviewed'), true);
            equal(Filters.filterSupportsInlineIssueCreate('doneissues'), false);

            equal(Filters.filterSupportsInlineIssueCreate('unknownfilter'), false);
            equal(Filters.filterSupportsInlineIssueCreate('filter2'), false); //filter without 'supportsInlineIssueCreate' flag
        });

        test("It provides a sensible defaultFilterId when no hidden field is in the query", function () {
            Filters.initialize();

            equal(Filters.getDefaultFilterId(), 'allopenissues');
        });

        test("It provides a sensible defaultFilterId when there are some filters hidden", function () {
            WrmData.claim.withArgs(WRM_GENERIC_FILTERS_KEY).returns(_.filter(GENERIC_FILTERS, function(item) { return item.id !== 'allopenissues'; }));
            Filters.initialize();

            equal(Filters.getDefaultFilterId(), 'allissues');
        });
    });
});
