define("jira/projectissuenavigator/services/dialogsoverrider", ["require"], function (require) {
    "use strict";

    var Dialog = require('jira/dialog/dialog');
    var Events = require("jira/util/events");
    var FormDialog = require('jira/dialog/form-dialog');
    var jQuery = require('jquery');
    var LabelsDialog = require('jira/dialog/labels-dialog');
    var Marionette = require("jira/projectissuenavigator/libs/marionette");

    function buildHandlerQuickEdit(eventEmitter) {
        return function () {
            var dialog = Dialog.current;
            eventEmitter.trigger("quickEdit", {
                issueId: dialog.issueId,
                issueKey: dialog.issueKey,
                message: dialog.options.issueMsg || "thanks_issue_updated"
            });
        };
    }

    function buildHandlerQuickCreateSubtasks(eventEmitter) {
        return function () {
            var dialog = Dialog.current;
            eventEmitter.trigger("quickCreateSubtask", {
                issueId: dialog.issueId,
                issueKey: dialog.issueKey,
                message: dialog.options.issueMsg || "thanks_issue_updated"
            });
        };
    }

    function isIssueDialog(instance) {
        return (typeof instance.isIssueDialog === "boolean" && instance.isIssueDialog) ||
            (jQuery.isFunction(instance.isIssueDialog) && instance.isIssueDialog());
    }

    function buildHandlerDialogShow(eventEmitter, getIssueIdStrategy, getIssueKeyStrategy) {
        return function(e, $el, instance) {
            if (instance) {
                instance.issueId = getIssueIdStrategy(instance);
                instance.issueKey = getIssueKeyStrategy(instance);

                // Clone dialog redirects to the cloned issue, we don't need to refresh current issue
                if (isIssueDialog(instance) && instance.options.id !== "clone-issue-dialog") {
                    // If the JIRA dialog contains a 'returnUrl' hidden field, the old method
                    // 'jira/dialog/form-dialog#_performRedirect' will redirect the browser to that URL. We override
                    // that method and mark the request has not redirected so the life-cycle of the request can
                    // continue and eventually call `instance.options.onDialogFinished`, which does an AJAX refresh
                    // of the current issue.
                    instance._performRedirect = function() {
                        this.redirected = false;
                    };

                    instance.options.onDialogFinished = function() {
                        instance.hide();

                        var eventName;
                        if (instance.options.id === "delete-issue-dialog") {
                            eventName = "issueDelete";
                        } else {
                            eventName = "issueUpdate";
                        }

                        eventEmitter.trigger(eventName, {
                            issueId: instance.issueId,
                            issueKey: instance.issueKey,
                            message: instance.options.issueMsg || "thanks_issue_updated"
                        });
                    };
                }

            }
        };
    }

    function buildHandlerDialogHide() {
        // This is a bug in AUI https://ecosystem.atlassian.net/browse/AUI-4087
        // AUI sets the property 'overflow' when a dialog is closed. It usually works, but fails for us
        // because we are setting the property 'overflow-y' in <body>, which is overriden by AUI's overflow.
        return function() {
            jQuery('body').css('overflow', '');
        };
    }

    return Marionette.Object.extend({
        _overrideLabelsDialogSubmitResponse: function() {
            this.oldLabelsDialogSubmitReponse = LabelsDialog.prototype._handleSubmitResponse;
            // We need the extra function wrap to make sure that each instance of DialogsOverrider
            // produces a handler with an unique memory reference, so we can detect later if the
            // current handler is the one generated by the current instance.
            LabelsDialog.prototype._handleSubmitResponse = function() {
                return FormDialog.prototype._handleSubmitResponse.apply(this, arguments);
            };
            this.currentLabelsDialogSubmitResponse = LabelsDialog.prototype._handleSubmitResponse;
        },

        _restoreLabelsDialogSubmitResponse: function() {
            // If somebody else has changed the _handleSubmitResponse, then do not restore our old
            // reference because we are going to break their assumptions.
            if (LabelsDialog.prototype._handleSubmitResponse !== this.currentLabelsDialogSubmitResponse) return;

            LabelsDialog.prototype._handleSubmitResponse = this.oldLabelsDialogSubmitReponse;
        },

        initialize: function(options) {
            this._overrideLabelsDialogSubmitResponse();

            this.handlerDialogShow = buildHandlerDialogShow(this, options.getIssueId, options.getIssueKey);
            this.handlerDialogHide = buildHandlerDialogHide(this, options.getIssueId, options.getIssueKey);
            Events.bind("Dialog.show", this.handlerDialogShow);
            Events.bind("Dialog.hide", this.handlerDialogHide);

            this.handlerQuickEditSuccess = buildHandlerQuickEdit(this);
            Events.bind("QuickEdit.sessionComplete", this.handlerQuickEditSuccess);

            this.handlerQuickCreateSubtaskSuccess = buildHandlerQuickCreateSubtasks(this);
            Events.bind("QuickCreateSubtask.sessionComplete", this.handlerQuickCreateSubtaskSuccess);
        },

        destroy: function() {
            this._restoreLabelsDialogSubmitResponse();
            Events.unbind("Dialog.show", this.handlerDialogShow);
            Events.unbind("Dialog.hide", this.handlerDialogHide);
            Events.unbind("QuickEdit.sessionComplete", this.handlerQuickEditSuccess);
            Events.unbind("QuickCreateSubtask.sessionComplete", this.handlerQuickCreateSubtaskSuccess);
        }
    });
});
