define("jira/projectissuenavigator/pages/issueview", ["require"], function(require) {
    "use strict";

    var _ = require("jira/projectissuenavigator/libs/underscore");
    var DarkFeatures = require("jira/components/issueviewer/services/darkfeatures");
    var DialogsOverrider = require("jira/projectissuenavigator/services/dialogsoverrider");
    var FocusShifter = require("jira/projectissuenavigator/services/focusshifter");
    var IssueComponentResolver = require("jira/components/issuecomponentresolver");
    var IssueView = require("jira/projectissuenavigator/pages/issueview/views/issue");
    var IssueViewTemplates = require("jira/components/issueviewer/templates");
    var JIRAFlag = require("jira/flag");
    var jQuery = require("jquery");
    var KeyboardShortcut = require("jira/ajs/keyboardshortcut/keyboard-shortcut");
    var Marionette = require("jira/projectissuenavigator/libs/marionette");
    var Metrics = require("jira/projectissuenavigator/services/metrics");
    var Notifications = require("jira/projectissuenavigator/services/notifications");
    var Pager = require("jira/components/pager");
    var Traces = require("jira/projectissuenavigator/services/traces");
    var WRMData = require("wrm/data");

    return Marionette.Object.extend({
        /**
         * @constructor
         * @param {Object} options
         * @param {module:jira/projectissuenavigator/entities/navigatorstate} options.state Shared state
         * @param {boolean} [options.showCollapse-false] Whether this page should have the collapse button
         */
        initialize: function(options) {
            this.showCollapse = options.showCollapse || false;
            this.state = options.state;
            this._buildIssueView();
            this._buildIssueComponent();
            this._buildDialogsOverrider();
            this._buildFocusShifter();
            this._buildPager();
        },

        /**
         * Clean up internal components when this instance is destroyed
         */
        onDestroy: function() {
            this.issueComponent.close();
            this.issueView.destroy();
            this.dialogsOverrider.destroy();
            this.focusShifter.destroy();
            this._destroyPager();
            if (this.el) {
                this.el.remove();
            }
        },

        /**
         * Clean up the pager
         */
        _destroyPager: function () {
            if (this.pager) {
                this.pager.destroy();
                this.stopListening(this.pager);
                delete this.pager;
            }
        },

        /**
         * Render the page's root element
         */
        render: function() {
            this.issueView.render();
            this.el = this.issueView.$el;
        },

        /**
         * Load the existing markup as the page's root element
         */
        attach: function(el) {
            this.issueView.attach(el);
            this.el = this.issueView.$el;
        },

        /**
         * Load an issue that has been previously rendered in the page
         *
         * @param {string} issueKey Issue key to load
         * @returns {jQuery.Promise} Promise resolved when the page components are ready to be used
         */
        loadFromPage: function(issueKey) {
            this._updateState(issueKey);
            this._updateURL(issueKey);
            this._updatePager();
            return this.issueComponent.loadIssueRenderedByTheServer({
                key: issueKey,
                project: {
                    id: WRMData.claim('projectId'),
                    key: WRMData.claim('projectKey'),
                    projectType: WRMData.claim('projectType')
                },
                detailView: false
            });
        },

        /**
         * Show an issue in the page
         *
         * @param {string} issueKey Issue to load
         * @returns {jQuery.Promise} Promise resolved when the issue is rendered on the screen and the page components
         *                           are ready to be used
         */
        load: function (issueKey) {
            this._updateState(issueKey);
            this._updateURL(issueKey);
            this._updatePager();
            return this.issueComponent.loadIssue({
                key: issueKey,
                detailView: false
            });
        },

        /**
         * Updates the URL to match the issue we are about to load.
         *
         * This needs to happen before the actual load:
         *   When we update the URL, the current URL will be part of the History using the current window title. Also,
         *   loading an issue will change the window title. So if we combine those two facts, the only way to send the
         *   current URL to the History using the current window title is to update the URL before issue loading.
         *
         * Also, this needs to be event-driven (i.e. throw an event) instead of method-driven (call a method in the
         * router) because our parent, ProjectIssueNavigator, is the one who knows if this was the initial load of an
         * issue. That bit will affect how the URL is updated.
         *
         * @param {string} issueKey IssueKey to be represented in the new URL
         */
        _updateURL: function(issueKey) {
            if(DarkFeatures.REDIRECT_FROM_GLOBAL_TO_PROJECT.enabled()) {
                this.trigger("updateUrl", {
                    issueKey: issueKey
                });
            }
        },

        /**
         * Update the pager to display the current searchResults and selected issue
         *
         * If there are no search results, or the current issue is not contained in those search results,
         * the pager is removed from the page
         */
        _updatePager: function() {
            var searchResults = this.state.get('searchResults');
            var issueKey = this.state.get('issue');

            if (searchResults && searchResults.hasIssueInSearch(issueKey)) {
                if (!searchResults.selected || searchResults.selected.get('key') !== issueKey) {
                    searchResults.select(issueKey);
                }
                this.pager.update(searchResults);
            } else {
                this._destroyPager();
            }
        },

        /**
         * Updates the state to reflect the issue displayed by this page
         *
         * @param {string} issueKey IssueKey to save in the state
         */
        _updateState: function (issueKey) {
            this.state.set('issue', issueKey);
        },

        /**
         * Adjust the size of the internal components, making the layout expand to fill the entire screen
         * (vertically), and applying the responsive rules to the IssueEditor
         */
        adjustSize: function () {
            this.issueView.maximizeHeight();
            this.issueComponent.applyResponsiveDesign();
        },

        /**
         * Constructs the Issue view, used as a placeholder for the IssueEditor
         */
        _buildIssueView: function() {
            this.issueView = new IssueView({
                showCollapse: this.showCollapse,
                collapseShortcutKey: KeyboardShortcut.getKeyboardShortcutKeys('fullscreen.issue')
            });
            this.listenTo(this.issueView, {
                "render": function() {
                    // When the issueView is rendered, inject the issueEditor inside
                    this.issueView.issueContainer._ensureElement();
                    this.issueComponent.setContainer(this.issueView.issueContainer.$el);

                    // If we have a pager, inject it into the view
                    if (this.pager) {
                        this.issueView.pager._ensureElement();
                        this.pager.show(this.issueView.pager.$el);
                    }
                },
                "collapse": function() {
                    this.trigger("collapse", "ui-button");
                }
            });
        },

        /**
         * Constructs the IssueEditor component (or a regular IssueViewer is Inline Editing is disabled).
         */
        _buildIssueComponent: function() {
            var IssueComponent = IssueComponentResolver.resolve();
            this.issueComponent = new IssueComponent({
                useJIRAEvents: false
            });

            this.listenTo(this.issueComponent, {
                "fieldsLoaded": function() {
                    // inline edit is ready. This can be disabled by admin or if you don't have edit permissions and there is no labels field on the issue
                    this.trigger("editorFieldsLoaded");
                },
                "loadError": function(properties) {
                    this.trigger("loadError", properties);
                },
                "loadComplete": function (model, data) {
                    this.trigger("editorLoaded", data);
                },
                "linkToIssue": function(options) {
                    this.trigger("linkToIssue", options);

                    // When we link to *any* issue (that includes issues to the same project), we don't want to show
                    // the pager. We could show the pager for issues in the same project & filter, but that will be
                    // confusing for the user. The presence of the pager is determined by the presence of searchResults,
                    // so we have to unset it here.

                    this.state.unset('searchResults');
                    this.state.unset('filter');
                    this.load(options.issueKey);
                },
                "individualPanelRendered": function(renderedPanel) {
                    this.trigger("individualPanelRendered", renderedPanel);
                },
                "renderMainView": function($el, issueId) {
                    this.trigger("viewRendered", $el, issueId);
                },
                "panelRendered": function(panelId, $ctx, $existing) {
                    this.trigger("panelRendered", panelId, $ctx, $existing);
                },
                "refineViewer": function (event) {
                    event.preventDefault();
                    this.issueComponent.updateIssueWithQuery(event.query);
                }
            });
        },

        /**
         * Build the DialogsOverrider component.
         *
         * This is used to ensure that all JIRA Dialogs (edit issue, labels, etc.) operate on the correct issue key
         */
        _buildDialogsOverrider: function() {
            function dataExtractor(dataKey, orElse, dialog) {
                return dialog && dialog.$activeTrigger && dialog.$activeTrigger.data(dataKey) || (typeof orElse === 'function') && orElse();
            }
            this.dialogsOverrider = new DialogsOverrider({
                getIssueId: _.partial(dataExtractor, "issueid", this.issueComponent.getIssueId.bind(this.issueComponent)),
                getIssueKey: _.partial(dataExtractor, "issuekey", this.issueComponent.getIssueKey.bind(this.issueComponent))
            });
            this.listenTo(this.dialogsOverrider, {
                "quickEdit quickCreateSubtask issueUpdate": function(data) {
                    this.issueComponent.refreshIssue().done(function () {
                        Notifications.show(data.message, data.issueKey);
                        Traces.traceIssueUpdated();
                    });
                },
                "issueDelete": function(data) {
                    Notifications.show(data.message, data.issueKey);

                    //Check if it is the same issue or a sub task what was deleted
                    if (data.issueKey === this.state.get('issue')) {
                        this.trigger("issueDelete", data);
                    } else {
                        this.issueComponent.refreshIssue();
                    }
                }
            });
        },

        /**
         * Build the Focus Shifter component, and link it to the IssueEditor
         */
        _buildFocusShifter: function() {
            this.focusShifter = new FocusShifter(this.issueComponent);
        },

        /**
         * Returns the issueId of the issue being displayed
         *
         * @returns {number}
         */
        getIssueId: function() {
            return this.issueComponent.getIssueId();
        },

        /**
         * Returns the issueKey of the issue being displayed
         *
         * @returns {string}
         */
        getIssueKey: function() {
            return this.issueComponent.getIssueKey();
        },

        /**
         * Returns the id of the project being displayed
         *
         * @returns {number|null}
         */
        getProjectId: function() {
            return this.issueComponent.getProjectId();
        },

        /**
         * Returns the key of the project being displayed
         *
         * @returns {string|null}
         */
        getProjectKey: function() {
            return this.issueComponent.getProjectKey();
        },

        /**
         * Returns the type of the project being displayed
         *
         * @returns {string|null}
         */
        getProjectType: function() {
            return this.issueComponent.getProjectType();
        },

        _buildPager: function() {
            this.pager = new Pager();
            this.listenTo(this.pager, {
                "next": function() {
                    this._loadNextIssue();
                },
                "previous": function() {
                    this._loadPreviousIssue();
                }
            });
        },

        /**
         * Load the next issue in the search results
         *
         * If there are no search results (or there is no next issue), this method is a noop
         */
        _loadNextIssue: function() {
            var searchResults = this.state.get('searchResults');
            if (!searchResults) return;

            searchResults.getNextIssue().done(function (issue) {
                this._loadIssueFromPager(issue, "pager:next");
            }.bind(this));
        },

        /**
         * Load the previous issue in the search results
         *
         * If there are no search results (or there is no previous issue), this method is a noop
         */
        _loadPreviousIssue: function() {
            var searchResults = this.state.get('searchResults');
            if (!searchResults) return;

            searchResults.getPreviousIssue().done(function (issue) {
                this._loadIssueFromPager(issue, "pager:previous");
            }.bind(this));
        },

        /**
         * Loads an issue as a result of a pager operation.
         */
        _loadIssueFromPager: function(issue, eventName) {
            var searchResults = this.state.get('searchResults');

            this.trigger(eventName, {
                current: searchResults.getPositionOfIssueInSearchResults(issue.id),
                total: searchResults.length
            });
            this.load(issue.get('key'));
        },

        /**
         * Returns the API implementation used by this page
         * @returns {Object}
         */
        getAPIImplementation: function() {
            // We use closures here to avoid binding every single method in this object.
            var page = this;
            return {
                "nextIssue": function() {
                    var searchResults = page.state.get('searchResults');
                    if (!searchResults) return;

                    Metrics.startIssueSearch();
                    return page._loadNextIssue();
                },
                "previousIssue": function() {
                    var searchResults = page.state.get('searchResults');
                    if (!searchResults) return;

                    Metrics.startIssueSearch();
                    return page._loadPreviousIssue();
                },
                "getActiveIssueId": function() {
                    return page.issueComponent.getIssueId();
                },
                "getActiveIssueKey": function() {
                    return page.issueComponent.getIssueKey();
                },
                "refreshActiveIssue": function() {
                    Metrics.startIssueSearch();
                    return page.issueComponent.refreshIssue();
                },
                "refreshIssue": function() {
                    Metrics.startIssueSearch();
                    return page.issueComponent.refreshIssue();
                },
                "editField": function(fieldId) {
                    if (!("getFields" in page.issueComponent)) {
                        return false;
                    }
                    var fields = page.issueComponent.getFields();
                    var field = fields && fields.get(fieldId);
                    var permitted = field && field.isEditable();

                    if (permitted) {
                        page.issueComponent.editField(field);
                    }

                    return permitted;
                },
                "getFields": function() {
                    if (!("getFields" in page.issueComponent)) {
                        return undefined;
                    }

                    var fields = page.issueComponent.getFields();
                    return fields.length ? fields : undefined;
                },
                "isSaving": function() {
                    if (!("hasSavesInProgress" in page.issueComponent)) {
                        return false;
                    }

                    return page.issueComponent.hasSavesInProgress();
                },
                "isLoading": function() {
                    return page.issueComponent.isCurrentlyLoading();
                },
                "hasAccessibleIssue": function() {
                    return !page.issueComponent.isShowingError();
                },
                "showLoadError": function() {
                    return JIRAFlag.showErrorMsg('', IssueViewTemplates.Body.errorsLoading(), {
                        persistent: false
                    });
                },
                "waitForSavesToComplete": function() {
                    if (!("hasSavesInProgress" in page.issueComponent)) {
                        return new jQuery.Deferred().resolve().promise();
                    }

                    var d = new jQuery.Deferred();
                    if (!page.issueComponent.hasSavesInProgress()) {
                        d.resolve();
                    } else {
                        var success = function() {
                            _.defer(function() {
                                if (!page.issueComponent.hasSavesInProgress()) {
                                    d.resolve();
                                }
                            });
                        };
                        page.issueComponent.once("saveSuccess", success);
                    }
                    return d.promise();
                },
                "openFocusShifter": function() {
                    page.focusShifter.show();
                },
                "toggleFullscreenIssue": function() {
                    if(page.showCollapse) {
                        page.trigger("collapse", "kb-shortcut");
                    }
                }
            };
        },

        focusIssueEditor: function() {
            this.issueComponent.focus();
        }
    });
});
