AJS.test.require("com.atlassian.jira.jira-projects-issue-navigator:application-test", function () {
    "use strict";

    require([
        "jira/components/search",
        "jira/projectissuenavigator/libs/marionette",
        "jira/projectissuenavigator/services/filters",
        "jira/projectissuenavigator/services/urlhelper",
        "jira/projectissuenavigator/services/linkscapturer",
        "jira/projectissuenavigator/libs/uri",
        "jira/projectissuenavigator/services/browser",
        "jira/projectissuenavigator/services/dialogsoverrider",
        "jira/projectissuenavigator/services/metrics",
        "jira/projectissuenavigator/pages/issuesearch/views/empty",
        "jira/components/detailslayout",
        "jira/components/search/results",
        "jira/projectissuenavigator/services/inline-issue-create",
        "jira/util/events",
        "jira/projects/test-utils/marionettemocker",
        "jira/projects/test-utils/mockutils",
        "jira/flag",
        "jira/message",
        "jira/projectissuenavigator/libs/backbone",
        "jquery",
        "jira/projectissuenavigator/libs/underscore",
        "jira/ajs/ajax/smart-ajax",
        "jira/projectissuenavigator/entities/navigatorstate",
        'jira/components/util/events'
    ], function (
        Search,
        Marionette,
        Filters,
        URLHelper,
        LinksCapturer,
        URI,
        Browser,
        DialogsOverrider,
        Metrics,
        EmptyView,
        DetailsLayout,
        Results,
        InlineIssueCreate,
        JIRAEvents,
        MarionetteMocker,
        MockUtils,
        JIRAFlag,
        Message,
        Backbone,
        jQuery,
        _,
        SmartAjax,
        State,
        Types
    ) {
        var Subnavigator = JIRA.Projects.Subnavigator;

        module("jira/projectissuenavigator/pages/issuesearch", {
            setup: function () {
                this.sandbox = sinon.sandbox.create();

                // Don't waste time doing network requests
                this.sandbox.useFakeServer();

                // Don't start the history management, it will mess up with our QUnit page
                this.sandbox.stub(Backbone.history, "start");

                this.sandbox.stub(Browser, "locationAssign");

                this.sandbox.stub(URLHelper, "extractProjectKeyFromCurrentUrl").returns("DEMO");

                this.detailsLayout = MarionetteMocker.createEventedMock(this.sandbox, DetailsLayout);
                this.subnavigator = MarionetteMocker.createEventedMock(this.sandbox, Subnavigator);
                this.linksCapturer = MarionetteMocker.createEventedMock(this.sandbox, LinksCapturer);
                this.inlineIssueCreate = MarionetteMocker.createEventedMock(this.sandbox, InlineIssueCreate);
                this.emptyView = MarionetteMocker.createEventedMock(this.sandbox, EmptyView);
                this.dialogsOverrider = MarionetteMocker.createEventedMock(this.sandbox, DialogsOverrider);

                this.sandbox.stub(Filters, "getJQL");

                JIRA.Projects.Subnavigator = this.subnavigator.constructor;
                this.IssueSearchConstructor = MockUtils.spyAll(this.sandbox, MockUtils.requireWithMocks("jira/projectissuenavigator/pages/issuesearch", {
                    "jira/components/detailslayout": this.detailsLayout.constructor,
                    "jira/projectissuenavigator/services/urlhelper": URLHelper,
                    "jira/projectissuenavigator/services/linkscapturer": this.linksCapturer.constructor,
                    "jira/projectissuenavigator/services/browser": Browser,
                    "jira/projectissuenavigator/pages/issuesearch/views/empty": this.emptyView.constructor,
                    "jira/projectissuenavigator/services/dialogsoverrider": this.dialogsOverrider.constructor,
                    "jira/util/events": JIRAEvents,
                    "jira/projectissuenavigator/services/metrics": Metrics,
                    "jira/ajs/ajax/smart-ajax": SmartAjax,
                    "jira/projectissuenavigator/services/inline-issue-create": this.inlineIssueCreate.constructor,
                    "jira/projectissuenavigator/services/filters": Filters,
                    "jira/flag": JIRAFlag,
                    "jira/message": Message
                }));
                this.search = this.sandbox.stub(new Search());
                this.search.search.returns(new jQuery.Deferred().promise());

                this.state = new State();

                this.issueSearch = new this.IssueSearchConstructor({
                    state: this.state,
                    search: this.search
                });
                this.issueSearch.show();
                this.el = this.issueSearch.el;

                // Stub the error flag to prevent flags from appearing in our tests
                this.stubErrorFlag();
            },

            stubErrorFlag: function() {
                this.sandbox.stub(SmartAjax, "buildSimpleErrorContent");

                if (this.issueSearch.showErrorFlag.restore) {
                    this.issueSearch.showErrorFlag.restore();
                }
                return this.sandbox.stub(this.issueSearch, "showErrorFlag");
            },

            teardown: function () {
                JIRA.Projects.Subnavigator = Subnavigator;
                this.sandbox.restore();
                jQuery(document.body).removeClass("page-type-split page-issue-navigator");
            },

            assertState: function(state) {
                _.each(state, function(value, key) {
                    if (key === null) {
                        ok(!this.state.has(key));
                    } else {
                        equal(this.state.get(key), value, "Internal state for " + key);
                    }
                }.bind(this));
            },

            assertEventTriggered: function(event, state) {
                var fn = this.issueSearch.trigger;
                sinon.assert.calledOnce(fn);
                var lastCall = fn.lastCall;
                equal(lastCall.args[0], event, "The correct event was dispatched");
                deepEqual(lastCall.args[1], state, "the correct payload was sent");
            },

            assertSearch: function(params, sidebarIssueSearch) {
                sidebarIssueSearch = sidebarIssueSearch || this.issueSearch;
                sinon.assert.calledOnce(sidebarIssueSearch.search);
                sinon.assert.calledWith(sidebarIssueSearch.search, params);
            },

            withSidebarIssueSearchThatCanCreateIssues: function() {
                var sidebarIssueSearch = new this.IssueSearchConstructor({
                    canCreateIssues: true,
                    router: this.router,
                    state: this.state,
                    search: this.search
                });
                sidebarIssueSearch.show();
                return sidebarIssueSearch;
            }
        });

        test("When searching data, it changes the state to represent the load params", function() {
            this.issueSearch.search({
                filter: 'updatedrecently',
                issue: 'TEMP-123',
                orderby: 'key DESC'
            });

            this.assertState({
                filter: 'updatedrecently',
                issue: 'TEMP-123',
                orderby: 'key DESC'
            });
        });

        test("When searching data for the second time, it uses the previous state if none is provided", function() {
            this.issueSearch.search({
                filter: 'updatedrecently',
                issue: 'TEMP-123',
                orderby: 'key DESC'
            });

            this.issueSearch.search({
                issue: 'TEMP-124'
            });

            this.assertState({
                filter: 'updatedrecently',
                issue: 'TEMP-124', //new state
                orderby: 'key DESC'
            });
        });

        test("When searching data for the second time, it can erase data from the state if null is provided", function() {
            this.issueSearch.search({
                filter: 'updatedrecently',
                issue: 'TEMP-123',
                orderby: 'key DESC'
            });

            this.issueSearch.search({
                orderby: null
            });

            this.assertState({
                filter: 'updatedrecently',
                issue: 'TEMP-123',
                orderby: null
            });
        });

        test("When searching data, it does a search using a JQL based on the filter, project and orderby states", function() {
            this.search.search.returns(new jQuery.Deferred().promise());
            this.state.set('project', 'DEMO');
            Filters.getJQL.withArgs('updatedrecently', 'DEMO', 'key DESC').returns('project = "DEMO" AND updated >= -1w ORDER BY key DESC');

            this.issueSearch.search({
                filter: 'updatedrecently',
                orderby: 'key DESC'
            });

            sinon.assert.calledOnce(this.search.search);
            sinon.assert.calledWith(this.search.search, 'project = "DEMO" AND updated >= -1w ORDER BY key DESC');
        });

        test("When searching data, it can jump to a particular startIndex", function() {
            var searchResults = new Results([], {
                issues: [
                    {id: 1, key: "DEMO-1"},
                    {id: 2, key: "DEMO-2"},
                    {id: 3, key: "DEMO-3"},
                    {id: 4, key: "DEMO-4"}
                ]
            });
            this.search.search.returns(new jQuery.Deferred().resolve(searchResults).promise());

            this.issueSearch.search({
                startIndex: "2"
            });

            sinon.assert.calledOnce(this.detailsLayout.load);
            sinon.assert.calledWith(this.detailsLayout.load, searchResults, "DEMO-3");
        });

        test("When searching data, it ignores the startIndex if there is a selected issue", function() {
            var searchResults = new Results([], {
                issues: [
                    {id: 1, key: "DEMO-1"},
                    {id: 2, key: "DEMO-2"},
                    {id: 3, key: "DEMO-3"},
                    {id: 4, key: "DEMO-4"}
                ]
            });
            this.search.search.returns(new jQuery.Deferred().resolve(searchResults).promise());

            this.issueSearch.search({
                startIndex: "1",
                issue: "DEMO-4"
            });

            sinon.assert.calledOnce(this.detailsLayout.load);
            sinon.assert.calledWith(this.detailsLayout.load, searchResults, "DEMO-4");
        });

        test("When searching data, if the load fails then an error flag is shown and the loading view is hidden", function() {
            this.search.search.returns(new jQuery.Deferred().reject());

            this.issueSearch.search();

            sinon.assert.calledOnce(this.issueSearch.showErrorFlag);
            sinon.assert.calledOnce(this.detailsLayout.hideLoading);
        });

        test("When searching data, if the load fails due to a 400. It should display the first error message in the response", function() {
            var errorMessage = "jira.jql.validation.no.such.field";
            var searchResults = {
                responseText: '{"errorMessages":["' + errorMessage + '"]}',
                status: 400
            };

            this.search.search.returns(new jQuery.Deferred().reject(searchResults));
            this.issueSearch.search();

            sinon.assert.calledOnce(this.issueSearch.showErrorFlag);
            sinon.assert.calledWith(this.issueSearch.showErrorFlag, errorMessage);
        });

        // JRASERVER-65528 - The JQL error messages are not HTML-safe
        test("The error flag should not allow XSS", function() {
            var errorMessage = "Error in the JQL Query: The quoted string '><b class='the-problem'>whoops, xss hole</b>' has not been completed. (line 1, character 54)";
            var escapedErrorMessage = AJS.escapeHtml(errorMessage);
            var searchResults = {
                responseText: '{"errorMessages":["' + errorMessage + '"]}',
                status: 400
            };

            this.stub(JIRAFlag, "showErrorMsg");
            this.issueSearch.showErrorFlag.restore();

            // Yes, this is odd syntax. You can thank YUICompressor for that.
            /* eslint-disable dot-notation */
            JIRAFlag.showErrorMsg.withArgs(errorMessage)['throws']("whoops, an XSS happened");
            JIRAFlag.showErrorMsg.withArgs(sinon.match.string, errorMessage)['throws']("whoops, an XSS happened");
            /* eslint-enable dot-notation */

            this.search.search.returns(new jQuery.Deferred().reject(searchResults));
            this.issueSearch.search();

            sinon.assert.calledOnce(JIRAFlag.showErrorMsg);
            sinon.assert.calledWith(JIRAFlag.showErrorMsg, sinon.match.string, escapedErrorMessage);
        });

        test("When searching data, if the load fails due to a timeout. It should display a timeout message.", function() {
            var searchResults = {
                statusText: SmartAjax.SmartAjaxResult.TIMEOUT,
                status: 0
            };

            this.search.search.returns(new jQuery.Deferred().reject(searchResults));
            this.issueSearch.search();

            sinon.assert.calledOnce(this.issueSearch.showErrorFlag);
            sinon.assert.calledWith(SmartAjax.buildSimpleErrorContent, searchResults);
        });

        test("When the DetailsLayout triggers a 'select' event, the event is re-triggered as 'issue:select'", function() {
            this.detailsLayout.trigger("select", {key: 'TEMP-456'});

            this.assertEventTriggered("issue:select", {
                issue: "TEMP-456"
            });
        });

        test("When the DetailsLayout triggers a 'select' event, the issue is updated in the state", function() {
            this.detailsLayout.trigger("select", {key: 'TEMP-456'});

            this.assertState({
                issue: 'TEMP-456'
            });
        });

        test("When the DetailsLayout triggers a 'empty' event, the event is re-triggered as 'issue:empty'", function() {
            this.detailsLayout.trigger("empty");

            this.assertEventTriggered("issue:empty");
        });

        test("When the DetailsLayout triggers an 'empty' event, the issue is removed from the state", function() {
            this.state.set('issue', 'TEMP-456');
            this.detailsLayout.trigger("empty");

            this.assertState({
                issue: null
            });
        });

        test("When the DetailsLayout triggers a 'list:sort' event, it loads new issues with the new orderby", function() {
            this.detailsLayout.trigger("list:sort", 'project = DEMO AND updated >= -1w ORDER BY updated DESC');

            this.assertSearch({
                orderby: "updated DESC"
            });
        });

        test("When the DetailsLayout triggers an 'error:loadpage' event, the sidebar issue nav shows an error flag", function() {
            this.detailsLayout.trigger("error:loadpage");

            sinon.assert.calledOnce(this.issueSearch.showErrorFlag);
        });

        test("When the Subnavigator triggers a 'itemSelected' event, it loads new issues with the new filter and the orderby is cleared", function() {
            this.subnavigator.trigger("itemSelected", {item: {id: "addedrecently", jql: "reporter = currentUser()"}, preventDefault: this.spy()});

            this.assertSearch({
                filter: "addedrecently",
                issue: null,
                orderby: null
            });
        });

        test("When the detailslayout triggers a 'render' event, an equivalent JIRA event is triggered", function() {
            var jiraEventSpy = this.spy(JIRAEvents, "trigger");

            this.detailsLayout.trigger("render");

            sinon.assert.calledOnce(jiraEventSpy);
            sinon.assert.calledWith(jiraEventSpy, Types.LAYOUT_RENDERED);
        });

        test("When a filter is selected in the Subnavigator, it starts recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "startIssueSearch");

            var fakeEvent = {item: {id: "addedrecently", jql: "reporter = currentUser()"}, preventDefault: this.spy()};
            this.subnavigator.trigger("itemSelected", fakeEvent);

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When a filter is selected in the Subnavigator, it triggers a 'filterSelected' event with the information of the filter", function() {
            var onFilterSelected = this.spy();
            this.issueSearch.on("filterSelected", onFilterSelected);
            var fakeEvent = {item: {id: "addedrecently", jql: "reporter = currentUser()"}, preventDefault: this.spy()};

            this.subnavigator.trigger("itemSelected", fakeEvent);

            sinon.assert.calledOnce(onFilterSelected);
            sinon.assert.calledWith(onFilterSelected, {filterId: 'addedrecently', source: 'subnavigator'});
        });

        test("When the list is sorted in the DetailsLayout, it starts recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "startIssueSearch");

            var fakeJQL = "";
            this.detailsLayout.trigger("list:sort", fakeJQL);

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When the list is refreshed in the DetailsLayout, it starts recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "startIssueSearch");

            this.detailsLayout.trigger("list:refresh");

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When an item is selected in the DetailsLayout, it starts recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "startIssueSearch");

            var fakeIssueData = {};
            this.detailsLayout.trigger("list:select", fakeIssueData);

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When the list of issue is paginated in the DetailsLayout, it starts recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "startIssueSearch");

            this.detailsLayout.trigger("list:pagination");

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When an issue is loaded from the cache in the DetailsLayout, it stops recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "endIssueSearch");

            var fakeEvent = {};
            this.detailsLayout.trigger("editorLoadedFromCache", fakeEvent);

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When an issue is loaded in the DetailsLayout, it stops recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "endIssueSearch");

            var fakeEvent = {};
            this.detailsLayout.trigger("editorLoaded", fakeEvent);

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When an issue is loaded in the DetailsLayout it should pass event.issueEditorOptions.issueRefreshedEvent===true as a issueRefreshedEvent parameter set to true", function() {
            var fakeEvent = {issueEditorOptions: {issueRefreshedEvent: true}};
            this.detailsLayout.trigger("editorLoaded", fakeEvent);

            sinon.assert.calledOnce(this.issueSearch.trigger);
            sinon.assert.calledWith(this.issueSearch.trigger, 'editorLoaded', {issueRefreshedEvent: true});
        });

        test("When an issue is loaded in the DetailsLayout it should pass event.issueEditorOptions.issueRefreshedEvent===false as a issueRefreshedEvent parameter set to false", function() {
            var fakeEvent = {issueEditorOptions: {issueRefreshedEvent: false}};
            this.detailsLayout.trigger("editorLoaded", fakeEvent);

            sinon.assert.calledOnce(this.issueSearch.trigger);
            sinon.assert.calledWith(this.issueSearch.trigger, 'editorLoaded', {issueRefreshedEvent: false});
        });

        test("When an issue is loaded in the DetailsLayout it should pass empty event.issueEditorOptions as a issueRefreshedEvent parameter set to false", function() {
            var fakeEvent = {};
            this.detailsLayout.trigger("editorLoaded", fakeEvent);

            sinon.assert.calledOnce(this.issueSearch.trigger);
            sinon.assert.calledWith(this.issueSearch.trigger, 'editorLoaded', {issueRefreshedEvent: false});
        });

        test("When the next issue is selected, it start recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "startIssueSearch");

            this.detailsLayout.trigger("pager:next", {current: 1, total: 2});

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When the previous issue is selected, it start recording browser metrics", function() {
            var metricsSpy = this.spy(Metrics, "startIssueSearch");

            this.detailsLayout.trigger("pager:previous", {current: 1, total: 2});

            sinon.assert.calledOnce(metricsSpy);
        });

        test("When the next issue is selected, it should trigger an event", function() {
            var onNextSpy = this.spy();

            this.issueSearch.on("pager:next", onNextSpy);

            this.detailsLayout.trigger("pager:next", {current: 1, total: 2});

            sinon.assert.calledOnce(onNextSpy);
        });

        test("When the previous issue is selected, it should trigger an event", function() {
            var onPreviousSpy = this.spy();

            this.issueSearch.on("pager:previous", onPreviousSpy);

            this.detailsLayout.trigger("pager:previous", {current: 1, total: 2});

            sinon.assert.calledOnce(onPreviousSpy);
        });

        test("When constructed with an user that can create issues, it ask the DetailsLayout to display the IIC", function() {
            this.withSidebarIssueSearchThatCanCreateIssues();

            equal(this.detailsLayout.constructor.lastCall.args[0].displayInlineIssueCreate, true);
        });

        test("When constructed with an user that can create issues, it ask the DetailsLayout to display the IIC", function() {
            this.withSidebarIssueSearchThatCanCreateIssues();

            equal(this.detailsLayout.constructor.lastCall.args[0].displayInlineIssueCreate, true);
        });

        test("When an issue is created in the DetailsLayout and the current filter supports inline issue creation, we reload the current filter and select the created issue", function() {
            this.stub(Filters, "filterSupportsInlineIssueCreate").withArgs('aFilterId').returns(true);
            var issueSearch = this.withSidebarIssueSearchThatCanCreateIssues();
            this.state.set('project', 'DEMO');
            issueSearch.search({
                filter: 'aFilterId',
                issue: 'DEMO-1'
            });
            issueSearch.search.reset();

            this.detailsLayout.trigger("issueCreated", {
                issue: {
                    issueKey: 'DEMO-2'
                },
                fields: {
                    summary: 'This is a test'
                }
            });

            this.assertSearch({
                issue: 'DEMO-2'
            }, issueSearch);

            this.assertState({
                filter: 'aFilterId',
                issue: 'DEMO-2',
                project: 'DEMO'
            }, issueSearch);
        });

        test("When an issue is created in the DetailsLayout and the current filter does not support inline issue creation, we show a flag", function() {
            this.stub(Filters, "filterSupportsInlineIssueCreate").withArgs('aFilterId').returns(false);
            this.stub(Message, "showSuccessMsg");
            this.state.set('project', 'DEMO');
            var issueSearch = this.withSidebarIssueSearchThatCanCreateIssues();
            issueSearch.search({
                filter: 'aFilterId',
                issue: 'DEMO-1'
            });
            issueSearch.search.reset();

            this.detailsLayout.trigger("issueCreated", {
                issue: {
                    issueKey: 'DEMO-2'
                },
                fields: {
                    summary: 'This is a test'
                }
            });

            sinon.assert.notCalled(issueSearch.search);
            this.assertState({
                filter: 'aFilterId',
                issue: 'DEMO-1',
                project: 'DEMO'
            }, issueSearch);
            sinon.assert.calledOnce(Message.showSuccessMsg);
        });

        test("When the DetailsLayout triggers a 'empty' event, if the IIC mode is enabled whe display the IIC widget", function() {
            var el = jQuery("<div></div>");
            var jql = "project=DEMO";
            var issueSearch = this.withSidebarIssueSearchThatCanCreateIssues();
            this.emptyView.inlineIssueCreateContainer = new Marionette.Region({el: el});
            issueSearch.emptyView = this.emptyView;
            this.state.set('searchResults', new Results([], {jql: jql}));

            this.detailsLayout.trigger("empty");

            sinon.assert.calledOnce(this.inlineIssueCreate.show);
            sinon.assert.calledWith(this.inlineIssueCreate.show, this.emptyView.inlineIssueCreateContainer.$el);
        });

        test("When the DetailsLayout triggers a 'empty' event, if the IIC is activated with the results JQL", function() {
            var el = jQuery("<div></div>");
            var jql = "project=DEMO";
            var issueSearch = this.withSidebarIssueSearchThatCanCreateIssues();
            this.emptyView.inlineIssueCreateContainer = new Marionette.Region({el: el});
            issueSearch.emptyView = this.emptyView;
            this.state.set('searchResults', new Results([], {jql: jql}));

            this.detailsLayout.trigger("empty");

            sinon.assert.calledOnce(this.inlineIssueCreate.activateWithJQL);
            sinon.assert.calledWith(this.inlineIssueCreate.activateWithJQL, jql);
        });

        test("When searching data and we correct the filter, it triggers a 'filterSelected' event with the information of the new filter", function() {
            var onFilterSelected = this.spy();
            this.issueSearch.on("filterSelected", onFilterSelected);
            var searchResults = new Results([], {
                issues: [
                    {id: 1, key: "DEMO-1"},
                    {id: 2, key: "DEMO-2"}
                ],
                jql: 'project = "DEMO" AND updated >= -1w ORDER BY key DESC'
            });
            this.search.search.returns(new jQuery.Deferred().resolve(searchResults).promise());

            this.issueSearch.search({
                filter: 'updatedrecently',
                issue: 'DEMO-3'  //Issue not in the results
            });

            sinon.assert.calledOnce(onFilterSelected);
            sinon.assert.calledWith(onFilterSelected, {filterId: 'allissues', source: 'issueFilterMismatch'});
        });

        test("When the user select the Manage Filters in the subnavigator, it doesn't change the title'", function() {
            var title = this.el.find("#issues-subnavigation-title");
            title.text("Open issues");

            this.subnavigator.trigger('itemSelected', {
                item: {
                    id: 'manage',
                    jql: true
                },
                preventDefault: function () {}
            });

            equal(title.text(), "Open issues");
        });

        test("When the user selects any other filter in the subnavigator, it changes the title'", function() {
            var title = this.el.find("#issues-subnavigation-title");
            title.text("Open issues");

            this.issueSearch.subnavigation.trigger('itemSelected', {
                item: {
                    id: 'allopenissues',
                    jql: true
                },
                preventDefault: function () {}
            });

            // We're expecting the title to be an empty string because there's
            // no web resource data injected in the test page, but it doesn't
            // matter because we just want to test that the title is actually
            // changed
            equal(title.text(), "");
        });

        test("When the user select the Manage Filters in the subnavigator, it triggers the event 'manageFilters'", function() {
            var onManageFilters = this.spy();
            this.issueSearch.on("manageFilters", onManageFilters);

            this.subnavigator.trigger('itemSelected', {
                item: {
                    id: 'manage'
                }
            });

            sinon.assert.calledOnce(onManageFilters);
        });

        test("When the DialogsOverrider is constructed, it is provided with the correct callbacks", function () {
            sinon.assert.calledOnce(this.dialogsOverrider.constructor);

            var options = this.dialogsOverrider.constructor.args[0][0];

            this.detailsLayout.getActiveIssueId.returns(1);
            equal(options.getIssueId(), 1);

            var dialogMock = {};
            dialogMock.$activeTrigger = {};
            dialogMock.$activeTrigger.data = sinon.stub();
            dialogMock.$activeTrigger.data.withArgs("issueid").returns(2);
            equal(options.getIssueId(dialogMock), 2);

            this.detailsLayout.getActiveIssueKey.returns("KEY-1");
            equal(options.getIssueKey(), "KEY-1");

            dialogMock.$activeTrigger.data.withArgs("issuekey").returns("KEY-2");
            equal(options.getIssueKey(dialogMock), "KEY-2");
        });

        test("When an issue is removed, it is also removed from the DetailsLayout", function () {
            this.detailsLayout.removeIssue.returns(new jQuery.Deferred().resolve().promise());
            this.dialogsOverrider.trigger('issueDelete', { issueId: 1 });

            sinon.assert.calledOnce(this.issueSearch.removeIssue);
            sinon.assert.calledWith(this.issueSearch.removeIssue, 1);

            sinon.assert.calledOnce(this.detailsLayout.removeIssue);
            sinon.assert.calledWith(this.detailsLayout.removeIssue, 1);
        });

        test("When the DetailsLayout triggers a 'editor:saveSuccess' event, the event is re-triggered as 'editor:saveSuccess'", function() {
            this.detailsLayout.trigger("editor:saveSuccess", {
                event: '123456',
                savedFieldIds: ['fieldId-1'],
                savedFieldTypes: ['fieldType-1'],
                duration: 123
            });

            this.assertEventTriggered("editor:saveSuccess", {
                issueId: '123456',
                savedFieldIds: ['fieldId-1'],
                savedFieldTypes: ['fieldType-1'],
                duration: 123
            });
        });

        test("When the DetailsLayout triggers a 'editor:editField' event, the event is re-triggered as 'editor:editField'", function() {
            var eventData = {
                issueId: '123456',
                issueKey: 'TEST-1',
                fieldId: 'field-123',
                fieldType: 'field-type'
            };
            this.detailsLayout.trigger("editor:editField", eventData);

            this.assertEventTriggered("editor:editField", eventData);
        });

        test("When the DetailsLayout triggers a 'editor:editFieldCancel' event, the event is re-triggered as 'editor:editFieldCancel'", function() {
            var eventData = {
                issueId: '123456',
                issueKey: 'TEST-1',
                fieldId: 'field-123',
                fieldType: 'field-type'
            };
            this.detailsLayout.trigger("editor:editFieldCancel", eventData);

            this.assertEventTriggered("editor:editFieldCancel", eventData);
        });

        test("When focusing the issue list, it delegates the call to the internal DetailsLayout component", function() {
            this.issueSearch.focusIssueList();

            sinon.assert.calledOnce(this.detailsLayout.focusList);
        });

        test("When focusing the issue editor, it delegates the call to the internal DetailsLayout component", function() {
            this.issueSearch.focusIssueEditor();

            sinon.assert.calledOnce(this.detailsLayout.focusEditor);
        });

        test("When the DetailsLayout triggers an 'individualPanelRendered' event, it is re-triggered with the same payload", function() {
            var payload = {data: 123};

            this.detailsLayout.trigger("individualPanelRendered", payload);

            this.assertEventTriggered("individualPanelRendered", payload);
        });

        test("When getting the project id, it delegates the call to the internal DetailsLayout component", function() {
            this.detailsLayout.getActiveProjectId.returns(123);

            var projectId = this.issueSearch.getProjectId();

            sinon.assert.calledOnce(this.detailsLayout.getActiveProjectId);
            equal(projectId, 123);
        });

        test("When getting the project key, it delegates the call to the internal DetailsLayout component", function() {
            this.detailsLayout.getActiveProjectKey.returns('TEST');

            var projectKey = this.issueSearch.getProjectKey();

            sinon.assert.calledOnce(this.detailsLayout.getActiveProjectKey);
            equal(projectKey, "TEST");
        });

        test("When getting the project type, it delegates the call to the internal DetailsLayout component", function() {
            this.detailsLayout.getActiveProjectType.returns('software');

            var projectType = this.issueSearch.getProjectType();

            sinon.assert.calledOnce(this.detailsLayout.getActiveProjectType);
            equal(projectType, "software");
        });

    });
});
