define("hipchat/widget/roomselector/roomselector-view",
        [
            "jquery",
            "backbone",
            "hipchat/widget/roomselector/roommapping-service",
            "hipchat/scopes-provider"
        ], function (
                $,
                Backbone,
                RoomMappingService,
                HipChatScopesProvider
        ) {

            var PRIVATE_ROOM_CLASS = "private-room";
            var PUBLIC_ROOM_CLASS = "public-room";

            function escapeResult(markup) {
                var replace_map = {
                    '\\': '&#92;',
                    '&': '&amp;',
                    '<': '&lt;',
                    '>': '&gt;',
                    '"': '&quot;',
                    "'": '&#39;',
                    "/": '&#47;'
                };

                return String(markup).replace(/[&<>"'\/\\]/g, function (match) {
                    return replace_map[match];
                });
            }

            function getProjectKey() {
                return $("[name='projectKey']").attr("content");
            }

            function makeSuggestionsHandler(loggedIn, hasManageRoomScope) {
                //this is only called when the integration is installed
                var roomServicePromise = RoomMappingService.roomServicePromise(true, getProjectKey());

                return Class.extend({
                    _formatResponse: function(rooms, query, showCreateRoom) {
                        var groupDescriptors = [];
                        if (rooms.length > 0) {

                            var suggestionsGroupDescriptor = new AJS.GroupDescriptor({
                                weight: 0,
                                label: loggedIn? AJS.I18n.getText("jira.plugins.hipchat.roommapping.rooms.all.label") :
                                        AJS.I18n.getText("jira.plugins.hipchat.roommapping.rooms.public.label")
                            });

                            _.each(rooms, function(item) {
                                suggestionsGroupDescriptor.addItem(new AJS.ItemDescriptor({
                                    value: item.id,
                                    label: item.text,
                                    html: escapeResult(item.text),
                                    title: item.text,
                                    meta: {
                                        roomId: item.id,
                                        roomName: item.roomName || item.text,
                                        existing: true
                                    },
                                    styleClass: item.isPrivate ? PRIVATE_ROOM_CLASS : PUBLIC_ROOM_CLASS
                                }));
                            });

                            groupDescriptors.push(suggestionsGroupDescriptor);
                        }


                        if (!loggedIn) {
                            var roomActionsGroupDescriptor = new AJS.GroupDescriptor({
                                weight: 1,
                                footerHtml: JIRA.Templates.HipChat.RoomSelector.loginDropdownFooter({
                                    hasManageRoomScope: hasManageRoomScope
                                })
                            });
                            roomActionsGroupDescriptor.addItem(new AJS.ItemDescriptor({
                                label: "filler",
                                html: "filler",
                                highlighted: true
                            }));

                            groupDescriptors.push(roomActionsGroupDescriptor);
                        } else if (showCreateRoom) {
                            var createRoomGroupDescriptor = new AJS.GroupDescriptor({
                                weight: 1,
                                label: AJS.I18n.getText("jira.plugins.hipchat.roommapping.rooms.create.label")
                            });

                            createRoomGroupDescriptor.addItem(new AJS.ItemDescriptor({
                                value: query,
                                label: query,
                                html: escapeResult(query) + "<span class='aui-lozenge aui-lozenge-subtle aui-lozenge-complete'>new</span>",
                                title: query,
                                meta: {
                                    roomId: query,
                                    roomName: query,
                                    existing: false
                                },
                                highlighted: true,
                                styleClass: PUBLIC_ROOM_CLASS
                            }));

                            groupDescriptors.push(createRoomGroupDescriptor);
                        }

                        return groupDescriptors;
                    },

                    execute: function(query) {
                        var deferred = jQuery.Deferred();
                        var self = this;
                        roomServicePromise.done(function(roomService) {
                            var rooms = roomService.getRooms();
                            var searchStr = query.toLocaleLowerCase();
                            var exactMatch = false;
                            var filteredRooms = _.filter(rooms, function(item) {
                                var lowerCasedItem = item.text.toLocaleLowerCase();
                                if (lowerCasedItem === searchStr) {
                                    exactMatch = true;
                                }

                                return lowerCasedItem.indexOf(searchStr) >= 0;
                            });

                            var showCreateRoom = hasManageRoomScope && (!exactMatch && searchStr.length > 0);
                            var suggestions = self._formatResponse(filteredRooms, query, showCreateRoom);

                            JIRA.trace("ajax.request.completed.RoomSelector");
                            deferred.resolve(suggestions, query);
                        }).fail(function(err) {
                            if (loggedIn && (err.status == 403 || err.status == 401)) {
                                var dialog = new AJS.Dialog({
                                    width: 600,
                                    height: 200,
                                    id: "oauth-failure",
                                    closeOnOutsideClick: true
                                });
                                dialog.addHeader("Authentication Failure");
                                dialog.addPanel("panel 0", JIRA.Templates.HipChat.Project.IssuePanel.oauthNoLongerValidMessage({}));
                                dialog.show();
                                deferred.reject(err);
                            } else {
                                var errorGroupDescriptor = new AJS.GroupDescriptor({
                                    weight: 1,
                                    footerHtml: JIRA.Templates.HipChat.RoomSelector.errorDropdownFooter({})
                                });
                                errorGroupDescriptor.addItem(new AJS.ItemDescriptor({
                                    label: "filler",
                                    html: "filler",
                                    highlighted: true
                                }));

                                deferred.resolve([errorGroupDescriptor], query);
                            }
                        });

                        return deferred;
                    }
                });
            }

            function initRoomSelector($el, loggedIn, maxWidth) {

                var hasManageRoomScope = HipChatScopesProvider.hasScope("manage_rooms");
                return new AJS.SingleSelect({
                    element: $el,
                    width: maxWidth,
                    suggestionsHandler: makeSuggestionsHandler(loggedIn, hasManageRoomScope),
                    submitInputVal: true,
                    matchingStrategy: "(.*)()({0})(.*)" // expected groups: prefix, spaceOrParenthesis (ignored), match, suffix
                });
            }

            var RoomSelectorView = Backbone.View.extend({

                events: {
                },

                initialize: function(options) {

                    if (!this.$el.length) {
                        return;
                    }

                    this.roomSelector = initRoomSelector(this.$el, options.loggedIn, options.maxWidth);
                    var self = this;
                    this.roomSelector.model.$element.on("selected", function(e, itemDescriptor) {
                        if (itemDescriptor.properties.value !== "") {
                            self.roomSelector.$field.addClass(itemDescriptor.properties.styleClass);
                            self.trigger("change");
                        } else {
                            self.roomSelector.$field.removeClass(PUBLIC_ROOM_CLASS).removeClass(PRIVATE_ROOM_CLASS);
                            self.trigger("clear");
                        }
                    });

                    this.roomSelector.model.$element.on("unselect", function(e) {
                        self.trigger("clear");
                    });
                },

                getSelectedRoom: function() {
                    var selectedDescriptor = this.roomSelector.getSelectedDescriptor();
                    var meta = JSON.parse(selectedDescriptor.properties.meta);
                    return {
                        id: meta.roomId,
                        roomName: meta.roomName,
                        existing: meta.existing
                    };
                },

                clearSelectedRoom: function() {
                    this.roomSelector.clear();
                    this.roomSelector._deactivate(); // We need that to get the placeholder back
                }
            });


            return RoomSelectorView;
        });