define("hipchat/widget/roomselector/roommapping-service",
[
    "wrm/context-path",
    "ajs", // only to make karma happy
    "jquery",
    "underscore"
],
function (
    wrmContextPath,
    AJS,
    $,
    _
) {

    var roomServicePromise = (function() {
        var roomMap = {}; // { "roomId": { roomId: "roomId", roomName: "roomName" }, ... }
        var rooms = []; // cache of [ { id: "roomId", text: "roomName } ] for use by select2 query. Derived from roomMap
        var listeners = {};

        function rebuildRoomData() {
            rooms = _.map(roomMap, function(item) { return {id: item.roomId, text: item.roomName, isPrivate: item.isPrivate}; });
            rooms = _.sortBy(rooms, function (item) { return item.text.toLocaleLowerCase(); });
            _.each(listeners, function(fn) {
                fn.apply(null, [ rooms ]);
            });
        }

        function createRoom(name) {
            // Note that we set cache to false to prevent IE9 caching previous results. See CONFDEV-29008.
            var dfd = $.Deferred();
            $.ajax({
                url: wrmContextPath() + '/rest/hipchat/integration/1.0/rooms?roomName=' + name,
                type: 'POST',
                dataType: 'json',
                cache: false
            }).done(function(data) {
                var room = {
                    roomId: data.id,
                    roomName: data.name
                };
                addRoom(room);

                dfd.resolve(room);
            }).fail(function(err) {
                dfd.reject(err);
            });

            return dfd;
        }

        function addRoom(room) {
            roomMap[room.roomId] = room;
            rebuildRoomData();
        }

        function removeRoom(room) {
            delete roomMap[room.roomId];
            rebuildRoomData();
        }

        function onChange(fn) {
            listeners[fn] = fn;
        }

        function offChange(fn) {
            delete listeners[fn];
        }

        function getRoomById(roomId) {
            return roomMap[roomId];
        }

        function getRooms() {
            return rooms;
        }

        var roomService = {
            createRoom: createRoom,
            addRoom: addRoom,
            removeRoom: removeRoom,
            getRoomById: getRoomById,
            getRooms: getRooms,
            onChange: onChange,
            offChange: offChange
        };

        var roomsAvailablePromise = (function (isHipChatConfigured, projectKey) {
            var roomLoader = $.Deferred();
            if (isHipChatConfigured) {
                // Note that we set cache to false to prevent IE9 caching previous results. See CONFDEV-29008.
                $.ajax({
                    url: wrmContextPath() + '/rest/hipchat/integration/1.0/rooms?projectKey=' + (_.isString(projectKey) ?
                            projectKey : ""),
                    dataType: 'json',
                    cache: false
                }).done(function (data) {
                    roomMap = _.object(_.map(data, function (item) {
                        return [item.id, {
                            roomId: item.id,
                            roomName: item.name,
                            isPrivate: item["private"]
                        }];
                    }));
                    rebuildRoomData(roomMap);
                    roomLoader.resolve(roomService);
                }).fail(function (err) {
                    roomLoader.reject(err);
                });
            } else {
                roomLoader.resolve(roomService);
            }
            return roomLoader.promise();
        });

        var promises = {};
        var promiseBuilder = function (isHipChatConfigured, projectKey) {
            if (promises[projectKey] === undefined) {
                promises[projectKey] = roomsAvailablePromise(isHipChatConfigured, projectKey);
            }

            return promises[projectKey];
        };
        promiseBuilder.reset = function() {
            promises = {};
        };

        return promiseBuilder;
    })();

    return {
        roomServicePromise: roomServicePromise
    };
});
