define("hipchat/feature/issuepanel/invitecontributors-view",
[
    "jquery",
    "backbone",
    "hipchat/base",
    "hipchat/feature/issuepanel/invitecontributors",
    "hipchat/invite/user-picker"
], function (
        $,
        Backbone,
        HipChat,
        InviteContributors,
        UserPicker
) {
    var InviteContributorsView = HipChat.View.extend({

        events: {
        },

        initialize: function (options) {
            this.loggedIn = options.loggedIn;
            this.canCreateRoom = options.canCreateRoom;
            this.canAssignRoom = options.canAssignRoom;
            this.isAdmin = options.isAdmin;
            this.hipChatUserName = options.hipChatUserName;
            this.issueKey = options.issueKey;
        },

        close: function () {
            this.unbind();
        },

        openInviteDialog: function (roomId) {
            var self = this;
            var issueKey = self.issueKey;
            if (!self.dialog) {
                self.dialog = new JIRA.FormDialog({
                    id: "hipchat-invite-contributors-dialog",
                    ajaxOptions: {
                        url: AJS.contextPath() + '/secure/HipChatInviteContributors.jspa?issueKey=' + issueKey,
                        data: {
                            decorator: "dialog",
                            inline: "false"
                        }
                    },
                    width: 400
                });
                self.initInviteContributorsDialog(self.dialog);
            }

            self.dialog.show();
        },

        initInviteContributorsDialog: function (dialog) {
            var self = this;
            self.dialogContentReadyCallback = function(e, data) {
                if (data.options.id === "hipchat-invite-contributors-dialog") {
                    self.initializeDialogUi();
                    self.initializeUserPicker();
                    self.initializeContributorsList();
                    // To make sure this block runs only once.
                    $(document).unbind("dialogContentReady", self.dialogContentReadyCallback);
                }
            };
            $(document).bind("dialogContentReady", {}, self.dialogContentReadyCallback);
        },

        getEmailDomain: function (email) {
                var atIdx;
                if(email && (atIdx = email.indexOf('@')) && atIdx >= 0) {
                    return email.substr(atIdx);
                }
                // Fallback
                return email;
        },

        setInviteButtonStatus: function() {
            var self = this;
            var hipChatUsersMap = self.model.get("hipChatUsersMap");
            var anyoneToInvite = false;
            var userKeys = Object.keys(hipChatUsersMap);
            userKeys.forEach(function (userKey) {
                if (!hipChatUsersMap[userKey].invitedAlready)
                    anyoneToInvite = true;
            });
            self.$inviteButton.prop('disabled', !anyoneToInvite);
        },

        addHipChatUser: function (hipChatUser) {
            var self = this;
            var hipChatUsersMap = self.model.get("hipChatUsersMap");
            if (hipChatUser.userKey in hipChatUsersMap) {
                console.log("User " + hipChatUser.userKey + " is already in the list, ignoring.");
                return;
            }
            if (!hipChatUser.emailDomain) {
                hipChatUser.emailDomain = self.getEmailDomain(hipChatUser.emailAddress);
            }
            hipChatUsersMap[ hipChatUser.userKey ] = hipChatUser;
            self.$inviteeList.append(self.inviteeTemplate({user: hipChatUser}));
            self.setInviteButtonStatus();
        },

        removeHipChatUser: function (userKey) {
            var self = this;
            var invitee = self.$inviteeList.find("[data-username='" + userKey + "']");
            invitee.remove();
            var hipChatUsersMap = self.model.get("hipChatUsersMap");
            if (!(userKey in hipChatUsersMap)) {
                console.log("User " + userKey + " is not in the list but it is asked to be removed, WTF?");
                return;
            }
            hipChatUsersMap = _.omit(hipChatUsersMap, userKey);
            self.model.set("hipChatUsersMap", hipChatUsersMap);
            self.setInviteButtonStatus();
        },

        submitInviteContributorsDialog: function () {
            var self = this;

            self.$inviteButton.attr("disabled", "disabled");
            self.$userFieldContainer.prop('disabled', "disabled");
            self.dialog.showFooterLoadingIndicator();
            var $errorContainer = $("#hipchat-invite-contributors-dialog").find(".dialog-errors");
            var dfd = $.Deferred();

            $errorContainer.empty();
            var hipChatUsersMap = self.model.get("hipChatUsersMap");
            var userKeys = Object.keys(hipChatUsersMap);
            if (!userKeys.length) {
                console.log("No user to be invited but this method is called, WTF?");
                return;
            }
            var mentionNames = [];
            userKeys.forEach(function (userKey) {
                var mentionName = hipChatUsersMap[userKey].mentionName;
                if (!hipChatUsersMap[userKey].invitedAlready && mentionNames.indexOf(mentionName) < 0)
                    mentionNames.push(mentionName);
            });
            self.model.save({mentionNames: mentionNames})
                    .done(function (response) {
                        dfd.resolve();
                        self.dialog.hideFooterLoadingIndicator();
                    }).fail(function (err) {
                        console.log('An error occurred while making a server-side call to invite contributors.');
                        dfd.reject(err);
                    });

            var updateStatus = function(success) {
                // Hide the remove buttons
                var inviteeIcons = self.$form.find(".icon-holder");
                inviteeIcons.each(function(index) {
                    $(this).hide();
                });
                // Show success icons
                var $inviteeStatuses = self.$form.find(".status-icon-holder");
                $inviteeStatuses.show();
                $inviteeStatuses.each(function(index) {
                    var $element = $(this);
                    if (!$element.data('already-invited')) {
                        if (success) {
                            $element.data('already-invited', true);
                        }
                        var $successIcon = $(this).find(".success");
                        var $failureIcon = $(this).find(".failure");
                        setTimeout(function() {
                            $($successIcon).css('display', success ? 'inline-block' : 'none');
                            $($failureIcon).css('display', success ? 'none' : 'inline-block');
                        }, 100 * index);
                    }
                });
            };

            dfd.done(function (room) {
                updateStatus(true);
                // Mark users
                var hipChatUsersMap = self.model.get("hipChatUsersMap");
                var userKeys = Object.keys(hipChatUsersMap);
                userKeys.forEach(function (userKey) {
                    hipChatUsersMap[userKey].invitedAlready = true;
                });
                self.model.set("hipChatUsersMap", hipChatUsersMap);
                JIRA.trace("hipchat.invite.contributors.successful");
                // $('#hipchat-invite-contributors-dialog-close').click();
            }).fail(function (err) {
                updateStatus(false);
                $errorContainer.empty();
                $errorContainer.append(AJS.I18n.getText("jira.plugins.hipchat.invitecontributors.dialog.error.invite", err.statusText));
                self.$inviteButton.removeAttr("disabled");
                self.$userFieldContainer.removeAttr("disabled");
            }).always(function () {
                self.dialog.hideFooterLoadingIndicator();
            });
        },

        initializeDialogUi: function () {
            var self = this;
            self.$form = $("#hipchat-invite-contributors-dialog").find("form");
            self.$userFieldContainer = self.$form.find('#dedicated-room-invite-users-picker-container');
            self.$userField = self.$form.find('#dedicated-room-invite-user');
            self.$closeButton = $('#hipchat-invite-contributors-dialog-close');
            self.$inviteButton = $('#hipchat-invite-contributors-dialog-submit');
            self.$inviteeList = self.$form.find('.hipchat-invitee-list');
            self.inviteeTemplate = JIRA.Templates.HipChat.Project.IssuePanel.invitee;

            self.$inviteeList.on('click', '.remove-invite-button', function (e) {
                e.preventDefault();
                var invitee = $(this).closest('.hipchat-invitee');
                self.removeHipChatUser(invitee.attr('data-username'));
            });

            self.$form.on('user-picker-user-selected', function (e, user) {
                self.addHipChatUser(user);
            });

            self.$inviteButton.on("click", function (e) {
                self.submitInviteContributorsDialog();
            });
        },

        initializeContributorsList: function() {
            var self = this;
            var issueKey = AJS.Meta.get("issue-key");
            self.model = new InviteContributors({issueKey: issueKey});
            self.setInviteButtonStatus();
            var $errorContainer = $("#hipchat-invite-contributors-dialog").find(".dialog-errors");
            $errorContainer.empty();
            var $spinner = self.$form.find('.hc-button-spinner');
            $spinner.css({display: "inline-block"});
            $spinner.spin();
            self.model.fetch({
                success: function (result) {
                    $spinner.spinStop();
                    $spinner.css({display: "none"});
                    self.$inviteeList.empty();
                    result.attributes.hipChatUsers.forEach(function (hipChatUser) {
                        self.addHipChatUser(hipChatUser);
                    });
                },
                error: function (result) {
                    $spinner.spinStop();
                    $spinner.css({display: "none"});
                    $errorContainer.empty();
                    $errorContainer.append(AJS.I18n.getText("jira.plugins.hipchat.invitecontributors.dialog.error.fetch"));
                }
            });
        },

        initializeUserPicker: function () {
            var self = this;

            function makeParams(val) {
                return {
                    maxResults: 10,
                    query: val,
                    showAvatar: true
                };
            }

            function processResults(data, page) {
                var results = _.reduce(data.users, function (m, item) {
                    m.push({
                        id: item.name,
                        text: item.displayName,
                        restObj: item
                    });
                    return m;
                }, []);
                return {
                    results: results
                };
            }

            function processChoice(props) {
                var deferred = $.Deferred();
                var $errorContainer = $("#dedicated-room-invite-users-picker-container").find(".user-picker-errors");

                $errorContainer.empty();
                var $spinner = self.$form.find('.hc-button-spinner');
                $spinner.css({display: "inline-block"});
                $spinner.spin();
                getExtraData(props.restObj.key).done(function (contributor) {
                    deferred.resolve(contributor);
                }).always(function () {
                    $spinner.spinStop();
                    $spinner.css({display: "none"});
                }).fail(function (err) {
                    $errorContainer.empty();
                    $errorContainer.append(AJS.I18n.getText("jira.plugins.hipchat.invitecontributors.not.hipchat.user.error"));
                    self.$userField.select2('val', '');
                    deferred.reject(err);
                });

                return deferred.promise();
            }

            function getExtraData(userKey) {
                var url = AJS.contextPath() + '/rest/hipchat/integrations/1.0/invite/user/' + userKey;
                var deferred = $.Deferred();
                $.ajax(url, {
                    type: "GET",
                    dataType: 'json'
                }).done(function (contributor) {
                    if (contributor.hipChatUser)
                        deferred.resolve(contributor);
                    else
                        deferred.reject();
                }).fail(function (err) {
                    deferred.reject(err);
                });
                return deferred.promise();
            }

            UserPicker(self.$userField, self.$form, {
                url: AJS.contextPath() + "/rest/api/2/user/picker",
                params: makeParams,
                results: processResults,
                choice: processChoice
            });
        }
    });

    return InviteContributorsView;
});