define("jira/editor/plugins/editor-content-postprocess", [
    "jira/editor/tinymce",
    "jira/editor/converter/util/strings",
    "jira/editor/util/temporaryAttachment",
    "jira/editor/analytics"
], function (tinymce,
             Strings,
             temporaryAttachmentUtil,
             Analytics
) {
    "use strict";

    var hasClass = function (node, cls) {
        var attr = node && node.attr('class') || '';
        return attr.indexOf(cls) >= 0;
    };

    var handleContentEditableFalse = function (editor) {
        editor.parser.addAttributeFilter('class', function (nodes) {
            var i = nodes.length;
            /**@type {tinymce.html.Node} */
            var node;
            while (i--) {
                node = nodes[i];
                if (hasClass(node, 'user-hover')) {
                    node.attr('contenteditable', 'false');
                    node.attr('tabindex', '-1');
                }
            }
        });
    };

    var stripEmptyAFromHeaders = function (editor) {
        editor.parser.addNodeFilter('h1,h2,h3,h4,h5,h6', function (nodes) {
            var getEmptyImmediateChildren = function (node, name) {
                var parent = node;
                var collection = [];

                for (node = parent.firstChild; node; node = node.next) {
                    if (node.name === name && !node.firstChild) {
                        collection.push(node);
                    }
                }
                return collection;
            };

            var i = nodes.length;
            /**@type {tinymce.html.Node} */
            var node;
            while (i--) {
                node = nodes[i];
                var children = getEmptyImmediateChildren(node, 'a');
                var filtered = children.filter(function withNameAttribute(node) {
                    return typeof node.attr('name') !== 'undefined';
                });
                filtered.forEach(function (node) {
                    node.remove();
                });

            }
        });
    };

    var normalizeCode = function (editor) {
        editor.parser.addNodeFilter('div,pre', function (nodes) {
            nodes.forEach(function (node) {
                if (node.parent && node.parent.name === "pre" && node.name !== "panel-title") {
                    node.unwrap();
                }

                if (!node.attr("class") && node.name === "pre") {
                    node.unwrap();
                }

                if (!hasClass(node, 'panel') || hasClass(node, 'panelHeader') || hasClass(node, 'panelContent')) {
                    return;
                }

                if (hasClass(node, 'preformatted')) {
                    node.attr('class', 'noformat panel');
                }
                if (node.lastChild && node.lastChild.firstChild && /^code/.test(node.lastChild.firstChild.attr("class"))) {
                    node.attr("data-language", node.lastChild.firstChild.attr("class"));
                    node.attr("class", "code panel");
                }

                var headerNode = node.firstChild;
                if (hasClass(headerNode, 'panelHeader')) {
                    var style = headerNode.attr('style');
                    headerNode.firstChild.unwrap();
                    headerNode.name = 'panel-title';
                    headerNode.attr('class', style);
                }
                var contentNode = node.lastChild;
                if (hasClass(contentNode, 'panel') || hasClass(contentNode, 'panelContent')) {
                    var style = contentNode.attr('style');
                    if (hasClass(node, 'noformat') || hasClass(node, 'code')) {
                        contentNode.firstChild.unwrap();
                    }
                    contentNode.unwrap();
                }

                if (hasClass(node, 'noformat') || hasClass(node, 'code')) {
                    node.name = 'pre';
                } else {
                    node.attr("class", "plain panel");
                }
            });
        });
    };

    /**
     * When you paste link to temporary attachment from other form - remove such attachment from pasted content as it will not render correctly.
     * @param editor
     */
    var removeInvalidTemporaryAttachments = function (editor) {
        editor.parser.addNodeFilter('img', function (nodes) {
            if (nodes.length === 0) {
                return;
            }
            var formToken = temporaryAttachmentUtil.getFormToken(editor.getContainer());
            if (!formToken) {
                console.warn('formToken not found, skipping invalid temporary attachment content post processing');
                return;
            }

            var isInvalidTemporaryAttachment = function (node) {
                var src = node.attr('src');
                if (Strings.isTemporaryAttachmentPath(src)) {
                    var urlPartial = '/secure/temporaryattachment/' + formToken;
                    return !Strings.contains(src, urlPartial);
                }
                return false;
            };

            nodes.filter(isInvalidTemporaryAttachment).forEach(function (node) {
                node.remove();
                Analytics.sendEvent('editor.plugin.content-postprocess.temporary-attachment.removed');
            });
        });
    };

    tinymce.create('tinymce.plugins.jira.EditorContentPostprocess', {
        init: function (editor) {
            editor.on('PreInit', function () {
                removeInvalidTemporaryAttachments(editor);
                handleContentEditableFalse(editor);
                stripEmptyAFromHeaders(editor);
                normalizeCode(editor);
            });
        }
    });

    tinymce.PluginManager.add('jira.editorcontentpostprocess', tinymce.plugins.jira.EditorContentPostprocess);
    tinymce.PluginManager.urls["jira.editorcontentpostprocess"] = true;

    return {
        configure: function (instance, settings) {
            settings.plugins.push('jira.editorcontentpostprocess');
            settings.editor_content_postprocess = {};

            return settings;
        }
    };
});