define("jira/editor/selection", [
    "jira/editor/constants"
], function (
    Constants
) {
    "use strict";

    var EditorSelection = function (editor) {
        this.editor = editor;
    };

    /**
     * Removes spaces from the selection within #text nodes.
     *
     * @returns {Boolean} true if selection has been changed
     */
    EditorSelection.prototype.trimSelection = function () {
        var changed = false;
        var range = this.editor.selection.getRng();
        if (range.collapsed) { // there is nothing to trim
            return false;
        }

        if (!range.toString().trim()) { // selection contains only whitespaces
            range.collapse(true);
            this.editor.selection.setRng(range);
            this.editor.focus();
            return true;
        }

        var start = range.startContainer;
        var startOffset = range.startOffset;
        var end = range.endContainer;
        var endOffset = range.endOffset;

        // left side
        if (start && start.nodeType === Node.TEXT_NODE) {
            var text = start.nodeValue.substring(startOffset);
            var spaces = matchContent(text, Constants.LEFT_WHITESPACE_MATCHER);
            if (spaces) {
                changed = true;

                if (text.length === spaces.length) {
                    range.setStartAfter(start);
                } else {
                    range.setStart(start, startOffset + spaces.length)
                }
            }
        }

        // right side
        if (end && end.nodeType === Node.TEXT_NODE) {
            var text = end.nodeValue.substring(0, endOffset);
            var spaces = matchContent(text, Constants.RIGHT_WHITESPACE_MATCHER);
            if (spaces) {
                changed = true;

                if (text.length === spaces.length) {
                    range.setEndBefore(end);
                } else {
                    range.setEnd(end, endOffset - spaces.length)
                }
            }
        }

        if (changed) {
                this.editor.selection.setRng(range);
        }
        return changed;
    };

    /**
     * @returns {Boolean} true if the current selection is not collapsed
     */
    EditorSelection.prototype.hasSelection = function () {
        return !this.editor.selection.getRng().collapsed;
    };

    return EditorSelection;


    function matchContent(text, regex) {
        var whitespaces = text.match(regex);
        return (null !== whitespaces) ? whitespaces[0] : '';
    }
});