define("jira/editor/schema", [
    "jira/editor/schema-builder",
    "jira/richeditor/editor-supported-operations",
    "jira/editor/tinymce",
    "jquery"
], function(
    SchemaBuilder,
    supportedOperations,
    tinymce,
    $
) {
    "use strict";

    var unbreakableNodes = ["strong", "b", "em", "i", "tt", "code", "samp", "sub", "sup", "del", "cite", "panel-title"];
    var unsupportedNesting = [["td", "table"], ["th", "table"], ["td", "p"], ["th", "p"]];
    var unwantedAttributes = [];
    var extendedValidElements = ["strong/b"];
    unwantedAttributes.push("xml:lang");
    unwantedAttributes.push("xml:space");

    var EditorSchema = {};
    EditorSchema.getSchemaSpecBuilder = function (builder) {
        if (undefined === builder) {
            builder = new SchemaBuilder();
        }
        return builder
            .withValidElements(supportedOperations.getRequiredHtmlTags())
            .withNoBreak(unbreakableNodes)
            .withNoSelfNest(unbreakableNodes)
            .withNoNest(unsupportedNesting)
            .withUnwantedAttributes(unwantedAttributes)
            .withExtendedValidElements(extendedValidElements)
            .withCustomElement("panel-title", ["span", "#text"], ["style"])
            .setValidChildren("pre", "br|span|#text|pre|p|panel-title")
    };
    EditorSchema.getPasteSchemaSpecBuilder = function (builder) {
        if (undefined === builder) {
            builder = EditorSchema.getSchemaSpecBuilder();
        }
        return builder
            .withUnwantedAttributes(["contenteditable"])
    };
    EditorSchema.getPasteInsidePreSchemaSpecBuilder = function (builder) {
        if (undefined === builder) {
            builder = new SchemaBuilder();
        }
        return builder
            .withValidElements(["br", "span", "p", "pre"])
            .withUnwantedAttributes(["style"])
    };

    EditorSchema.getSchemaFromSpec = function (schemaSpec) {
        var schema = new tinymce.html.Schema({
            schema: schemaSpec.getType(),
            custom_elements: schemaSpec.getCustomElements(),
            valid_children: schemaSpec.getValidChildren(),
            valid_elements: schemaSpec.getValidElements(),
            extended_valid_elements: schemaSpec.getExtendedValidElements()
        });
        schemaSpec.fixChildren(schema);
        return schema;
    };

    function sanitizeMarkup(html) {
        if( "textContent" in document.body ) {
            html = html.replace(/(<br>|<br \/>)/g, '\n')
        }
        var node = $('<div>').addClass('richeditor-sanitize').append(html)[0];
        try {
            // this is is needed for FF, otherwise it removes new lines
            $(node).appendTo(document.body);
            return AJS.escapeHTML(node.textContent || node.innerText);
        } finally {
            $(node).remove();
        }
    }

    EditorSchema.sanitizeHtml = function sanitizePastedHtml(html, editor, schemaSpec) {
        var $selection = $(editor.selection.getStart()).add(editor.selection.getEnd()).add(editor.selection.getNode());

        if ($selection.is('pre, pre > *')) {
            html = $selection.clone().html(sanitizeMarkup(html)).prop("outerHTML");
        }

        if ($selection.is('panel-title, panel-title > *')) {
            html = sanitizeMarkup(html);
        }

        var schema = EditorSchema.getSchemaFromSpec(schemaSpec);
        var htmlDomParser = new tinymce.html.DomParser(editor.settings, schema);
        var htmlSerializer = new tinymce.html.Serializer(editor.settings, schema);

        var dom = htmlDomParser.parse(html, {
            forced_root_block: undefined
        });
        return htmlSerializer.serialize(dom);
    };

    return EditorSchema;
});