define("jira/editor/schema-builder", [
    "jira/editor/tinymce",
    "jquery",
    "underscore",
    "jira/util/logger"
], function(
    tinymce,
    $,
    _,
    logger
) {
    "use strict";

    var SchemaBuilder = function() {
        this.noBreak = [];
        this.noNest = [];
        this.customElementChildren = {};
        this.customElementAttrs = {};
        this.validChildren = [];
        this.unwantedAttributes = [];
        this.extendedValidElements = [];
        this.baseSchema = 'html5';
        this.validElements = [];
    };
    /**
     * Registers a custom element along with allowed children.
     * The text node is added by default.
     *
     * @example
     * withCustomElement('x-task', ['p']);
     * withCustomElement('x-task', false);
     * withCustomElement('x-task');
     * withCustomElement('x-task', ['p'], ['content', 'done']);
     *
     * @param {string} name custom element name
     * @param {array.string|false=} children
     *  when array provided: tag names of allowed children;
     *  when false: no children allowed;
     *  when undefined / unspecified: allow p and #comment nodes
     * @param {array.string=} attributes list of allowed attributes
     * @returns {SchemaBuilder}
     */
    SchemaBuilder.prototype.withCustomElement = function (name, children, attributes) {
        if (!isValidCustomElementName(name)) {
            logger.error('Could not add custom element: incorrect `name` given:', name,
                '(see https://www.w3.org/TR/custom-elements/#valid-custom-element-name)');
            return this;
        }
        if (!isValidChildrenSpec(children)) {
            logger.error('Could not add custom element: incorrect `children` given:', children);
            return this;
        }
        if (!isValidAttrSpec(attributes)) {
            logger.error('Could not add custom element: incorrect `attributes` given:', attributes);
            return this;
        }

        if (false === children) {
            children = [];
        } else if (undefined === children) {
            // adding '#comment' to be consistent with tinymce addValidChildren method
            children = ['p', '#comment'];
        } else {
            children.push('#comment');
        }

        if (!attributes) {
            attributes = [];
        }

        this.customElementChildren[name] = _.uniq(children);
        this.customElementAttrs[name] = _.uniq(attributes);
        return this;
    };
    SchemaBuilder.prototype.withBaseSchema = function withBaseSchema(schema) {
        this.baseSchema = schema;
        return this;
    };
    SchemaBuilder.prototype.withValidElements = function withValidElements(validElements) {
        this.validElements = validElements;
        return this;
    };
    SchemaBuilder.prototype.withNoBreak = function withNoBreak(elements) {
        this.noBreak.push.apply(this.noBreak, elements);
        return this;
    };
    SchemaBuilder.prototype.withNoSelfNest = function withNoNest(elements) {
        this.noNest.push.apply(this.noNest, elements.map(function (el) { return [el, el]; }));
        return this;
    };
    SchemaBuilder.prototype.setValidChildren = function setValidChildren(parent, childrenSpec) {
        this.validChildren.push(parent + "[" + childrenSpec + "]");
        return this;
    };
    SchemaBuilder.prototype.withExtendedValidElements = function (extendedValidElements) {
        this.extendedValidElements.push.apply(this.extendedValidElements, extendedValidElements);
        return this;
    };
    /**
     * @param {Array} parentChildElements [ [parent, child], [parent, child], ... ]
     * @returns {SchemaBuilder}
     */
    SchemaBuilder.prototype.withNoNest = function withNoNest(parentChildElements) {
        this.noNest.push.apply(this.noNest, parentChildElements);
        return this;
    };
    SchemaBuilder.prototype.withUnwantedAttributes = function withUnwantedAttributes(attributes) {
        this.unwantedAttributes.push.apply(this.unwantedAttributes, attributes);
        return this;
    };
    SchemaBuilder.prototype.build = function build() {
        var baseSchema = this.baseSchema;
        var customElements = [];
        var validChildren = [];
        var validElements = [];
        var extendedValidElements = [];
        var validChildrenForCustomElements = {};

        this.noBreak.forEach(function(tagName) {
            noBreak(tagName);
        });
        this.noNest.forEach(function(parentChild) {
            noNest.apply(null, parentChild);
        });

        customElements.push.apply(customElements, Object.keys(this.customElementChildren));
        // tinymce schema.children structure: { 'node': {}, '#text': {}, ... }
        $.each(this.customElementChildren, function(name, children) {
            validChildrenForCustomElements[name] = children.reduce(function(spec, child) {
                spec[child] = {};
                return spec;
            }, {});
        });

        validChildren.push.apply(validChildren, this.validChildren);

        validElements = decorateElements(
            this.validElements,
            this.unwantedAttributes,
            baseSchema);

        extendedValidElements.push.apply(extendedValidElements, _.map(this.customElementAttrs, function(attrs, name) {
            return name + (attrs.length ? '[' + attrs.join('|') + ']' : '');
        }));
        extendedValidElements.push.apply(extendedValidElements, this.extendedValidElements);

        return {
            getValidChildren: function getValidChildren() {
                return validChildren.join(",");
            },
            getValidElements: function getValidElements() {
                return validElements.join(",");
            },
            getExtendedValidElements: function getExtendedValidElements() {
                return extendedValidElements.join(",");
            },
            getCustomElements: function() {
                return customElements.join(",");
            },
            getType: function getType() {
                return baseSchema;
            },
            /**
             * Due to tinymce bug custom elements' children need to be set manually on the schema object
             * https://github.com/tinymce/tinymce/issues/3280
             *
             * @param {tinymce.html.Schema} schema children property will be altered
             */
            fixChildren: function(schema) {
                $.extend((schema || {}).children, validChildrenForCustomElements);
            }
        };


        function noNest(parent, child) {
            validChildren.push("-" + parent + "[" + child + "]");
        }

        function noBreak(tagName) {
            noNest(tagName, "br");
        }
    };

    return SchemaBuilder;

    /**
     * Uses tinymce schema but limit it to given set of elements.
     * Thus tinymce convention is used regarding valid elements, children and so on, please see the doc:
     * {@link https://www.tinymce.com/docs/configure/content-filtering/#valid_elements}
     *
     * @param {Array} elements html tags which are allowed to use
     * @param {Array} withoutAttributes attributes which will we removed from schema
     * @param {String} baseSchema any supported value by tinymce such as "html5", "html5-strict", "html4"
     * @returns {Array} valid_elements for tinymce
     */
    function decorateElements(elements, withoutAttributes, baseSchema) {
        var srcSchema = new tinymce.html.Schema({
            schema: baseSchema
        });
        var srcElements = srcSchema.elements;

        // add constrains
        srcElements.img.attributesRequired = ["src"];
        srcElements.img.attributes['imagetext'] = {};
        srcElements.img.attributesOrder.push('imagetext');

        return elements.map(function (el) {
            var srcSpec = srcElements[el];
            if (!srcSpec) {
                // remove element if it does not exist in schema
                return false;
            }

            // see @link: "!" is "Enables removal of elements with no attributes. They can still have content though."
            var postfix = (srcSpec.removeEmptyAttrs) ? "!" : "";
            var prefix = "";
            if (srcSpec.paddEmpty) {
                // see @link: "#" is "Enables padding of empty elements. This will pad with &nbsp; if they are empty."
                prefix = "#";
            } else if (srcSpec.removeEmpty) {
                // see @link: "-" is "Enables removal of empty elements such as <strong />"
                prefix = "-";
            }

            var attributes = srcSpec.attributesOrder.map(function (attr) {
                var prefix = "";
                if ((srcSpec.attributesRequired || []).indexOf(attr) !== -1) {
                    // see @link: "!" is "Makes attributes required. If none of the required attributes are set,
                    //  the element will be removed."
                    prefix = "!";
                }
                return prefix + attr;
            }).filter(function (attr) {
                // remove unwanted attributes such as "contenteditable"
                return withoutAttributes.indexOf(attr) === -1;
            });

            // required format by tinymce, example: #a[!href|title]
            return prefix + el + postfix + "[" + attributes.join("|") + "]";
        }).filter(function (el) {
            return !!el;
        });
    }

    function isValidCustomElementName(name) {
        //@link https://www.w3.org/TR/custom-elements/#valid-custom-element-name
        return typeof name === 'string' && name.indexOf('-') > 0;
    }

    function isValidChildrenSpec(children) {
        if (_.isArray(children)) {
            return _.every(children, function(tagName) {
                return typeof tagName === 'string';
            });
        }
        return false === children || undefined === children;
    }

    function isValidAttrSpec(attrs) {
        if (_.isArray(attrs)) {
            return _.every(attrs, function(attr) {
                return typeof attr === 'string';
            });
        }
        return undefined === attrs;
    }
});