AJS.test.require([
    'com.atlassian.jira.plugins.jira-editor-plugin:analytics',
], function () {

    var Analytics = require('jira/editor/analytics');
    var Performance = window.performance;

    var MARK1 = 'rte-exec-time';
    var MARK2 = 'cross-module';
    var MARK3 = 'extra-attributes';
    var ANALYTIC_PERF_KEY = "editor.instance.perf.";

    module('Analytics measure', {
        setup: function () {
            this.sandbox = sinon.sandbox.create();
            var clock = this.clock = sinon.useFakeTimers();

            var moduleA = {
                stage1: function () {
                    clock.tick(1000);
                    Analytics.mark(MARK2);
                },
                stage2: function () {
                    clock.tick(1000);
                    return Analytics.oneShotMeasure(MARK2, {}, 'stage2', true);
                },
                stage3: function () {
                    clock.tick(1000);
                    return Analytics.oneShotMeasure(MARK2);
                },
                computeSomeNPProblem: function () {
                    Analytics.mark(MARK1);
                    // hard work here
                    clock.tick(1000);
                    return Analytics.oneShotMeasure(MARK1);
                }
            };
            var moduleB = {
                asyncStage: function () {
                    clock.tick(1000);
                    return Analytics.oneShotMeasure(MARK2, {}, 'async-stage', true);
                },
                embedExtraAttributes: function(extraAttributes) {
                    Analytics.mark(MARK3);
                    clock.tick(1000);
                    return Analytics.oneShotMeasure(MARK3, extraAttributes);
                }
            };

            this.sandbox.stub(Analytics, "sendEvent");
            this.context = AJS.test.mockableModuleContext();
            this.context.mock('jira/editor/fake-module-a', moduleA);
            this.context.mock('jira/editor/fake-module-b', moduleB);
        },

        teardown: function () {
            this.clock.restore();
            this.sandbox.restore();
            Performance.clearMarks(MARK1);
            Performance.clearMarks(MARK2);
        }
    });

    test('Should measure time between A and B points', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');

        var duration = moduleA.computeSomeNPProblem();
        ok(duration >= 0, 'measured time should be greater than zero');
    });

    test('Should send an analytic event with default name', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');

        moduleA.computeSomeNPProblem();
        ok(Analytics.sendEvent.calledOnce, '"sendEvent" method should be called exactly once');
        var eventKey = ANALYTIC_PERF_KEY + 'exec-time';
        ok(Analytics.sendEvent.calledWith(eventKey), 'event key should be equal ' + eventKey);

        moduleA.stage1();
        moduleA.stage3();
        ok(Analytics.sendEvent.calledTwice, '"sendEvent" method should be called exactly twice');
        var eventKey = ANALYTIC_PERF_KEY + MARK2;
        ok(Analytics.sendEvent.calledWith(eventKey), 'event key should be equal ' + eventKey);
    });

    test('Should clear marks', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');

        moduleA.computeSomeNPProblem();
        var entries = Performance.getEntriesByName(MARK1);
        ok(!entries.length, '"' + MARK1 + '" marks should be cleared');
    });

    test('Should be able to measure twice based on the one starting point', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');

        moduleA.stage1();
        var duration = moduleA.stage2();
        ok(duration >= 0, 'first measure should be done');

        var duration2 = moduleA.stage3();
        ok(duration2 >= 0, 'second measure should be done');
    });

    test('Should not clear marks when `keepEntries` flag is true', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');

        moduleA.stage1();
        moduleA.stage2();
        var entries = Performance.getEntriesByName(MARK2);
        ok(entries.length, '"' + MARK2 + '" marks should exist');

        moduleA.stage3();
        var entries = Performance.getEntriesByName(MARK2);
        ok(!entries.length, '"' + MARK2 + '" marks should be cleared');
    });

    test('Should send an analytic event with custom name', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');

        moduleA.stage1();
        moduleA.stage2();
        ok(Analytics.sendEvent.calledOnce, '"sendEvent" method should be called exactly once');
        var eventKey = ANALYTIC_PERF_KEY + 'stage2';
        ok(Analytics.sendEvent.calledWith(eventKey), 'event key should be equal ' + eventKey);
    });

    test('Should be able to do cross-module measure', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');
        var moduleB = this.context.require('jira/editor/fake-module-b');

        moduleA.stage1();
        var duration = moduleA.stage2();

        ok(duration >= 0, 'measure inside first module should be done');
        ok(Analytics.sendEvent.calledOnce, '"sendEvent" method should be called exactly once');
        var eventKey = ANALYTIC_PERF_KEY + 'stage2';
        ok(Analytics.sendEvent.calledWith(eventKey), 'event key should be equal ' + eventKey);

        var duration2 = moduleB.asyncStage();

        ok(duration2 >= 0, 'measure inside second module should be done');
        ok(Analytics.sendEvent.calledTwice, '"sendEvent" method should be called exactly twice');
        var eventKey = ANALYTIC_PERF_KEY + 'async-stage';
        ok(Analytics.sendEvent.calledWith(eventKey), 'event key should be equal ' + eventKey);
    });

    test('Should be defensive when User Timing API is not implemented', function () {
        var _mark = Performance.mark;
        try {
            Performance.mark = undefined;
            var moduleA = this.context.require('jira/editor/fake-module-a');
            moduleA.computeSomeNPProblem();

            ok(true, 'Exception has not been thrown');
        } catch (e) {
            ok(false, 'Exception has been thrown');
        } finally {
            Performance.mark = _mark;
        }
    });

    test('Should send an analytic event with default attribute', function () {
        var moduleB = this.context.require('jira/editor/fake-module-b');

        moduleB.embedExtraAttributes(null);
        ok(Analytics.sendEvent.calledWithMatch('', sinon.match.has('time')), '"time" attribute should be sent');
    });

    test('Should send an analytic event with extra attributes', function () {
        var moduleB = this.context.require('jira/editor/fake-module-b');

        moduleB.embedExtraAttributes({abc: 123, count: 10});
        ok(Analytics.sendEvent.calledWithMatch('',
            sinon.match.has('time').and(sinon.match.has('abc')).and(sinon.match.has('count'))),
            '"time", "abc" and "count" attributes should be sent');
    });

    test('Should not allow to override "time" attribute by given extra attributes', function () {
        var moduleB = this.context.require('jira/editor/fake-module-b');

        var duration = moduleB.embedExtraAttributes({count: 10, time: 'no-way'});
        ok(Analytics.sendEvent.calledWithMatch('',
            sinon.match.has('time', duration).and(sinon.match.has('count'))),
            '"time" attribute should not be overridden and "count" attribute should be sent');
    });


    module('Analytics startMeasure', {
        setup: function () {
            this.sandbox = sinon.sandbox.create();
            var clock = this.clock = sinon.useFakeTimers();

            var moduleA = {
                stage1: function () {
                    return Analytics.startMeasure();
                },
                computeSomeNPProblem: function () {
                    var measurement = Analytics.startMeasure();
                    // hard work here
                    clock.tick(1000);
                    return measurement.measure(MARK1);
                }
            };
            var moduleB = {
                stage2: function (measurement) {
                    clock.tick(1000);
                    return measurement.measure(MARK1);
                }
            };

            this.sandbox.stub(Analytics, "sendEvent");
            this.context = AJS.test.mockableModuleContext();
            this.context.mock('jira/editor/fake-module-a', moduleA);
            this.context.mock('jira/editor/fake-module-b', moduleB);
        },

        teardown: function () {
            this.clock.restore();
            this.sandbox.restore();
        }
    });

    test('Should measure time between A and B points', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');

        var duration = moduleA.computeSomeNPProblem();
        ok(duration >= 0, 'measured time should be greater than zero');
    });

    test('Should be able to measure twice based on the one starting point', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');

        var measurement = moduleA.stage1();
        this.clock.tick(1000);
        var duration = measurement.measure();
        ok(duration >= 0, 'first measure should be done');

        this.clock.tick(1000);
        var duration2 = measurement.measure();
        ok(duration2 >= duration, 'second measure should be done');
    });

    test('Should send an analytic event with the given name', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');
        var eventName = 'stage1';

        var measurement = moduleA.stage1();
        measurement.measure(eventName);

        ok(Analytics.sendEvent.calledOnce, '"sendEvent" method should be called exactly once');
        var eventKey = ANALYTIC_PERF_KEY + eventName;
        ok(Analytics.sendEvent.calledWith(eventKey), 'event key should be equal ' + eventKey);
    });

    test('Should be able to do cross-module measure', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');
        var moduleB = this.context.require('jira/editor/fake-module-b');
        var eventName1 = 'stage1';

        var measurement = moduleA.stage1();
        this.clock.tick(1000);
        var duration = measurement.measure(eventName1);

        ok(duration >= 0, 'measure inside first module should be done');
        ok(Analytics.sendEvent.calledOnce, '"sendEvent" method should be called exactly once');
        var eventKey = ANALYTIC_PERF_KEY + eventName1;
        ok(Analytics.sendEvent.calledWith(eventKey), 'event key should be equal ' + eventKey);

        var duration2 = moduleB.stage2(measurement);

        ok(duration2 >= 0, 'measure inside second module should be done');
        ok(Analytics.sendEvent.calledTwice, '"sendEvent" method should be called exactly twice');
        var eventKey = ANALYTIC_PERF_KEY + MARK1;
        ok(Analytics.sendEvent.calledWith(eventKey), 'event key should be equal ' + eventKey);
    });

    test('Should send an analytic event with default attribute', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');
        var measurement = moduleA.stage1();

        measurement.measure();

        ok(Analytics.sendEvent.calledWithMatch('', sinon.match.has('time')), '"time" attribute should be sent');
    });

    test('Should send an analytic event with extra attributes', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');
        var measurement = moduleA.stage1();

        measurement.measure('', {abc: 123, count: 10});

        ok(Analytics.sendEvent.calledWithMatch('',
            sinon.match.has('time').and(sinon.match.has('abc')).and(sinon.match.has('count'))),
            '"time", "abc" and "count" attributes should be sent');
    });

    test('Should not allow to override "time" attribute by given extra attributes', function () {
        var moduleA = this.context.require('jira/editor/fake-module-a');
        var measurement = moduleA.stage1();

        var duration = measurement.measure('', {count: 10, time: 'no-way'});

        ok(Analytics.sendEvent.calledWithMatch('',
            sinon.match.has('time', duration).and(sinon.match.has('count'))),
            '"time" attribute should not be overridden and "count" attribute should be sent');
    });
});