define("jira/editor/converter/util/rte-strings", [
    'exports',
    'underscore'
], function(
    exports,
    _
) {
    /**
     * Checks whether string consist of whitespace characters only, according to HTML 4 standard.
     * @param str the given string
     * @returns {boolean} true if strings consist of whitespace characters only
     *
     * http://www.w3.org/TR/html401/struct/text.html#whitespace
     *
     * Please note, javascript has a broader definition of whitespace characters.
     * These whitespaces characters will not be market as whitespace by the function on purpose.
     * https://msdn.microsoft.com/en-us/library/2yfce773(v=vs.94).aspx
     */
    exports.isWhitespace = function (str) {
        if (_.isEmpty(str)) {
            return false;
        }
        var whitespaceMap = {
            '\t' : 1,
            '\n' : 1,
            '\u0020' : 1, // whitespace
            '\u0009' : 1,
            '\u000A' : 1,
            '\u000D' : 1,
            '\u000C' : 1,
            '\u200B' : 1
        };

        var onlyWhitespaces = true;
        _.each(str, function(ch) {
            onlyWhitespaces = onlyWhitespaces && (ch in whitespaceMap);
        });

        return onlyWhitespaces;
    };

    /**
     * Trims whitespace characters from the string beginning and end
     * @param str string to trim
     * @returns {string} string after without whitespace characters before and after text
     */
    exports.trim = function (str) {
        if (_.isEmpty(str)) {
            return ''
        }
        return this.trimLeading(this.trimTrailing(str));
    };

    /**
     * Trims whitespaces from the string end
     * @param str string to trim
     * @returns {string}
     */
    exports.trimTrailing = function (str) {
        if (_.isEmpty(str)) {
            return ''
        }
        var i;
        for (i = str.length - 1; i >= 0; i--) {
            if (!this.isWhitespace(str[i])) {
                break;
            }
        }
        return str.substring(0, i + 1);
    };

    /**
     * Trims whitespaces from the string start
     * @param str string to trim
     * @returns {string}
     */
    exports.trimLeading = function (str) {
        if (_.isEmpty(str)) {
            return ''
        }
        var i;
        for (i = 0; i < str.length; i++) {
            if (!this.isWhitespace(str[i])) {
                break;
            }
        }
        return str.substring(i, str.length);
    };

    /**
     * Convert given string into DOM tree
     * @param html
     * @returns {HTMLElement}
     */
    exports.parseHtml = function (html) {
        var parser = new DOMParser();
        var dom = parser.parseFromString(html, 'text/html').body;
        return dom;
    };
});