/* eslint-disable max-len */
AJS.test.require(['com.atlassian.jira.plugins.jira-editor-plugin:converter'], function () {
    var Strings = require('jira/editor/converter/util/strings');
    var FeatureManager = require('jira/featureflags/feature-manager');
    var HtmlConverter = require('jira/editor/converter');

    var commonModule = {
        setup: function (assert) {
            this.sandbox = sinon.sandbox.create();

            this.sandbox.stub(Strings, "getBaseUrl", function () {
                return document.location.origin + AJS.contextPath();
            });

            this.featureManagerStub = this.sandbox.stub(FeatureManager, 'isFeatureEnabled');
            this.htmlConverter = HtmlConverter;

            this.assert = assert;
        },

        teardown: function () {
            this.sandbox.restore();
        },

        assertConversion: function (html, markup, testName) {
            var done = this.assert.async();

            this.htmlConverter.convert(html)
                .then(function (result) {
                    equal(result, markup, testName);
                    done();
                })
                .fail(function (e) {
                    throw e;
                });
        }
    };


    module('HtmlConverter', commonModule);

    test('Test simple headings', function () {
        this.assertConversion('<h1>Biggest heading</h1>', 'h1. Biggest heading');
        this.assertConversion('<h2>Bigger heading</h2>', 'h2. Bigger heading');
        this.assertConversion('<h3>Big heading</h3>', 'h3. Big heading');
        this.assertConversion('<h4>Normal heading</h4>', 'h4. Normal heading');
        this.assertConversion('<h5>Small heading</h5>', 'h5. Small heading');
        this.assertConversion('<h6>Smallest heading</h6>', 'h6. Smallest heading');
    });

    test('Test simple text effects', function () {
        this.assertConversion('<strong>Some bold</strong>', '*Some bold*', '<strong>');
        this.assertConversion('<b>Some bold</b>', '*Some bold*', '<b>');
        this.assertConversion('<em>Some emphasis 1</em>', '_Some emphasis 1_', '<em>');
        this.assertConversion('<i>Some emphasis 2</i>', '_Some emphasis 2_', '<i>');
        this.assertConversion('<tt>Some monospaced</tt>', '{{Some monospaced}}', '<tt>');
        this.assertConversion('<samp>Some other monospaced</samp>', '{{Some other monospaced}}', '<samp>');
        this.assertConversion('<code>Some other monospaced</code>', '{{Some other monospaced}}', '<code>');
        this.assertConversion('<ins>underline text</ins>', '+underline text+', '<ins>');
        this.assertConversion('<u>underline text</u>', '+underline text+', '<u>');
    });

    test('Test simple text effects 2', function () {
        this.assertConversion('<sup>upper line</sup>', '^upper line^', '<sup>');
        this.assertConversion('<sub>lower line</sub>', '~lower line~', '<sub>');
        this.assertConversion('<del>empty line</del>', '-empty line-', '<del>');
    });

    test('Test advanced text effects 1', function () {
        this.assertConversion('<b>Some bold and <em>emphasis</em></b>', '*Some bold and _emphasis_*');
        this.assertConversion('<b>Some bold and <sup>superscript</sup></b>', '*Some bold and ^superscript^*');
        this.assertConversion('<sup>superscript<b>and some bold</b> inside</sup>', '^superscript*and some bold* inside^');
    });

    test('Test advanced text effects 2', function () {
        this.assertConversion('<p>Some paragraph and <em>emphasis</em></p>', 'Some paragraph and _emphasis_');
        this.assertConversion('<p>Some paragraph and <sub>subscript</sub></p>', 'Some paragraph and ~subscript~');
    });

    test('Test advanced text effects 3', function () {
        this.assertConversion('<b>Some bold and </b>regular<b> and bold again!</b>', '*Some bold and* regular *and bold again!*');
    });

    test('Test advanced text effects 4', function () {
        this.assertConversion('<b><em><ins>test</ins></em></b>', '*_+test+_*');
    });

    test('Test simple links and mentions', function () {
        this.assertConversion('<a href="http://example.com/">example.com</a>', '[example.com|http://example.com/]');
        this.assertConversion('<a href="' + AJS.contextPath() + '/browse/KEY-1000">KEY-1000</a>', 'KEY-1000');
        this.assertConversion('<a href="' + AJS.contextPath() + '/secure/ViewProfile.jspa?name=admin" class="user-hover">admin</a>', '[~admin]');
    });

    test('Test auto complete links markup', function() {
        this.assertConversion('<a class="external-link" href="http://example.com" rel="nofollow" data-mce-href="http://example.com">link title</a>', '[link title|http://example.com/]');
        this.assertConversion('<a class="external-link" href="mailto:mail@example.com" rel="nofollow" data-mce-href="mailto:mail@example.com">mail@example.com<sup><img class="rendericon" src="/jira/images/icons/mail_small.gif" alt="" width="13" height="12" align="absmiddle" border="0" data-mce-src="../images/icons/mail_small.gif"></sup></a>',
            '[mail@example.com|mailto:mail@example.com]');
        this.assertConversion('<a href="#anchor" data-mce-href="#anchor">anchor</a>', '[#anchor]');
    });

    test('Test auto complete links markup with custom anchors', function() {
        this.assertConversion('<a href="#2" data-mce-href="#2">#test1</a>', '[#test1|#2]');
        this.assertConversion('<a href="#test2" data-mce-href="#test2">test1</a>', '[test1|#test2]');
        this.assertConversion('<span class="nobr"><a class="external-link" href="mailto:test@onet.pl" rel="nofollow" data-mce-href="mailto:test@onet.pl">mailto with anchor<sup><img class="rendericon" src="/jira/images/icons/mail_small.gif" alt="" width="13" height="12" align="absmiddle" border="0" data-mce-src="../images/icons/mail_small.gif"></sup></a></span>',
            '[mailto with anchor|mailto:test@onet.pl]');
        this.assertConversion('<span class="nobr"><a href="mailto:test@onet.pl" class="external-link" rel="nofollow" title="Follow link">mailto&nbsp;<b>with</b>&nbsp;anchor<sup><img class="rendericon" src="/jira/images/icons/mail_small.gif" height="12" width="13" align="absmiddle" alt="" border="0"></sup></a></span>',
            '[mailto *with* anchor|mailto:test@onet.pl]');
        this.assertConversion('<span class="nobr"><a href="mailto:test@onet.pl" class="external-link" rel="nofollow" title="Follow link">mailto&nbsp;<img class="emoticon" src="/jira/images/icons/emoticons/thumbs_up.png" height="16" width="16" align="absmiddle" alt="" border="0">&nbsp;anchor<sup><img class="rendericon" src="/jira/images/icons/mail_small.gif" height="12" width="13" align="absmiddle" alt="" border="0"></sup></a></span>',
            '[mailto (y) anchor|mailto:test@onet.pl]');
    });

    test('Test links markup with custom anchors', function() {
        this.assertConversion('<a title="table-bug.mov" href="/jira/browse/undefined" data-filename="table-bug.mov" data-mce-href="/jira/browse/undefined">table-bug.mov<sup><img class="rendericon" src="/jira/images/icons/link_attachment_7.gif" alt="" width="7" height="7" align="absmiddle" border="0" data-mce-src="/jira/images/icons/link_attachment_7.gif"></sup></a>', '[^table-bug.mov]');
        this.assertConversion('<a title="table-bug.mov" href="/jira//projects/HSP/issues/HSP-1?filter=allopenissues/undefined" data-filename="table-bug.mov" data-mce-href="/jira/browse/undefined">table-bug.mov<sup><img class="rendericon" src="/jira/images/icons/link_attachment_7.gif" alt="" width="7" height="7" align="absmiddle" border="0" data-mce-src="/jira/images/icons/link_attachment_7.gif"></sup></a>', '[^table-bug.mov]');
    });

    test('Resolved issue', function () {
        this.assertConversion('<p><a href="' + (document.location.origin+AJS.contextPath()) + '/browse/QRQRRQ-1" title="asddasdsa" class="issue-link" data-issue-key="QRQRRQ-1"><del>QRQRRQ-1</del></a></p>', 'QRQRRQ-1');
    });

    test('Context links', function () {
        var url =  document.location.origin + AJS.contextPath() + '/secure/CreateIssue.jspa?pid=100&issuetype=1';
        this.assertConversion('<a href="' + url +'">'+url+'</a>', '[' + url + ']');
        this.assertConversion('<a href="' + url +'">name</a>', '[name|' + url + ']');
    });

    test('Relative urls', function () {
        var path = "secure/CreateIssue.jspa";
        var url =  document.location.origin + AJS.contextPath() + "/" + path;

        this.assertConversion('<a href="..' + AJS.contextPath() + '/' + path +'">'+url+'</a>', '[' + url + ']');
        this.assertConversion('<a href="..' + AJS.contextPath() + '/' + path +'">name</a>', '[name|' + url + ']');
    });

    test('Test emoticons', function () {
        this.assertConversion('<p><img class="emoticon" src="/jira/images/icons/emoticons/smile.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/sad.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/tongue.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/biggrin.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/wink.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/thumbs_up.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/thumbs_down.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/information.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/check.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/error.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/warning.png" height="16" width="16" align="absmiddle" alt="" border="0"></p>', ':) :( :P :D ;) (y) (n) (i) (/) (x) (!)');

        this.assertConversion('<p><img class="emoticon" src="/jira/images/icons/emoticons/add.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/forbidden.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/help_16.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/lightbulb_on.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/lightbulb.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/star_yellow.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/star_red.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/star_green.png" height="16" width="16" align="absmiddle" alt="" border="0"> <img class="emoticon" src="/jira/images/icons/emoticons/star_blue.png" height="16" width="16" align="absmiddle" alt="" border="0"></p>', '(+) (-) (?) (on) (off) (*) (*r) (*g) (*b)');

        this.assertConversion('<p><img class="emoticon" src="/jira/images/icons/emoticons/flag.png" height="16" width="16" align="absmiddle" alt="" border="0"> flag<br></p>', '(flag) flag');
        this.assertConversion('<p><img class="emoticon" src="/jira/images/icons/emoticons/flag_grey.png" height="16" width="16" align="absmiddle" alt="" border="0"> flag<br></p>', '(flagoff) flag');
    });

    test('Test orphan emoticons', function () {
        this.assertConversion('<p><img class="emoticon" src="' + AJS.contextPath() + '/images/icons/emoticons/some_flag.png" height="16" width="16" align="absmiddle" alt="" border="0"> flag<br></p>', '!' + document.location.origin + AJS.contextPath()+'/images/icons/emoticons/some_flag.png! flag');
    });

    test('External image link', function () {
        this.assertConversion('<img src="http://www.atlassian.com/logo.png" />', '!http://www.atlassian.com/logo.png!');
    });

    test('Resized image', function () {
        this.assertConversion('<img src="http://www.atlassian.com/logo.png" width="12" height="34" />', '!http://www.atlassian.com/logo.png|width=12,height=34!');
    });

    test('Temporary attachment', function () {
        this.assertConversion('<img src="/jira/secure/temporaryattachment/698314513339d71eca0f10435fd428e86a48d85a/temp9136841870699312033_image-2017-05-24-12-59-53-459.png" width="416" height="85" border="0">', '!image-2017-05-24-12-59-53-459.png|width=416,height=85!');
    });

    test('Test no formatting', function () {
        this.assertConversion('<div class="code panel"><div class="codeContent panelContent"><pre class="code-java">&lt;b&gt;bold&lt;i&gt;italic&lt;i&gt;&lt;/b&gt;\nnew line</pre></div></div>', '{code:java}\n<b>bold<i>italic<i></b>\nnew line{code}');
    });

    test('Test code new lines', function () {
        this.assertConversion('<div class="code panel" style="border-width: 1px;" data-mce-style="border-width: 1px;"><div class="codeContent panelContent"><pre class="code-java"><span class="code-keyword">this</span> is code<br/><br/>new</pre></div></div>', '{code:java}\nthis is code\n\nnew{code}');
        this.assertConversion('<div class="code panel" style="border-width: 1px;" data-mce-style="border-width: 1px;"><div class="codeContent panelContent"><pre class="code-java">test <b>bold</b>\n<u>new</u> line</pre></div></div>', '{code:java}\ntest bold\nnew line{code}');
        this.assertConversion('<div class="code panel" style="border-width: 1px;" data-mce-style="border-width: 1px;"><div class="codeContent panelContent"><pre class="code-java">test <span><span>test<br/>test2</span>span\n</span><span>after</span></pre></div></div>', '{code:java}\ntest test\ntest2span\nafter{code}');
    });

    test('Test code new lines (alternate markup)', function () {
        this.assertConversion('<pre class="code panel" data-language="code-java"><span class="code-keyword">this</span> is code<br/><br/>new</pre>', '{code:java}\nthis is code\n\nnew{code}');
        this.assertConversion('<pre class="code panel" data-language="code-java">test <b>bold</b>\n<u>new</u> line</pre>', '{code:java}\ntest bold\nnew line{code}');
        this.assertConversion('<pre class="code panel" data-language="code-java">test <span><span>test<br/>test2</span>span\n</span><span>after</span></pre>', '{code:java}\ntest test\ntest2span\nafter{code}');
        this.assertConversion('<pre class="code panel" data-language="code-java"><panel-title>subject</panel-title>test <span><span>test<br/>test2</span>span\n</span><span>after</span></pre>', '{code:java|title=subject}\ntest test\ntest2span\nafter{code}');
    });

    test('Test simple list bullets', function () {
        this.assertConversion('<ul><li>some</li><li>bullet</li><li>points</li></ul>', '* some\n * bullet\n * points');
    });

    test('Test tricky list bullet list with nest-level properly skips letters', function () {
        this.assertConversion('<ul><li>T<ul><li>T<ul><li>T<ul><li><ul><li>T<ul><li><ul><li>T<ul><li><ul><li>T<ul><li>T</li></ul></li><li>T</li></ul></li><li>T</li><li>T</li></ul></li><li>T</li></ul></li><li>T</li></ul></li><li>T</li></ul></li><li>T</li></ul></li><li>T</li></ul></li><li>T</li></ul></li><li>T</li><li>T<ul><li>T<ul><li>T<ul><li>T<ul><li>T<ul><li>T<ul><li>T<ul><li>T</li><li>T<ul><li>T<ul><li>T</li></ul></li><li>T</li></ul></li></ul></li><li>T</li></ul></li></ul></li><li>T</li></ul></li></ul></li><li>T</li></ul></li><li>T</li></ul></li><li>T</li></ul>', '* T\n ** T\n *** T\n **** \n ***** T\n ****** \n ******* T\n ******** \n ********* T\n ********** T\n ********* T\n ******** T\n ******** T\n ******* T\n ****** T\n ***** T\n **** T\n *** T\n ** T\n * T\n * T\n ** T\n *** T\n **** T\n ***** T\n ****** T\n ******* T\n ******** T\n ******** T\n ********* T\n ********** T\n ********* T\n ******* T\n ***** T\n *** T\n ** T\n * T');
    });

    test('Test simple ordered list', function () {
        this.assertConversion('<ol><li>some</li><li>ordered</li><li>lists</li></ol>', '# some\n # ordered\n # lists');
    });

    test('Test tricky list bullet ordered list with nest-level properly skips letters', function () {
        this.assertConversion('<ol><li>T<ol><li>T<ol><li>T<ol><li><ol><li>T<ol><li><ol><li>T<ol><li><ol><li>T<ol><li>T</li></ol></li><li>T</li></ol></li><li>T</li><li>T</li></ol></li><li>T</li></ol></li><li>T</li></ol></li><li>T</li></ol></li><li>T</li></ol></li><li>T</li></ol></li><li>T</li></ol></li><li>T</li><li>T<ol><li>T<ol><li>T<ol><li>T<ol><li>T<ol><li>T<ol><li>T<ol><li>T</li><li>T<ol><li>T<ol><li>T</li></ol></li><li>T</li></ol></li></ol></li><li>T</li></ol></li></ol></li><li>T</li></ol></li></ol></li><li>T</li></ol></li><li>T</li></ol></li><li>T</li></ol>', '# T\n ## T\n ### T\n #### \n ##### T\n ###### \n ####### T\n ######## \n ######### T\n ########## T\n ######### T\n ######## T\n ######## T\n ####### T\n ###### T\n ##### T\n #### T\n ### T\n ## T\n # T\n # T\n ## T\n ### T\n #### T\n ##### T\n ###### T\n ####### T\n ######## T\n ######## T\n ######### T\n ########## T\n ######### T\n ####### T\n ##### T\n ### T\n ## T\n # T');
    });

    test('Test alternate unordered list', function () {
        this.assertConversion('<ul class="alternate" type="square"><li>some</li><li>bullet</li><li>points</li></ul>', '- some\n - bullet\n - points');
    });

    test('Test simple table with header', function () {
        this.assertConversion('<table class="confluenceTable"><tbody><tr><th class="confluenceTh">heading 1</th><th class="confluenceTh">heading 2</th><th class="confluenceTh">heading 3</th></tr><tr><td class="confluenceTd">col A1</td><td class="confluenceTd">col A2</td><td class="confluenceTd">col A3</td></tr><tr><td class="confluenceTd">col B1</td><td class="confluenceTd">col B2</td><td class="confluenceTd">col B3</td></tr></tbody></table>', '||heading 1||heading 2||heading 3||\n|col A1|col A2|col A3|\n|col B1|col B2|col B3|');
    });

    test('Test table with head and ordinary cells in a row', function () {
        this.assertConversion('<table class="confluenceTable"><tbody><tr><th class="confluenceTh"> th </th><th class="confluenceTh"> td </th><td class="confluenceTd"> td </td></tr></tbody></table>', '|| th || td | td |');

        this.assertConversion('<div class="content-inner" style="max-height: none;"><div class="table-wrap"><table class="confluenceTable"><tbody><tr><th class="confluenceTh"> th </th><td class="confluenceTd"> td </td><th class="confluenceTh"> th </th></tr></tbody></table></div></div>', '|| th | td || th ||');
    });

    test('Test list in a table', function () {
        this.assertConversion('<table class="confluenceTable"><tbody><tr><td class="confluenceTd"><ul><li>test</li><li>test 2</li></ul></td></tr></tbody></table>', '| * test\n * test 2|');
    });

    test('Test multi column table with a list', function() {
        this.assertConversion('<div class="table-wrap"><table class="confluenceTable mce-item-table" data-mce-selected="1"><tbody><tr><th class="confluenceTh">Heading 1</th><th class="confluenceTh">Heading 2</th><th class="confluenceTh">head</th></tr><tr><td class="confluenceTd">Col A1</td><td class="confluenceTd">Col A2</td><td class="confluenceTd"><br data-mce-bogus="1"></td></tr><tr><td class="confluenceTd"><br data-mce-bogus="1"></td><td class="confluenceTd"><ul><li>list1</li><li>list2</li></ul></td><td class="confluenceTd"><br data-mce-bogus="1"></td></tr></tbody></table></div>', '||Heading 1||Heading 2||head||\n|Col A1|Col A2| |\n| | * list1\n * list2| |');
    });

    test('Test list in a table', function () {
        this.assertConversion('<p>Paragraph:</p>\n<ul>\n\t<li>Some</li>\n\t<li>List</li>\n</ul>\n', 'Paragraph:\n * Some\n * List');
    });

    test('Test table with paragraph', function() {
        this.assertConversion('<table width="100%" cellspacing="0" cellpadding="4" class="mce-item-table"><tbody><tr valign="top"><td width="50%"><p>a</p></td><td width="50%"><p>b</p></td></tr><tr valign="top"><td width="50%"><p>c</p></td><td width="50%"><p>d</p></td></tr></tbody></table>', '|a|b|\n|c|d|');
        this.assertConversion('<p>para before</p><table width="100%" cellspacing="0" cellpadding="4" class="mce-item-table"><tbody><tr valign="top"><td width="50%"><p>a</p></td><td width="50%"><p>b</p></td></tr></tbody></table><p>para with <b>bold</b> after</p>', 'para before\n|a|b|\n\npara with *bold* after');
    });

    test('Test table with multiple new lines', function() {
        this.assertConversion('<table width="100%" cellspacing="0" cellpadding="4" class="mce-item-table"><tbody><tr valign="top"><td width="50%"><p>a</p></td><td width="50%"><p>b</p></td></tr><tr valign="top"><td width="50%"><p>c</p></td><td width="50%"><p>d</p><p>d2</p><p>d3</p></td></tr></tbody></table>',
            '|a|b|\n|c|d\nd2\nd3|');
        this.assertConversion('<table width="100%" cellspacing="0" cellpadding="4" class="mce-item-table"><tbody><tr valign="top"><td width="50%"><p>a</p></td><td width="50%"><p>b</p></td></tr><tr valign="top"><td width="50%"><p>c</p></td><td width="50%">d<br/>d2</td></tr></tbody></table>',
            '|a|b|\n|c|d\nd2|');
        this.assertConversion('<table width="100%" cellspacing="0" cellpadding="4" class="mce-item-table"><tbody><tr valign="top"><td width="50%"><p>a</p></td><td width="50%"><p>b</p></td></tr><tr valign="top"><td width="50%"><p>c</p></td><td width="50%">d<br/><br/>d2</td></tr></tbody></table>',
            '|a|b|\n|c|d\n'+decodeURI('%C2%A0')+'\nd2|');
    });

    test('Test table after table', function() {
        this.assertConversion(
            '<table><tbody><tr><th>H1</th><th>H2</th></tr><tr><td>A1</td><td>A2</td></tr></tbody></table>' +
            '<table><tbody><tr><th>H1</th><th>H2</th></tr><tr><td>A1</td><td>A2</td></tr></tbody></table>',

            '||H1||H2||\n' +
            '|A1|A2|\n' +
            '\n' +
            '||H1||H2||\n' +
            '|A1|A2|',

            'Test table after table');
    });

    test('Test table after table with surrounding markup', function() {
        this.assertConversion(
            '<p>OLOLO</p>' +
            '<table><tbody><tr><th>H1</th><th>H2</th></tr><tr><td>A1</td><td>A2</td></tr></tbody></table>' +
            '<table><tbody><tr><th>H1</th><th>H2</th></tr><tr><td>A1</td><td>A2</td></tr></tbody></table>' +
            '<p>OLOLO</p>',

            'OLOLO\n' +
            '||H1||H2||\n' +
            '|A1|A2|\n' +
            '\n' +
            '||H1||H2||\n' +
            '|A1|A2|\n' +
            '\n' +
            'OLOLO',

            'Test table after table with surrounding markup');
    });

    test('Test footer macro', function () {
        // in jira func tests this one will give different results
        this.assertConversion('<div class="footer-macro">footer</div>',
            typeof AJS.namespace("RefPlugin.Macros.Footer.convert") !== "function" ? 'footer' : '{footer}footer{footer}');
    });

    test('Test external macro', function () {
        this.assertConversion('<pre class="external-macro" data-original-text="{macro}" data-command="macro">&lt;b&gt;</pre>', '{macro}<b>{macro}');
        this.assertConversion('<ul><li><pre class="external-macro" data-original-text="{macro}" data-command="macro" data-new-line-before="true">&lt;b&gt;</pre></li></ul>', '* \n{macro}<b>{macro}');
        // this is a bug/problem in NewLines, but fixing it was out of scope
        this.assertConversion('<ul><li><pre class="external-macro" data-original-text="{macro}" data-command="macro" data-new-line-after="true">&lt;b&gt;</pre></li></ul>', '* \n{macro}<b>{macro}');
    });

    test('Test image placeholder conversion', function () {
        this.assertConversion('<img imagetext="this is example markup" />', '!this is example markup!');
    });

    module('HtmlConverter Newline', commonModule);

    test('Test simple newline', function () {
        this.assertConversion('<h1><a name="Simpleheadingandnothingmore."></a>Simple heading and nothing more.</h1>', 'h1. Simple heading and nothing more.');

    });

    test('Test simple newline', function () {
        this.assertConversion('<h1><a name="Simpleheading"></a>Simple heading</h1>\n<p>Simple text</p>', 'h1. Simple heading\n\nSimple text');
    });

    test('Test newline before ordered list', function () {
        this.assertConversion('<h1><a name="Heading."></a>Heading.</h1>\n<ol>\n\t<li>some</li>\n\t<li>ordered</li>\n\t<li>lists</li>\n</ol>', 'h1. Heading.\n # some\n # ordered\n # lists');
    });

    test('Test newline before ordered list wo newline', function () {
        this.assertConversion('<h1><a name="Heading."></a>Heading.</h1>\n<ol><li>some</li><li>ordered</li><li>lists</li></ol>', 'h1. Heading.\n # some\n # ordered\n # lists');
    });

    test('Test newline before ordered list without newline', function () {
        this.assertConversion('<h1><a name="Heading."></a>Heading.</h1>\n<ol><li>some</li><li>ordered</li><li>lists</li></ol>', 'h1. Heading.\n # some\n # ordered\n # lists');
    });

    test('Test newline in list item keeps it in a single bullet', function () {
        this.assertConversion('<ul>\n\t<li>one</li>\n\t<li>two</li>\n\t<li>three<br/>\ntext<br/>\nte<br/>\nxt</li>\n</ul>', '* one\n * two\n * three\ntext\nte\nxt');
    });

    test('Test table wiki markup', function () {
        this.assertConversion('||Heading 1||Heading 2||<br>||Col A2|', '||Heading 1||Heading 2||\n||Col A2|');
    });

    test('Test table wiki markup in paragraph', function () {
        this.assertConversion('<p><br>||Heading 1||Heading 2||<br>||Col A2|<br></p>', '||Heading 1||Heading 2||\n||Col A2|');
    });

    test('Test table wiki markup in paragraph', function () {
        this.assertConversion("<p>Test paragraph:</p>\n<ul>\n\t<li>option 1</li>\n\t<li>option 2</li>\n\t<li>option 3</li>\n\t<li>option 4</li>\n</ul>\n\n\n<p> Closing paragraph</p>", "Test paragraph:\n * option 1\n * option 2\n * option 3\n * option 4\n\n Closing paragraph");
    });

    test('Test table cell with block elements', function() {
        this.assertConversion('<table><tr><td class="confluenceTd">text</td><td class="confluenceTd"><pre>pre text</pre></td></tr></table>', '|text|pre text|');
        this.assertConversion('<table><tr><td class="confluenceTd">text</td><td class="confluenceTd"><div>div text</div></td></tr></table>', '|text|div text|');
        this.assertConversion('<table><tr><td class="confluenceTd">text</td><td class="confluenceTd"><div><p>para text</p></div><div><p>para text2</p></div></td></tr></table>', '|text|para text\npara text2|');
        this.assertConversion('<table><thead><tr><th class="confluenceTd"><div>head text</div></th></tr></thead></table>', '||head text||');
    });

    test('2 paragraphs', function () {
        this.assertConversion("<p>paragraph one</p>\n\n<p> paragraph two</p>", "paragraph one\n\n paragraph two");
    });

    test('newline in brs in p', function() {
        this.assertConversion("<p>First paragraph</p>\n\n<p> Paragraph with break<br/>\n after br</p>\n\n<p> last paragraph</p>", "First paragraph\n\n Paragraph with break\n after br\n\n last paragraph");
    });

    test('newline in a paragraph', function() {
        this.assertConversion("<p>header</p><p>\n\n\ncenter\n\n\n</p><p>footer</p>", "header\n\ncenter\n\nfooter");
        // checks the same html but with no whitespaces. Results should match
        this.assertConversion("<p>header</p><p>center</p><p>footer</p>", "header\n\ncenter\n\nfooter");
    });

    test('horizontal line', function() {
        this.assertConversion("<p>header</p><hr/><p>footer</p>", "header\n----\nfooter");
        this.assertConversion("<p>test</p><hr><p><br>test2</p>", "test\n----\n\ntest2");
    });

    test('error on image render', function() {
        this.assertConversion('<span class="error">Unable to render embedded object: File (image-2016-02-19-20-02-33-318.png) not found.</span>', '!image-2016-02-19-20-02-33-318.png!');
        this.assertConversion("<span class=\"error\">No usable issue stored in the context, unable to resolve filename 'voting.png'</span>", '!voting.png!');
    });

    test('plain image markup', function() {
        this.assertConversion('<p>image: <img src="../secure/attachment/10000/10000_test.png" align="left" border="0" /></p>', 'image: !test.png|align=left!');
        this.assertConversion('<p>image: <img src="../secure/attachment/10000/10000_test.png" align="absmiddle" border="0" /></p>', 'image: !test.png|align=absmiddle!');
        this.assertConversion('<p>image: <span class="image-wrap"><img src="../secure/attachment/10000/10000_test.png" vspace="4" border="0" /></span></p>', 'image: !test.png|vspace=4!');
        this.assertConversion('<p><span class="image-wrap"><img style="border: 0px solid black;" src="https://dujrsrsgsd3nh.cloudfront.net/img/emoticons/jira-1431522426.png" alt="" data-mce-src="https://dujrsrsgsd3nh.cloudfront.net/img/emoticons/jira-1431522426.png" data-mce-style="border: 0px solid black;"></span>', '!https://dujrsrsgsd3nh.cloudfront.net/img/emoticons/jira-1431522426.png!');
    });

    test('resized image markup', function() {
        this.assertConversion('<p>image: <span class="image-wrap"><img src="../secure/attachment/10000/10000_test.png" border="0" width="13" height="31"/></span></p>', 'image: !test.png|width=13,height=31!');
    });

    test('thumbnail image markup', function() {
        this.assertConversion('<p>image: <a id="10000_thumb" title="test.png" href="' + AJS.contextPath() + '/secure/attachment/10000/10000_test.png" data-mce-href="../secure/attachment/10000/10000_test.png"><img src="http://localhost:2990' + AJS.contextPath() + '/secure/thumbnail/10000/_thumb_10000.png" alt="" data-mce-src="../secure/thumbnail/10000/_thumb_10000.png"></a><br data-mce-bogus="1"></p>', 'image: !test.png|thumbnail!');
        this.assertConversion('<p>image: <span class="image-wrap" style=""><a id="10000_thumb" href="' + AJS.contextPath() + '/secure/attachment/10000/10000_test.png" title="test.png" file-preview-type="image" file-preview-id="10000" file-preview-title="test.png" resolved=""><img src="http://localhost:2990/' + AJS.contextPath() + '/secure/thumbnail/10000/_thumb_10000.png" style="border: 0px solid black"></a></span></a>', 'image: !test.png|thumbnail!')
    });

    test('rested relative image src', function() {
        this.sandbox.restore();
        this.sandbox.stub(Strings, "getBaseUrl", function() {
            return document.location.origin + AJS.contextPath() + "/browse/";
        });

        this.assertConversion('<p>image: <a id="10000_thumb" title="test.png" href="../secure/attachment/10000/10000_test.png" data-mce-href="../secure/attachment/10000/10000_test.png"><img data-mce-selected="1" src="http://localhost:2990' + AJS.contextPath() + '/secure/thumbnail/10000/_thumb_10000.png" alt="" data-mce-src="../secure/thumbnail/10000/_thumb_10000.png" id="123" border="0"></a></p>', 'image: !test.png|id=123,thumbnail!');
    });

    test('Panels markup(panel, code, noformat)', function() {
        this.assertConversion('<div class="panel" style="border-bottom-width: 3px;background-color: #FF0000;border-color: #FACE8D;border-style: dashed;"><div class="panelHeader" style="border-bottom-width: 3px;border-bottom-style: dashed;border-bottom-color: #FACE8D;background-color: #00FFFF;"><b>My title</b></div><div class="panelContent" style="background-color: #FF0000;"><p><b>Some text with a title</b></p></div></div>', '{panel:title=My title|borderStyle=dashed|borderWidth=3px|borderColor=#face8d|titleBGColor=#00ffff|bgColor=#ff0000}\n*Some text with a title*\n{panel}');
        this.assertConversion('<div class="code panel" style="background-color: #FF0000;border-color: #FACE8D;border-style: dashed;border-bottom-width: 3px;"><div class="codeHeader panelHeader" style="border-bottom-width: 3px;border-bottom-color: #FACE8D;background-color: #00FFFF;"><b>OOM.java</b></div><div class="codeContent panelContent" style="background-color: #FF0000; border-bottom-width: 3px;"><pre class="code-java"><span class="code-keyword">throw</span> OOM();</pre></div></div>', '{code:java|title=OOM.java|borderWidth=3px|borderColor=#face8d|titleBGColor=#00ffff|bgColor=#ff0000}\nthrow OOM();{code}');
        this.assertConversion('<div class="preformatted panel" style="background-color: #FF0000;border-color: #FACE8D;border-style: dashed;"><div class="preformattedHeader panelHeader" style="border-bottom-width: 3px;border-bottom-style: dashed;border-bottom-color: #FACE8D;background-color: #00FFFF;"><b>our title</b></div><div class="preformattedContent panelContent" style="background-color: #FF0000;"><pre>not formatted text<br>well yeah</pre></div></div>', '{noformat:title=our title|borderStyle=dashed|borderWidth=3px|borderColor=#face8d|titleBGColor=#00ffff|bgColor=#ff0000}\nnot formatted text\nwell yeah{noformat}');
    });

    test('Panels markup(code, noformat) (new markup)', function() {
        this.assertConversion('<div class="panel" style="background-color: #deadbf; border-width: 1px;" data-mce-style="background-color: #deadbf; border-width: 1px;"><panel-title style="border-bottom-width: 1px; background-color: #deadbf;" data-mce-style="border-bottom-width: 1px; background-color: #deadbf;">subject</panel-title><p>content<br></p><p>new para</p></div>',
            '{panel:title=subject|titleBGColor=#deadbf|bgColor=#deadbf}\ncontent\n\n\nnew para\n{panel}');
        this.assertConversion('<pre class="code panel" style="background-color: #ff0000;border-bottom-width: 10px;" data-language="code-java" data-mce-style="background-color: #ff0000;border-bottom-width: 10px;"><panel-title style="border-bottom-width: 10px; border-bottom-color: #face8d; background-color: #00ffff;" data-mce-style="border-bottom-width: 10px; border-bottom-color: #face8d; background-color: #00ffff;">OOM.java3</panel-title><span class="code-keyword">throw</span> OOM();2 </pre>', '{code:java|title=OOM.java3|borderWidth=10px|borderColor=#face8d|titleBGColor=#00ffff|bgColor=#ff0000}\nthrow OOM();2 {code}');
        this.assertConversion('<pre class="noformat panel" style="background-color: #ff00ff;"><panel-title style="border-bottom-width: 4px;background-color: #00FFFF;" >lol</panel-title>kek</pre>', '{noformat:title=lol|borderWidth=4px|titleBGColor=#00ffff|bgColor=#ff00ff}\nkek{noformat}');
    });

    test('Weird markup', function() {
        this.assertConversion('<div>just a text</div>', 'just a text');
        this.assertConversion('<a href="http://www.atlassian.com/test">adjacent</a><a href="http://www.atlassian.com/test2">link</a>', '[adjacent|http://www.atlassian.com/test] [link|http://www.atlassian.com/test2]');
        this.assertConversion('<div class="mod-content"><ul class="item-details" id="peopledetails"><li class="people-details">before<dl><dt>Assignee:</dt><dd><span id="assignee-val" class="view-issue-field editable-field inactive" title="Click to edit">text</span></dd></dl>', '* before\nAssignee:\ntext');
    });

    test('Panel markup', function() {
        this.assertConversion('<div class="panel" style="border-width: 1px;"><div class="panelHeader" style="border-bottom-width: 1px;"><strong>My title</strong></div><div class="panelContent"><p>Some text with a title</p></div></div>', '{panel:title=My title}\nSome text with a title\n{panel}');
        this.assertConversion('<div class="panel" style="border-width: 1px;">\n<div class="panelHeader" style="border-bottom-width: 1px;"><strong>My title</strong></div>\n<div class="panelContent">\n<p>Some text with a title</p>\n</div>\n</div>', '{panel:title=My title}\nSome text with a title\n{panel}');
    });

    test('Fusion issue link', function() {
        this.assertConversion('<p>\n<span class="jira-issue-macro" data-jira-key="PIPE-989">\n<a href="https://sdog.jira-dev.com/browse/PIPE-989" class="jira-issue-macro-key"><img class="icon" src="https://sdog.jira-dev.com/secure/viewavatar?size=xsmall&amp;avatarId=20012&amp;avatarType=issuetype">PIPE-989</a> - <span class="summary">Flaky test it.com.atlassian.plugins.jira.screenshot.webdriver.TestIssuePaste.testPastingMultiTypeContentWithRtfAndImagePastesText</span>\n<span class="aui-lozenge aui-lozenge-subtle aui-lozenge-complete jira-macro-single-issue-export-pdf">New</span></span></p>', 'PIPE-989');
    });

    test('Combined markup', function() {
        this.assertConversion('<p><div class="panel" style="border-width: 1px;" data-mce-style="border-width: 1px;"><div class="panelHeader" style="border-bottom-width: 1px;" data-mce-style="border-bottom-width: 1px;"><strong>title</strong></div><div class="panelContent"><p>content</p></div></div><table class="confluenceTable mce-item-table" data-mce-selected="1"><tbody><tr><td class="confluenceTd">cell1</td><td class="confluenceTd">cell2</td></tr></tbody></table></p>',
            '{panel:title=title}\ncontent\n{panel}\n|cell1|cell2|');
    });

    module('HtmlConverter Special Chars Escaping', commonModule);

    test('List with curly braces', function() {
        this.assertConversion('<ul><li>GET /info (software info)</li><li>GET /request</li><li>GET /request/{id}</li><li>GET/POST /request/{id}/comment</li><li>GET/POST/DELETE /request/{id}/participant</li><li>GET /request/{id}/sla</li><li>GET /request/{id}/status</li><li>GET /servicedesk</li><li>GET /servicedesk/{serviceDeskId}</li><li>GET /servicedesk/{serviceDeskId}/requesttype/</li><li>GET /servicedesk/{serviceDeskId}/requesttype/{requestTypeId}</li><li>GET /servicedesk/{serviceDeskId}/requesttype/{requestTypeId}/field</li></ul>', '* GET /info (software info)\n * GET /request\n * GET /request/\\{id}\n * GET/POST /request/\\{id}/comment\n * GET/POST/DELETE /request/\\{id}/participant\n * GET /request/\\{id}/sla\n * GET /request/\\{id}/status\n * GET /servicedesk\n * GET /servicedesk/\\{serviceDeskId}\n * GET /servicedesk/\\{serviceDeskId}/requesttype/\n * GET /servicedesk/\\{serviceDeskId}/requesttype/\\{requestTypeId}\n * GET /servicedesk/\\{serviceDeskId}/requesttype/\\{requestTypeId}/field');
        this.assertConversion('Let\'s use some {code} and {quote} macros in a single line', 'Let\'s use some \\{code} and \\{quote} macros in a single line');
        this.assertConversion('<div class="code panel"><div class="codeContent panelContent"><pre class="code-java">{}</pre></div></div>', '{code:java}\n{}{code}');
        this.assertConversion('<div class="code panel"><div class="codeContent panelContent"><pre class="code-java">\\{}</pre></div></div>', '{code:java}\n\\{}{code}');
    });

    module('HtmlConverter TinyMCE', commonModule);

    test('color', function() {
        this.assertConversion('<span style="color: rgb(20, 137, 44);">color</span>', '{color:#14892c}color{color}');
        this.assertConversion('<span style="color: rgb(0, 0, 0);">color</span>', '{color:#000000}color{color}');
        this.assertConversion('<span style="color: RED">color</span>', '{color:#FF0000}color{color}');
        this.assertConversion('<span style="color: #00ff00">color</span>', '{color:#00ff00}color{color}');
        this.assertConversion('<span mce-style="color: #00ff00" style="color: rgb(0, 255, 0);">color</span>', '{color:#00ff00}color{color}');
    });

    test('code', function() {
        this.assertConversion('<div class="code panel"><div class="codeContent panelContent"><pre class="code-java">Code macro</pre></div></div>', '{code:java}\nCode macro{code}');
        this.assertConversion('<div class="code panel"><div class="codeContent panelContent"><pre class="code-xml">&lt;xml&gt;xml content&lt;/xml&gt;</pre></div></div>', '{code:xml}\n<xml>xml content</xml>{code}');
    });

    test('code pasted into empty code block', function() {
        this.assertConversion('<div class="code panel">Code macro</div>', '{code}\nCode macro{code}');
        this.assertConversion('<div class="code panel">&lt;xml&gt;xml content&lt;/xml&gt;</div>', '{code}\n<xml>xml content</xml>{code}');
        this.assertConversion('<div class="code panel"><pre class="code-java">Code macro</pre></div>', '{code:java}\nCode macro{code}');
        this.assertConversion('<div class="code panel"><pre class="code-xml">&lt;xml&gt;xml content&lt;/xml&gt;</pre></div>', '{code:xml}\n<xml>xml content</xml>{code}');
        this.assertConversion('<div class="code panel"><div class="codeContent panelContent">Code macro</div></div>', '{code}\nCode macro{code}');
        this.assertConversion('<div class="code panel"><div class="codeContent panelContent">&lt;xml&gt;xml content&lt;/xml&gt;</div></div>', '{code}\n<xml>xml content</xml>{code}');
    });

    test('noformat', function() {
        this.assertConversion('<div class="preformatted panel"><div class="preformattedContent panelContent"><pre>&lt;marquee&gt;WOW MUCH AMAZE&lt;/marquee&gt;</pre></div></div>',
            '{noformat}\n<marquee>WOW MUCH AMAZE</marquee>{noformat}', 'HTML inside {noformat}');

        this.assertConversion('<div class="preformatted panel"><div class="preformattedContent panelContent"><pre>{code:java}Thread.currentThread().join();{code}</pre></div></div>',
            '{noformat}\n{code:java}Thread.currentThread().join();{code}{noformat}', 'Code inside {noformat} is not rendered');
        this.assertConversion('<div class="preformatted panel"><div class="preformattedContent panelContent"><pre>* First\n* Forth\n* Eighth\n* Eighth</pre></div></div>',
            '{noformat}\n* First'+ '\n' +
            '* Forth' + '\n' +
            '* Eighth' + '\n' +
            '* Eighth{noformat}', 'Wiki markup List inside {noformat}');
        this.assertConversion('<div class="preformatted panel"><div class="preformattedContent panelContent"><pre>!thug_life.jpg|thumbnail!</pre></div></div>',
            '{noformat}\n!thug_life.jpg|thumbnail!{noformat}', 'Wiki markup Attachment inside {noformat}');
    });

    test('noformat (new markup)', function() {
        this.assertConversion('<pre class="noformat panel">&lt;marquee&gt;WOW MUCH AMAZE&lt;/marquee&gt;</pre>',
            '{noformat}\n<marquee>WOW MUCH AMAZE</marquee>{noformat}', 'HTML inside {noformat}');

        this.assertConversion('<pre class="noformat panel">{code:java}Thread.currentThread().join();{code}</pre>',
            '{noformat}\n{code:java}Thread.currentThread().join();{code}{noformat}', 'Code inside {noformat} is not rendered');
        this.assertConversion('<pre class="noformat panel">* First\n* Forth\n* Eighth\n* Eighth</pre>',
            '{noformat}\n* First'+ '\n' +
            '* Forth' + '\n' +
            '* Eighth' + '\n' +
            '* Eighth{noformat}', 'Wiki markup List inside {noformat}');
        this.assertConversion('<pre class="noformat panel">!thug_life.jpg|thumbnail!</pre>',
            '{noformat}\n!thug_life.jpg|thumbnail!{noformat}', 'Wiki markup Attachment inside {noformat}');
        this.assertConversion('<pre class="noformat panel"><panel-title>subject</panel-title>content</pre>',
            '{noformat:title=subject}\ncontent{noformat}');
    });

    test('content pasted into empty noformat block', function() {
        this.assertConversion('<div class="preformatted panel"><div class="preformattedContent panelContent"><pre>Preformatted</pre></div>' +
            '<p>First paragraph</p><p>Second paragraph</p></div>', '{noformat}\nPreformatted\nFirst paragraph\nSecond paragraph{noformat}', 'Some text typed outside {noformat} <pre>');
        this.assertConversion('<div class="preformatted panel">!thug_life.jpg|thumbnail!</div>', '{noformat}\n!thug_life.jpg|thumbnail!{noformat}', 'Wiki markup Attachment inside {noformat}');
        this.assertConversion('<div class="preformatted panel"><pre>!thug_life.jpg|thumbnail!</pre></div>', '{noformat}\n!thug_life.jpg|thumbnail!{noformat}', 'Wiki markup Attachment inside {noformat}');
        this.assertConversion('<div class="preformatted panel"><div class="preformattedContent panelContent">!thug_life.jpg|thumbnail!</div></div>', '{noformat}\n!thug_life.jpg|thumbnail!{noformat}', 'Wiki markup Attachment inside {noformat}');
    });

    test('quotes', function() {
        this.assertConversion('<blockquote><p>Quotes are working.</p></blockquote>', '{quote}Quotes are working.\n{quote}');
    });

    test('cite', function() {
        this.assertConversion('<cite><p>Citation is working.</p></cite>', '??Citation is working.\n??');
        this.assertConversion('<p><cite>Citation is working.</cite></p>', '??Citation is working.??');
        this.assertConversion('<p><cite>Citation <b>does</b> <u>allow</u> <em>formatting</em> inside.</cite></p>', '??Citation *does* +allow+ _formatting_ inside.??');
    });

    test('mention links', function() {
        this.assertConversion('<a href="https://jdog.jira-dev.com/secure/ViewProfile.jspa?name=brodriguez" class="user-hover" rel="brodriguez" title="Follow link">Bender Rodriguez [Futurama]</a>', '[~brodriguez]');
        this.assertConversion('<a href="/browse/ViewProfile.jspa?name=brodriguez" class="user-hover" rel="brodriguez" title="Follow link">Bender Rodriguez [Futurama]</a>', '[~brodriguez]');
        this.assertConversion('<a href="ViewProfile.jspa?name=brodriguez" class="user-hover" rel="brodriguez" title="Follow link">Bender Rodriguez [Futurama]</a>', '[~brodriguez]');
        this.assertConversion('<p>– Admin Instrator <a href="https://freezer-instance-458.jira-dev.com/wiki/display/SDSD/Administrator" data-mce-href="../wiki/display/SDSD/Administrator">Administrator</a></p>', '– Admin Instrator [Administrator]');
    });

    test("mention special characters", function () {
        this.assertConversion('<a href="http://localhost:2990/jira/secure/ViewProfile.jspa?name=hacker-_.%21%7E*%27%28+%29hacker" class="user-hover" rel="hacker-_.!~*\'( )hacker" title="Follow link">hacker-_.!~*\'( )hacker</a>', '[~hacker-_.!~*\'( )hacker]');
        this.assertConversion('<a href="http://localhost:2990/jira/secure/ViewProfile.jspa?name=hacker-_.%21%7E*%27%28+%29hacker" class="user-hover" rel="hacker-_.!~*\'( )hacker" title="Follow link">hacker-_.!~*\'( )hacker</a>', '[~hacker-_.!~*\'( )hacker]');
        this.assertConversion('<a href="https://mszerszynski-2016-05-23t12-31-06.jira-dev.com/wiki/display/HP/h%40cker+page">h@cker page</a>', '[h@cker page]');
        this.assertConversion('<a href="https://mszerszynski-2016-05-23t12-31-06.jira-dev.com/wiki/display/HP/hacker-_.%21%7E*%27%28%29hacker">hacker-_.!~*\'()hacker</a>', "[hacker-_.!~*'()hacker]");
        this.assertConversion('<a href="https://mszerszynski-2016-05-23t12-31-06.jira-dev.com/wiki/display/HP/reserved+characters-_.%21%7E*%27%28+%29">reserved characters-_.!~*\'( )</a>', "[reserved characters-_.!~*'( )]");
    });

    test('test content', function() {
        this.assertConversion('<h3>Steps to reproduce:</h3><ol>' +
            '<li><span>Install JIRA Capture 2.7.2.1</span><br></li>' +
            '<li><span>Create a Test Session in a JIRA issue by navigating to the issue, and clicking on More &gt;&gt; Create Test Session.</span><br></li>' +
            '<li><span>Start the test session by clicking on cogwheel icon and select Start</span><br></li>' +
            '<li><span>Login with a different user without administration privileges.</span><br></li>' +
            '<li><span>Go to same Project.</span><br></li>' +
            '<li><span>Go to the Test Session Tab.</span><br></li>' +
            '<li><span>Go to the Test Session.</span><br></li>' +
            '<li><span>Click Join button.</span><br></li></ol>' +
            '<h3>Expected Behaviour:</h3><p>Test session completes successfully.</p>' +
            '<h3>Current Behaviour:</h3><p>XSRF security token missing error: XSRF.jpg</p>' +
            '<p><span class="image-wrap"><img src="' + Strings.getBaseUrl() + '/secure/attachment/124130/124130_bonfire+xsrf.png" style="border: 0px solid black;" alt="" data-mce-src="../secure/attachment/124130/124130_bonfire+xsrf.png" data-mce-style="border: 0px solid black;"></span></p>' +
            '<p>Tested on jira.atlassian.com</p>',

            'h3. Steps to reproduce:\n' +
            ' # Install JIRA Capture 2.7.2.1\n\n' +
            ' # Create a Test Session in a JIRA issue by navigating to the issue, and clicking on More >> Create Test Session.\n\n' +
            ' # Start the test session by clicking on cogwheel icon and select Start\n\n' +
            ' # Login with a different user without administration privileges.\n\n' +
            ' # Go to same Project.\n\n' +
            ' # Go to the Test Session Tab.\n\n' +
            ' # Go to the Test Session.\n\n' +
            ' # Click Join button.\n\n\n' +
            'h3. Expected Behaviour:\n\n' +
            'Test session completes successfully.\n' +
            'h3. Current Behaviour:\n\n' +
            'XSRF security token missing error: XSRF.jpg\n\n' +
            '!bonfire xsrf.png!\n\n' +
            'Tested on jira.atlassian.com');
    });

    test('test lists in table', function () {
        this.assertConversion('<div class="table-wrap"><table class="confluenceTable"><tbody><tr><td class="confluenceTd">with<ul><li>text</li><li>before</li></ul></td></tr></tbody></table></div>', '|with\n * text\n * before|');
        this.assertConversion('<div class="table-wrap"><table class="confluenceTable"><tbody><tr><td class="confluenceTd"><ul><li>without</li><li>text</li><li>before</li></ul></td></tr></tbody></table></div>', '| * without\n * text\n * before|');
        this.assertConversion('<div class="table-wrap"><table class="confluenceTable"><tbody><tr><td class="confluenceTd">&nbsp;<ul><li>dd</li></ul></td></tr></tbody></table></div>', '|\u00A0\n * dd|');
        this.assertConversion('<div class="table-wrap"><table class="confluenceTable"><tbody><tr><td class="confluenceTd"> <ul><li>without</li><li>text</li><li>before</li></ul></td></tr></tbody></table></div>', '|  * without\n * text\n * before|');

        this.assertConversion('<div class="table-wrap"><table class="confluenceTable"><tbody><tr><th class="confluenceTh"><div class="panel" style="border-width: 1px;"><div class="panelHeader" style="border-bottom-width: 1px;"><strong>Panel</strong></div><div class="panelContent"><ul><li>list</li><li>in</li><li>panel</li></ul></div></div></th></tr></tbody></table></div>', '||{panel:title=Panel}\n * list\n * in\n * panel{panel}||');
        this.assertConversion('<div class="table-wrap"><table class="confluenceTable"><tbody><tr><th class="confluenceTh"><ul><li>multiple<ol><li>lists</li></ol></li></ul></th><th class="confluenceTh"><ul><li>in<ul><li>different</li></ul></li></ul></th></tr><tr><td class="confluenceTd"><ol><li>columns</li></ol></td><td class="confluenceTd">and rows<ul><li>whoa</li></ul></td></tr></tbody></table></div>', '|| * multiple\n ## lists|| * in\n ** different||\n| # columns|and rows\n * whoa|');

        this.assertConversion('<div class="table-wrap"><table class="confluenceTable"><tbody><tr><th class="confluenceTh"><span class="image-wrap"><a id="10001_thumb" href="' + AJS.contextPath() + '/secure/attachment/10001/10001_example_small.png" title="example_small.png" data-mce-href="../secure/attachment/10001/10001_example_small.png"><img src="http://localhost:2990' + AJS.contextPath() + '/secure/thumbnail/10001/_thumb_10001.png" style="border: 0px solid black;" data-mce-src="../secure/thumbnail/10001/_thumb_10001.png" data-mce-style="border: 0px solid black;"></a></span><ul class="alternate" type="square"><li>types</li><li>1</li><li>2</li><li>3</li><li>4</li></ul></th></tr></tbody></table></div>', '||!example_small.png|thumbnail!\n - types\n - 1\n - 2\n - 3\n - 4||');
        this.assertConversion('<div class="table-wrap"><table class="confluenceTable"><tbody><tr><td class="confluenceTd"><span class="image-wrap"><a id="10001_thumb" href="' + AJS.contextPath() + '/secure/attachment/10001/10001_example_small.png" title="example_small.png" data-mce-href="../secure/attachment/10001/10001_example_small.png"><img src="http://localhost:2990' + AJS.contextPath() + '/secure/thumbnail/10001/_thumb_10001.png" style="border: 0px solid black;" data-mce-src="../secure/thumbnail/10001/_thumb_10001.png" data-mce-style="border: 0px solid black;"></a></span>&nbsp;<ul><li>&nbsp;sdsadas</li></ul></td></tr></tbody></table></div>', '|!example_small.png|thumbnail!\u00A0\n * \u00A0sdsadas|');

        this.assertConversion('<div class="table-wrap"><table data-mce-selected="1" class="confluenceTable mce-item-table"><tbody><tr><td class="confluenceTd"><p>Col A2</p><ul><li>list item</li></ul></td></tr></tbody></table></div>', '|Col A2\n * list item|');
        this.assertConversion('<div class="table-wrap"><table data-mce-selected="1" class="confluenceTable mce-item-table"><tbody><tr><td class="confluenceTd"><ins>Col A2</ins><ul><li>list item</li></ul></td></tr></tbody></table></div>', '|+Col A2+\n * list item|');
    });

    test('test table markup', function () {
        this.assertConversion('<table><caption>caption</caption><tr><td>content</td></tr></table>', '|caption|\n|content|');
    });

    test('test newline before heading and after quote', function () {
        this.assertConversion('<body><blockquote><p>Blockquote</p></blockquote><h1>heading</h1></body>', '{quote}Blockquote\n{quote}\nh1. heading');
    });

    test('test newline before heading and after list', function () {
        this.assertConversion('<p>List</p><ul>' +
            '<li>Item 1</li>' +
            '<li>Item 2</li>' +
            '<li>Item 3</li>' +
            '</ul>' +
            '<h3>Header</h3>',

            'List\n' +
            ' * Item 1\n' +
            ' * Item 2\n' +
            ' * Item 3\n' +
            '\n' +
            'h3. Header');
    });

    test('test heading inside a table', function () {
        this.assertConversion('<div class="table-wrap"><table class="confluenceTable mce-item-table" data-mce-selected="1"><tbody>' +
            '<tr><th class="confluenceTh">Heading 1</th><th class="confluenceTh">Heading 2</th></tr>' +
            '<tr><td class="confluenceTd"><h1>Col A1</h1></td><td class="confluenceTd"><h2>Col A2</h2></td></tr>' +
            '<tr><td class="confluenceTd"><h4>Col B1</h4></td><td class="confluenceTd"><h3>Col B2</h3></td></tr>' +
            '<tr><td class="confluenceTd"><h5>Col C1</h5></td><td class="confluenceTd"><h6>Col C2</h6></td></tr>' +
            '</tbody></table></div>',

            '||Heading 1||Heading 2||\n' +
            '|h1. Col A1|h2. Col A2|\n' +
            '|h4. Col B1|h3. Col B2|\n' +
            '|h5. Col C1|h6. Col C2|');
    });

    test('Should escape table markup from table content', function () {
        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">heading 1</th>\n        <th class="confluenceTh">heading | 2</th>\n        <th class="confluenceTh">heading 3</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col A1</td>\n        <td class="confluenceTd">col A2</td>\n        <td class="confluenceTd">col A3</td>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col B1</td>\n        <td class="confluenceTd">col B2</td>\n        <td class="confluenceTd">col B3</td>\n    </tr>\n    </tbody>\n</table>',
            '||heading 1||heading \\| 2||heading 3||\n|col A1|col A2|col A3|\n|col B1|col B2|col B3|');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">|heading 1</th>\n        <th class="confluenceTh">heading 2</th>\n        <th class="confluenceTh">heading 3</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col A1</td>\n        <td class="confluenceTd">col A2</td>\n        <td class="confluenceTd">col A3</td>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col B1</td>\n        <td class="confluenceTd">col B2</td>\n        <td class="confluenceTd">col B3</td>\n    </tr>\n    </tbody>\n</table>',
            '||\\|heading 1||heading 2||heading 3||\n|col A1|col A2|col A3|\n|col B1|col B2|col B3|');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">heading 1</th>\n        <th class="confluenceTh">heading 2</th>\n        <th class="confluenceTh">heading 3|</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col A1</td>\n        <td class="confluenceTd">col A2</td>\n        <td class="confluenceTd">col A3</td>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col B1</td>\n        <td class="confluenceTd">col B2</td>\n        <td class="confluenceTd">col B3</td>\n    </tr>\n    </tbody>\n</table>',
            '||heading 1||heading 2||heading 3\\|||\n|col A1|col A2|col A3|\n|col B1|col B2|col B3|');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">heading 1</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">|col A3</td>\n    </tr>\n    </tbody>\n</table>',
            '||heading 1||\n|\\|col A3|');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">heading 1</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col| A3</td>\n    </tr>\n    </tbody>\n</table>',
            '||heading 1||\n|col\\| A3|');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">heading 1</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col A3|</td>\n    </tr>\n    </tbody>\n</table>',
            '||heading 1||\n|col A3\\||');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">heading 1</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">col \n\n|\n\nA3</td>\n    </tr>\n    </tbody>\n</table>',
            '||heading 1||\n|col \n\n\\|\n\nA3|');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">hea|||||||||di|||||||ng 1</th>\n        <th class="confluenceTh">|head||    ing |2|||</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">||co||l| A||||3</td>\n        <td class="confluenceTd">normal text</td>\n    </tr>\n    </tbody>\n</table>',
            '||hea\\|\\|\\|\\|\\|\\|\\|\\|\\|di\\|\\|\\|\\|\\|\\|\\|ng 1||\\|head\\|\\|    ing \\|2\\|\\|\\|||\n|\\|\\|co\\|\\|l\\| A\\|\\|\\|\\|3|normal text|');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">heading 1</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">co\\|l</td>\n    </tr>\n    </tbody>\n</table>',
            '||heading 1||\n|co\\\\|l|');

        this.assertConversion('<table class="confluenceTable">\n    <tbody>\n    <tr>\n        <th class="confluenceTh">heading 1</th>\n    </tr>\n    <tr>\n        <td class="confluenceTd">co\\\\|l</td>\n    </tr>\n    </tbody>\n</table>',
            '||heading 1||\n|co\\\\\\|l|');
    });

    test('test DomParser returning null for document.body', function() {
        this.sandbox.stub(DOMParser.prototype, "parseFromString", function () {
            return {body: null};
        });

        this.assertConversion('', '');

        DOMParser.prototype.parseFromString.restore();
    });

    test('test IE11 conversion of empty content', function() {
        this.assertConversion('', '');
    });

    test('variables are not escaped when the feature is turned on', function() {
        this.featureManagerStub
            .withArgs("jira.renderer.consider.variable.format")
            .returns(true);

        this.assertConversion('<p>text with ${variable.name} inside</p>', 'text with ${variable.name} inside');
        this.assertConversion('<p>text with ${variable1} ${variable2} inside ${variable3}</p>', 'text with ${variable1} ${variable2} inside ${variable3}');
        this.assertConversion('<p>text with {code}${variable1}{code} code</p>', 'text with \\{code}${variable1}\\{code} code');
        this.assertConversion('<p>text with ${malformed.variable1 code</p>', 'text with $\\{malformed.variable1 code');
        this.assertConversion('<p><b>${code}</b></p>', '*${code}*');
        this.assertConversion('<p>${just.variable}</p>', '${just.variable}');
        this.assertConversion('<p>${code}</p>', '${code}');
        this.assertConversion('<p>{code} ${code}</p>', '{code} ${code}');
    });

    test('variables are escaped when the feature is turned off', function() {
        this.featureManagerStub
            .withArgs("jira.renderer.consider.variable.format")
            .returns(false);

        this.assertConversion('<p>text with ${variable.name} inside</p>', 'text with $\\{variable.name} inside');
        this.assertConversion('<p>text with ${variable1} ${variable2} inside ${variable3}</p>', 'text with $\\{variable1} $\\{variable2} inside $\\{variable3}');
        this.assertConversion('<p>text with {code}${variable1}{code} inside</p>', 'text with \\{code}$\\{variable1}\\{code} inside');
        this.assertConversion('<p>text with ${malformed.variable1 code</p>', 'text with $\\{malformed.variable1 code');
        this.assertConversion('<p><b>${code}</b></p>', '*$\\{code}*');
        this.assertConversion('<p>${just.variable}</p>', '$\\{just.variable}');
        this.assertConversion('<p>${code}</p>', '$\\{code}');
        this.assertConversion('<p>{code} ${code}</p>', '\\{code} $\\{code}');
    });
});