define("jira/editor/converter/handlers/span-handler", [
    "jira/editor/converter/handlers/node-handler",
    "jira/editor/converter/handlers/text-handler",
    "jira/editor/converter/handlers/empty-handler",
    "jira/editor/converter/util/strings",
    "jira/editor/converter/util/color-converter",
    "jquery"
], function (
    NodeHandler,
    TextHandler,
    EmptyHandler,
    Strings,
    ColorConverter,
    $
) {
    "use strict";

    var SpanHandler = NodeHandler.extend({
        init: function SpanHandler(node) {
            this.node = node;
            this.$node = $(this.node);
            var href = this.node.getAttribute('href');

            if (this.getStyle(this.node, 'textDecoration') === 'underline') {
                this.decorator = 'underline';
            } else if (this.node.className === 'error') {
                var text = this.$node.text();
                if (Strings.getFilenameFromError(text)) {
                    this.decorator = 'attachment-error';
                }
            } else if (Strings.length(this.getStyle(this.node, 'color')) > 0) {
                var colorRaw = this.getStyle(this.node, 'color').toLowerCase();
                if (Strings.contains(colorRaw, 'rgb')) {
                    this.color = ColorConverter.colorToHex(colorRaw);
                } else if (Strings.contains(colorRaw, '#')) {
                    this.color = colorRaw;
                } else if (colorRaw in colorMap) {
                    this.color = colorMap[colorRaw.toLowerCase()];
                }
                if (this.color !== undefined) {
                    this.decorator = 'color';
                }
            } else if (this.node.className === 'jira-issue-macro' && this.$node.attr('data-jira-key')) {
                this.decorator = "jira-issue-macro";
            }
        },

        before: function () {
            if (this.decorator === 'attachment-error') {
                return TextHandler.createTextHandler('!');
            } else if (this.decorator === 'underline') {
                return TextHandler.createTextHandler('+');
            } else if (this.decorator === 'color') {
                return TextHandler.createPreformattedTextHandler('{color:' + this.color + '}');
            }
            return new EmptyHandler();
        },

        after: function () {
            if (this.decorator === 'attachment-error') {
                return TextHandler.createTextHandler('!');
            } else if (this.decorator === 'underline') {
                return TextHandler.createTextHandler('+');
            } else if (this.decorator === 'color') {
                return TextHandler.createPreformattedTextHandler('{color}');
            }
            return new EmptyHandler();
        },

        children: function () {
            if (this.decorator === 'attachment-error') {
                var filename = Strings.getFilenameFromError(this.$node.text());
                return [TextHandler.createTextHandler(filename)];
            } else if (this.decorator === "jira-issue-macro") {
                return TextHandler.createTextHandler(this.$node.attr('data-jira-key'));
            } else {
                return this._super();
            }
        },

        getStyle: function (node, style) {
            if (style in this.node.style) {
                return this.node.style[style];
            }
            return undefined;
        }
    });

    var colorMap = {
        'white': '#FFFFFF',
        'silver': '#C0C0C0',
        'gray': '#808080',
        'black': '#000000',
        'red': '#FF0000',
        'maroon': '#800000',
        'yellow': '#FFFF00',
        'olive': '#808000',
        'lime': '#00FF00',
        'green': '#008000',
        'aqua': '#00FFFF',
        'teal': '#008080',
        'blue': '#0000FF',
        'navy': '#000080',
        'purple': '#800080',
        'orange': '#FFA500',
        'brown': '#AA4B00',
        'mahogany': '#C04000',
        'pink': '#FFC0CB',
        'yellow orche': '#CF912A',
        'harlequin': '#3FFF00',
        'mauvelous': '#EF98AA',
        'dark red': '#8B0000',
        'saffron': '#F4C430',
        'sepia': '#704214',
        'fuchsia': '#FF00FF',
        'magenta': '#FF00FF',
        'marsala': '#955251',
        'indigo': '#000080'
    };

    return SpanHandler;
});