define("jira/editor/converter/handlers/node-handler", [
    "jira/editor/converter/util/strings",
    "jira/editor/converter/util/rte-strings",
    "jira/editor/converter/new-lines/new-lines",
    "jira/lib/class",
    "underscore",
    "jquery"
], function (
    Strings,
    RteStrings,
    NewLines,
    Class,
    _,
    $
) {
    "use strict";

    var handlerMap = {};
    var handlerStack = [];

    /**
     * @class NodeHandler
     * @extends Class
     *
     * Parent class for all node handlers
     */
    var NodeHandler = Class.extend({
        init: function NodeHandler(node) {
            this.node = node;
        },

        before: function () {
            return new NodeHandler.EmptyHandler();
        },

        processText: function () {
            return '';
        },

        after: function () {
            return new NodeHandler.EmptyHandler();
        },

        children: function () {
            var result = [];
            // converting NodeList to array of handlers
            var childNodes = this.node.childNodes;
            // dry run to filter out all newlines
            var childNodesWithContent = _.filter(childNodes, function (childNode, i) {
                var handler = this.createChildHandler(childNode, i, childNodes.length);
                if (handler.processText() === '') {
                    // hack to detect composite nodes, as they have child nodes but text in them is empty
                    return true;
                }
                if (RteStrings.isWhitespace(handler.processText()) && Strings.contains(handler.processText(), '\n') && !handler.preformatted) {
                    // reject all white-space strings
                    return false;
                }
                return true;
            }, this);

            // after we filter all the empty handlers, we can properly check whether node is the last
            _.each(childNodesWithContent, function (childNode, i) {
                // pre processing
                result.push(this.createBeforeChild(childNode, i, childNodesWithContent.length));
                // Handle child
                result.push(this.createChildHandler(childNode, i, childNodesWithContent.length));
                // post processing
                result.push(this.createAfterChild(childNode, i, childNodesWithContent.length));
            }, this);
            return result;
        },

        createBeforeChild: function (childNode, childIndex, childrenTotal) {
            return new NodeHandler.EmptyHandler();
        },

        createAfterChild: function (childNode, childIndex, childrenTotal) {
            return new NodeHandler.EmptyHandler();
        },

        createChildHandler: function (childNode, childIndex, childrenTotal) {
            return NodeHandler.createHandler(childNode);
        },

        /**
         * Handle new lines creation before/after element
         *
         * Between elements may be exist multiple lines and NewLines markers,
         * all sibling markers are merged to one newline text handler
         *
         * New lines may also be marked as optional - it is used in complex edge cases (like lists in tables)
         *
         * @see jira/editor/converter/new-lines/new-lines
         */
        newLinesBefore: function () {
            var isChildOfBlock = NodeHandler._isBlock(this.node.parentNode) || $(this.node).parents().is("th, td");
            if (NodeHandler._isBlock(this.node) && !($(this.node).is(':first-child') && isChildOfBlock)) {
                return NewLines.single(true);
            } else {
                return NewLines.empty();
            }
        },

        newLinesAfter: function () {
            return NewLines.empty();
        }
    });

    /**
     * @class EmptyHandler
     * @extends NodeHandler
     *
     * A null-object to indicate no processing is required
     */
    NodeHandler.EmptyHandler = NodeHandler.extend({
        init: function EmptyHandler(node) {
            this.node = node;
        },

        before: function () {
            // no-op
            throw undefined;
        },

        after: function () {
            // no-op
            throw undefined;
        },

        processText: function () {
            return '';
        },

        children: function () {
            return [];
        }
    });

    NodeHandler.addHandler = function (selector, handler) {
        selector = selector.toLowerCase();
        handlerMap[selector] = handler;
        handlerStack.push(selector);
    };

    NodeHandler.getHandler = function (node) {
        // text doesn't have tagName but has nodeName
        var nodeName = node.nodeName.toLowerCase();
        return handlerMap[(nodeName + '.' + (node.className || '')).toLowerCase()]
            || handlerMap[nodeName]
            || handlerStack.reduce(function (r, selector) {
                return r || $(node).is(selector) && handlerMap[selector];
            }, null) || NodeHandler;
    };

    NodeHandler.createHandler = function (node) {
        return new (NodeHandler.getHandler(node))(node);
    };

    NodeHandler.isInsideTable = function (node) {
        return $(node).parents('td').length > 0 || $(node).parents('th').length > 0;
    };

    NodeHandler.singleNewLineExceptTable = function (node, optional) {
        if (NodeHandler.isInsideTable(node)) {
            return NewLines.empty();
        }
        return NewLines.single(optional);
    };

    // computing styles is time consuming on FF, especially for table elements
    NodeHandler._isBlock = _.memoize(function (node) {
        if (!node || !node.tagName) {
            return false;
        }

        var testNode = document.createElement(node.tagName);
        document.body.appendChild(testNode);

        try {
            return window.getComputedStyle(testNode).display === "block";
        } finally {
            document.body.removeChild(testNode);
        }
    }, function (node) {
        if (!node || !node.tagName) {
            return "";
        }

        return node.tagName;
    });

    NodeHandler.isEmptyTextNode = function (element) {
        if (!element) {
            return false;
        }
        return element.nodeType === Node.TEXT_NODE && RteStrings.trim(element.nodeValue) === "";

    };

    return NodeHandler;
});