define("jira/editor/converter/handlers/list-handler", [
    "jira/editor/converter/handlers/node-handler",
    "jira/editor/converter/handlers/text-handler",
    "jira/editor/converter/handlers/empty-handler",
    "jira/editor/converter/handlers/list-type-matcher",
    "jquery",
    "jira/editor/converter/new-lines/new-lines",
    "jira/editor/context-detector"
], function (
    NodeHandler,
    TextHandler,
    EmptyHandler,
    ListTypeMatcher,
    $,
    NewLines,
    ContextDetector
) {
    "use strict";

    /**
     * @class ListHandler
     * @extends NodeHandler
     *
     * Defines ListHandler to handle lists of items.
     * Lists can be nested, which defines it's own processing rules.
     *
     * Parent list will handle it's child lists.
     * see also ListItemHandler
     */
    return NodeHandler.extend({
        init: function ListHandler(node, depth) {
            this.node = node;
            this.type = this.node.getAttribute('type'); // optional
            if (typeof depth === 'undefined') {
                this.depth = 0;
            } else {
                this.depth = depth;
            }
            this.bulletDecorator = ListTypeMatcher.getBulletDecorator(node);
        },

        /**
         * Overrides building a child node.
         */
        createChildHandler: function (node) {
            var Handler = NodeHandler.getHandler(node);
            if (node.tagName === 'LI') {
                // other handler processing
                return new Handler(node, this.depth, this.bulletDecorator);
            } else {
                return new Handler(node);
            }
        },

        before: function() {
            return new EmptyHandler();
        },

        newLinesBefore: function () {
            // add newline if inside table and previous node is not empty text
            if (this.depth === 0 && ContextDetector.detectTable(this.node) && this.node.previousSibling && !NodeHandler.isEmptyTextNode(this.node.previousSibling)) {
                return NewLines.single(true);
            }

            return NewLines.empty();
        },

        newLinesAfter: function () {
            if (this.depth === 0 && !$(this.node).is(':last-child')) {
                // the topmost list has additional newline padding
                return new NewLines(2, true);
            }
            return NewLines.empty();
        },

        /**
         * Overrides processing each child node for lists.
         *
         * @param childNode child node that will be processed after this method
         * @param childIndex
         * @param childrenTotal
         * @returns empty or a text handler
         */
        createAfterChild: function(childNode, childIndex, childrenTotal) {
            if (childIndex !== childrenTotal - 1) {
                // add a new line for each item, but the last one
                return TextHandler.newlineHandler();
            }
            return new EmptyHandler();
        }
    });
});