define("jira/editor/converter/handlers/image-handler", [
    "jira/editor/converter/handlers/node-handler",
    "jira/editor/converter/handlers/text-handler",
    "jira/editor/converter/handlers/empty-handler",
    "jquery",
    "jira/editor/converter/util/strings",
    "wrm/context-path"
], function (NodeHandler,
             TextHandler,
             EmptyHandler,
             $,
             Strings,
             wrmContextPath) {
    "use strict";

    var IMAGE_PATH = '/images/icons/';
    var EMOTE_PATH = '/images/icons/emoticons/';

    var IMAGE_TYPES = {
        EMOTE: 'emote',
        EXTERNAL: 'external-image',
        ATTACHMENT: 'attachment-image',
        TEMPORARY_ATTACHMENT: 'temporary-attachment-image',
        PLACEHOLDER: 'placeholder'
    };

    /**
     * @class ImageHandler
     * @extends NodeHandler
     */
    var ImageHandler = NodeHandler.extend({
        init: function ImageHandler(node) {
            this.node = node;
            this.imageType = this._getImageType(node);
        },

        _getImageType: function (node) {
            if (this.node.getAttribute('imagetext')) {
                return IMAGE_TYPES.PLACEHOLDER;
            }
            var imageType = this.node.getAttribute('data-image-type');
            if (imageType) {
                return imageType;
            }

            var src = this.node.getAttribute('src');
            if (Strings.isEmotePath(src) && node.classList.contains('emoticon')) {
                if (Strings.contains(src, IMAGE_PATH + 'mail_small') && this.$(node).parents('a').is('.external-link')) {
                    return;
                }
                return IMAGE_TYPES.EMOTE;
            }
            if (Strings.isAttachmentPath(src) || Strings.isThumbnailPath(src) || src.match(/^((data|blob):http)|(data:image)/)) {
                return IMAGE_TYPES.ATTACHMENT;
            }
            if (Strings.isTemporaryAttachmentPath(src)) {
                return IMAGE_TYPES.TEMPORARY_ATTACHMENT;
            }

            return IMAGE_TYPES.EXTERNAL;
        },

        before: function () {
            var prev = this.node.previousSibling;
            var whitespace = (prev && prev.nodeType === Node.TEXT_NODE && !Strings.endsWith(prev.textContent, ' ')) ? ' ' : '';

            if (this.imageType && this.imageType !== IMAGE_TYPES.EMOTE) {
                return TextHandler.createTextHandler(whitespace + '!')
            }
            return new EmptyHandler();
        },

        children: function () {
            return this.imageType ? TextHandler.createTextHandler(this._getImageSource()) : new EmptyHandler();
        },

        after: function () {
            var next = this.node.nextSibling;
            var whitespace = (next && next.nodeType === Node.TEXT_NODE && !Strings.startsWith(next.textContent, ' ')) ? ' ' : '';

            if (this.imageType && this.imageType !== IMAGE_TYPES.EMOTE) {
                return TextHandler.createTextHandler('!' + whitespace);
            }
            return new EmptyHandler();
        },

        _getImageSource: function () {
            if (this.imageType === IMAGE_TYPES.PLACEHOLDER) {
                return this.node.getAttribute('imagetext');
            } else if (this.imageType === IMAGE_TYPES.EMOTE) {
                var emoteSource = this.node.getAttribute('src');
                var key = Strings.substringAfter(emoteSource, EMOTE_PATH).replace(/\.(png|gif)$/, '');

                if (key in emoticonMap) {
                    return emoticonMap[key];
                }

                if (Strings.startsWith(emoteSource, wrmContextPath())) {
                    return '!' + document.location.origin + emoteSource + '!';
                }
            } else if (this.imageType === IMAGE_TYPES.ATTACHMENT || this.imageType === IMAGE_TYPES.TEMPORARY_ATTACHMENT || this.imageType === IMAGE_TYPES.EXTERNAL) {
                var src = this.node.getAttribute('data-filename') ||
                    this.node.getAttribute('href') ||
                    this.node.getAttribute('src');

                if (this.imageType === IMAGE_TYPES.ATTACHMENT) {
                    src = Strings.getFilenameFromHref(src);
                } else if (this.imageType === IMAGE_TYPES.TEMPORARY_ATTACHMENT && !this.node.getAttribute('data-filename')) {
                    src = Strings.getFilenameFromHref(src, true);
                }

                var attributesString = ImageHandler.getAttrs(this.node);
                if (attributesString) {
                    return src + attributesString;
                }
                return src;
            }
        }
    });

    ImageHandler.getAttrs = function (node) {
        if ($(node).is('img.emoticon')) {
            return '';
        }

        var attrs = [];
        if (node.width) {
            attrs.push('width=' + node.width);
        }
        if (node.height) {
            attrs.push('height=' + node.height);
        }
        if (node.id) {
            attrs.push('id=' + node.id);
        }
        if (node.vspace) {
            attrs.push('vspace=' + node.vspace);
        }
        if (node.align) {
            attrs.push('align=' + node.align);
        }
        else if (node.style.textAlign) {
            attrs.push('align=' + node.style.textAlign);
        }
        else if (node.style['float']) {
            attrs.push('align=' + node.style['float']);
        }
        if (Strings.isThumbnailPath(node.getAttribute('src'))) {
            attrs.push('thumbnail');
        }

        return attrs.length > 0 ? '|' + attrs.join(',') : '';
    };

    var emoticonMap = {
        'smile': ':)', // We can't distinguish between ':-)' and ':)' , they render the same
        'sad': ':(',
        'tongue': ':P',
        'biggrin': ':D',
        'wink': ';)',
        'thumbs_up': '(y)',
        'thumbs_down': '(n)',
        'information': '(i)',
        'check': '(/)',
        'error': '(x)',
        'warning': '(!)',
        'add': '(+)',
        'forbidden': '(-)',
        'help_16': '(?)',
        'lightbulb_on': '(on)',
        'lightbulb': '(off)',
        'star_yellow': '(*)', // We can't distinguish between '(*y)' and '(*)', they render the same
        'star_red': '(*r)',
        'star_green': '(*g)',
        'star_blue': '(*b)',
        'flag': '(flag)',
        'flag_grey': '(flagoff)'
    };

    return ImageHandler;
});