define("jira/editor/converter/factories/whitespace-extractor", [
    "jira/editor/constants",
    "jira/editor/converter/handlers/text-handler",
    "jira/editor/converter/factories/left-trim",
    "jira/editor/converter/factories/right-trim"
], function (
    Constants,
    TextHandler,
    LeftTrim,
    RightTrim
) {
    "use strict";

    return function handlerFactory(BaseHandler) {

        /**
         * @class WhiteSpaceExtractor
         * @extends NodeHandler
         */
        return BaseHandler.extend({
            init: function WhiteSpaceExtractor(node) {
                this._super(node);

                // has to be before createChildHandler is called, it matters for `after`
                this.leftWhitespaces = matchNodeContent(node.firstChild, Constants.LEFT_WHITESPACE_MATCHER);
                this.rightWhitespaces = matchNodeContent(node.lastChild, Constants.RIGHT_WHITESPACE_MATCHER);

                this.leftTrimChildren = !!this.leftWhitespaces;
                this.rightTrimChildren = !!this.rightWhitespaces;
            },

            before: function before() {
                var handler = this._super();
                return this._extractLeftSpaces(handler) || handler;
            },

            after: function after() {
                var handler = this._super();
                return this._extractRightSpaces(handler) || handler;
            },

            createChildHandler: function createChildHandler(childNode, childIndex, childrenTotal) {
                var handler = this._super(childNode, childIndex, childrenTotal);
                if (this.leftTrimChildren && 0 === childIndex) {
                    handler = LeftTrim(handler);
                }
                if (this.rightTrimChildren && childrenTotal - 1 === childIndex) {
                    handler = RightTrim(handler);
                }
                return handler;
            },

            _extractLeftSpaces: function _extractLeftSpaces(parentHandler) {
                if (!this.leftTrimChildren) {
                    return null;
                }

                var factoryMethod = getTextHandlerClass(parentHandler);
                return factoryMethod(this.leftWhitespaces + parentHandler.processText());
            },

            _extractRightSpaces: function _extractRightSpaces(parentHandler) {
                if (!this.rightTrimChildren) {
                    return null;
                }

                var factoryMethod = getTextHandlerClass(parentHandler);
                return factoryMethod(parentHandler.processText() + this.rightWhitespaces);
            }
        });
    };

    function matchNodeContent(node, regex) {
        var textContent = (node && node.textContent) || '';
        var whitespaces = textContent.match(regex);
        return (null !== whitespaces) ? whitespaces[0] : null;
    }

    function getTextHandlerClass(handler) {
        return (handler.preformatted)
            ? TextHandler.createPreformattedTextHandler
            : TextHandler.createTextHandler;
    }
});