define('jira/editor/context-manager', [
    'jira/editor/context-detector',
    'backbone',
    'underscore'
], function (
    ContextDetector,
    Backbone,
    _
) {
    var ContextManager = function (editor) {
        this._disableStates = {
            all: null,
            table: null,
            a: null
        };
        this._tagList = _.difference(Object.keys(this._disableStates), 'all');
        this._editorInstance = editor;

        editor.on("selection:update", function (e) {
            var all = e.insidePreformatted || e.preformattedSelected;
            this._setDisable(all, "all");
            this._setDisable(all || e.insideTable, "table");
            this._setDisable(all || e.insideA, "a");
        }.bind(this));

        editor.on("tabs:changed", function () {
            this._setDisable(false, "all");
            this._setDisable(false, "table");
            this._setDisable(false, "a");
        }.bind(this));

        _.extend(this, Backbone.Events);
    };

    ContextManager.prototype._setDisable = function(disableState, tag) {
        if (this._disableStates[tag] === disableState) {
            return;
        }

        this._disableStates[tag] = disableState;

        this.trigger("change:" + tag, {
            disableState: disableState
        });
        if ('all' === tag) {
            this._tagList.forEach(function(tag) {
                this._setDisable(disableState, tag);
            }, this);
        }
    };

    ContextManager.prototype.isPreContextActive = function() {
        return true === this._disableStates['all'];
    };

    ContextManager.prototype.isContextActive = function(contextName) {
        return true === this._disableStates[contextName];
    }

    ContextManager.prototype.getPasteSchemaForContext = function () {
        return ContextDetector.detectPre(this._editorInstance.editor.selection.getNode())
            ? this._editorInstance.pasteInsidePreSchemaSpec : this._editorInstance.pasteSchemaSpec;
    };

    return ContextManager;
});