UPM.define("ManageApplicationsView", [
    'jquery',
    'underscore',
    'UpmAjax',
    'UpmDialog',
    'UpmEnvironment',
    'UpmXsrfTokenState',
    'UpmInstaller',
    'BaseView',
    'ApplicationModel',
    'ApplicationCollectionView',
    'AvailableApplicationCollectionView',
    'ProgressDialog',
    'ProgressDialogModel',
    'AuiFlagManager',
    'CustomerAgreementDialog',
    'UpmAnalytics',
    'ManageApplicationsEnvironment'
], function($,
            _,
            UpmAjax,
            UpmDialog,
            UpmEnvironment,
            UpmXsrfTokenState,
            UpmInstaller,
            BaseView,
            ApplicationModel,
            ApplicationCollectionView,
            AvailableApplicationCollectionView,
            ProgressDialog,
            ProgressDialogModel,
            AuiFlagManager,
            CustomerAgreementDialog,
            UpmAnalytics,
            ManageApplicationsEnvironment) {


    'use strict';


    return BaseView.extend({

        events: {
            "click #upload-application-trigger": "_uploadApplicationToCollection",
            "click a.sidebar-link": "_sidebarLinkHandler"
        },

        _initEvents: function () {
            this.listenTo(UpmAjax.getEventTarget(), 'ajaxError', this._onAjaxError);
        },

        _postInitialize: function () {
            this.model.setApplications(this._getJsonBlobFromDOM('#manage-applications-apps-json', []));
            this.model.setAvailableApplications(this._getJsonBlobFromDOM('#manage-applications-available-apps-json', []));
            this._addOrphanedLicensesFromUninstalledApps();

            _.forEach(this.$el.find('#manage-applications-flag-notification-data .flag-notification'), function(notification) {
                AuiFlagManager.delayedFlag({
                    type: $(notification).data('type').toLowerCase(),
                    body: $(notification).data('content'),
                    close: 'manual'
                });
            });

            new ApplicationCollectionView({
                collection: this.model.getApplications(),
                el: this.$el.find('#manage-applications-list')
            }).render();

            new AvailableApplicationCollectionView({
                collection: this.model.getAvailableApplications(),
                el: this.$el.find('#manage-applications-available-apps-list')
            }).render();

            this.listenTo(this.model.getApplications(), 'startInstallation', this.disableInstallations);
            this.listenTo(this.model.getApplications(), 'stopInstallation', this.enableInstallations);
            this.listenTo(this.model.getApplications(), 'installed', this._refreshSidebar);
            this.listenTo(this.model.getApplications(), 'uninstalled', this._onApplicationUninstalled);
            this.listenTo(this.model.getApplications(), 'unlicensed', this._onApplicationUnlicensed);
            this.listenTo(this.model.getAvailableApplications(), 'tryAvailableApp', this._tryNewApplication);
            this._logNotificationAnalytics();
        },

        /**
         * Updates the display after an application has been removed.
         */
        _onApplicationUninstalled: function(appModel) {
            if (!appModel.isLicensed()) {
                this.model.getApplications().remove(appModel);
            }
            this._refreshSidebar();
        },

        /**
         * Updates the display after an application has been unlicensed.
         */
        _onApplicationUnlicensed: function (appModel) {
            if (appModel.getLicensedButNotInstalled()) {
                this.model.getApplications().remove(appModel);
            }
        },

        /**
         * Send analytics when sidebar links are clicked. Links open in a new
         * window via target="_blank" so no need to prevent default behavior
         */
        _sidebarLinkHandler: function (e) {
            UpmAnalytics.logEvent($(e.target).data().analyticsKey);
        },

        /**
         * Sync the available apps collection. If it is empty, hide the containing
         * div to hide the list header. Any time the sidebar is refreshed, run through
         * the main apps, and check for orphaned license apps. If there are any, copy
         * over the links hash from the sidebar so the app can get its appropriate binary link
         */
        _refreshSidebar: function() {
            var self = this;
            return this.model.updateAvailableApplications()
                .then(function () {
                    self.model.getApplications().each(function (appModel) {
                        var sidebarApp = self.model.findAvailableAppByKey(appModel.getKey());
                        if (appModel.getLicensedButNotInstalled() && sidebarApp && sidebarApp.has('links')) {
                            appModel.set({
                                name: sidebarApp.getName(),
                                links: _.extend(appModel.getLinks(), sidebarApp.getLinks())
                            });
                        }
                    });
                })
                .always(function () {
                    self.$el.find('#manage-applications-available-apps')
                        .toggleClass('hidden', !self.model.hasAvailableApplications());
                });
        },
        
        /**
         * Prompt the user to upload an application either via file upload or url.
         * File uploads are posted to an iframe on the page. Urls are posted to the server.
         * If not cancelled, the dialog will change to a progress dialog while the installation happens
         */
        _uploadApplicationToCollection: function (e) {

            e.preventDefault();
            UpmXsrfTokenState.refreshToken();
            UpmAnalytics.logEvent('manageapps-upload-link-click');

            var progressDialog,
                app = new ApplicationModel(),
                progressModel = new ProgressDialogModel(),
                startProgressCallback = function () {
                    progressDialog = new ProgressDialog({
                        model: progressModel,
                        data: {
                            headerHtml: AJS.I18n.getText('upm.applications.dialog.upload.header'),
                            messageHtml: AJS.I18n.getText('upm.applications.dialog.upload.progress')
                        }
                    });
                };

            UpmInstaller.openInstallDialog(
                startProgressCallback,
                { uploadApplication: true }
            ).then(
                _.bind(function (response) {
                    return app.pollStatusThenSync(response,
                        _.bind(progressModel.setProgressPercent, progressModel));
                }, this)
            ).then(
                _.bind(function () {
                    this.model.getApplications().add(app);
                    this.model.getApplications().trigger('installed');
                    progressDialog.finishProgress();

                    AuiFlagManager.flag({
                        type: 'success',
                        body: AJS.I18n.getText('upm.applications.flags.installation.success', app.getName()),
                        close: 'auto'
                    });

                    UpmAnalytics.logEvent('manageapps-upload-success', { appKey: app.getKey(), version: app.getVersion() });
                    UPM.trace('application-uploaded');
                }, this)
            ).fail(
                function (xhr, status, msg) {
                    progressDialog.close();
                    if (msg) {
                        UpmAjax.getEventTarget().trigger('ajaxError', msg);
                        UpmAnalytics.logEvent('manageapps-upload-failure');
                        UPM.trace('application-upload-failed');
                    }
                }
            );
        },

        /**
         * Install application from the "Other Applications" sidebar
         */
        _tryNewApplication: function (availableAppModel) {
            var app = this.model.findAppByKey(availableAppModel.getKey());
            if (app && app.isLicensed() && app.isInstallable()) {
                this._installAlreadyLicensedApplication(app.getKey());
            } else {
                this._installAndTrialAvailableApplication(availableAppModel);
            }
        },

        /**
         * Initiate download and eval flow. Prompts for AID credentials, then installs an
         * application and gets an evaluation license for it
         */
        _installAndTrialAvailableApplication: function (availableAppModel) {
            var appKey = availableAppModel.getKey(),
                aidOptions = {
                    applicationFormat: true,
                    applicationName: availableAppModel.getName()
                },
                app,
                me = this;

            var addAppModelAndInstall = function() {
                if (app) {
                    return $.Deferred().resolve();
                } else {
                    // Add app to collection, then begin installation from binary uri
                    me.model.getApplications().add(
                        _.extend(availableAppModel.toJSON(), {
                            progressMessage: AJS.I18n.getText('upm.applications.installing'),
                            platform: false
                        })
                    );
                    app = me.model.findAppByKey(appKey);

                    // Prevents multiple simultaneous installations
                    me.disableInstallations();

                    me._scrollToApplication(appKey);
                    return app.installFromBinaryUri();
                }
            };

            return new CustomerAgreementDialog().getResult().then(
                function() {
                    if (ManageApplicationsEnvironment.isDataCenter()) {
                        return addAppModelAndInstall().then(function() {
                            AuiFlagManager.flag({
                                type: 'warning',
                                body: AJS.I18n.getText('upm.applications.installation.eval.fail.datacenter', app.getName(), AJS.I18n.getText('upm.applications.notifications.datacenter.suffix')),
                                close: 'manual'
                            });
                            app.stopLoading();
                        });
                    } else {
                        return addAppModelAndInstall().then(
                            function () {
                                app.stopLoading();
                                return new UpmDialog({
                                    template: com.atlassian.upm.application.evalRedirectDialog
                                }).getResult().then(function(){ return app.startEvaluation(); });
                            },
                            // If the app has a version set, then it got it from the REST api, so it must've finished
                            // installing. Otherwise, remove it from the collection list.
                            // Don't forget to re-enable uploads so the user can try again.
                            function (xhr, status, msg) {
                                if (app) {
                                    app.stopLoading();
                                    if (app.isApplicationStub()) {
                                        me.model.getApplications().remove(app);
                                        UpmAjax.getEventTarget().trigger('ajaxError', msg);
                                    }
                                } else if (msg) {
                                    UpmAjax.getEventTarget().trigger('ajaxError', msg);
                                }
                                me.enableInstallations();
                            });
                    }
                }
            ).done(
                function () {
                    if (!ManageApplicationsEnvironment.isDataCenter()) {
                        var postInstallLink = app.getLinks()['post-install'];
                        AuiFlagManager.delayedFlag({
                            type: 'success',
                            body: (postInstallLink ?
                                AJS.I18n.getText('upm.applications.flags.installation.success.postinstall', app.getName(), postInstallLink) :
                                AJS.I18n.getText('upm.applications.flags.installation.success', app.getName())),
                            close: postInstallLink ? 'manual' : 'auto'
                        });
                        UPM.trace('eval-success');
                    }
                    me.enableInstallations();
                }
            )
        },

        /**
         * Installs an already licensed application. This is called if the user clicks the "Try it for free"
         * button on an available app in the sidebar when the app is already licensed and in their main
         * applications list. This way, they are not prompted for AID credentials, and no Eval license is requested
         */
        _installAlreadyLicensedApplication: function (applicationKey) {
            var app = this.model.findAppByKey(applicationKey);
            this.disableInstallations();
            this._scrollToApplication(applicationKey);
            app.installFromBinaryUri().then(
                _.bind(function () {
                    var postInstallLink = app.getLinks()['post-install'];
                    app.stopLoading();
                    AuiFlagManager.delayedFlag({
                        type: 'success',
                        body: (postInstallLink ?
                            AJS.I18n.getText('upm.applications.flags.installation.success', app.getName()) :
                            AJS.I18n.getText('upm.applications.flags.installation.success.postinstall', app.getName(), postInstallLink)
                            ),
                        close: postInstallLink ? 'manual' : 'auto'
                    });
                    this.enableInstallations();
                }, this)
            ).fail(
                _.bind(function (xhr, status, msg) {
                    app.stopLoading();
                    this.enableInstallations();
                    if (msg) {
                        UpmAjax.getEventTarget().trigger('ajaxError', msg);
                    }
                })
            );
        },

        /**
         * Append faux application representations from orphaned licenses left behind by uninstalled
         * applications. Try to take advantage of the available-app data of apps in the sidebar
         * (if mpac is available) to fill in the binary file links to offer a one-click install flow for the user.
         */
        _addOrphanedLicensesFromUninstalledApps: function () {
            var orphanedLicenses = this._getJsonBlobFromDOM('#manage-applications-uninstalled-app-licenses', []);
            _.forEach(orphanedLicenses, _.bind(function (application) {
                var sidebarApp = this.model.findAvailableAppByKey(application.key);
                    sidebarApp = sidebarApp ? sidebarApp.toJSON() : {};
                this.model.getApplications().add(
                    _.extend({ licensedButNotInstalled: true }, application, sidebarApp)
                );
            }, this));
        },

        /*
         * UI Convenience Methods
         */
        _scrollToApplication: function (key) {
            setTimeout(function () {
                $('html,body').animate({
                    scrollTop: $('.application-item[data-application-key="' + key + '"]').offset().top
                });
            });
        },

        disableInstallations: function () {
            this.$el.find('#upload-application-trigger, .available-app button.try, .update-notification-action')
                .attr('disabled', 'disabled')
                .attr('aria-disabled', 'true');
        },

        enableInstallations: function () {
            this.$el.find('#upload-application-trigger, .available-app button.try, .update-notification-action')
                .removeAttr('disabled')
                .removeAttr('aria-disabled');
        },

        _getJsonBlobFromDOM: function (selector, defaultValue) {
            var json = this.$el.find(selector).attr('data-value');
            if (json) {
                json = JSON.parse(json);
            }
            return json || defaultValue;
        },

        /**
         * When ajax errors occur, throw a flag
         */
        _onAjaxError: function (errorMessage) {
            var err = _.isString(errorMessage) ? errorMessage : AJS.I18n.getText('upm.applications.ajax.error.unknown');
            AuiFlagManager.flag({
                type: 'error',
                body: err,
                close: 'manual'
            });
        },

        /**
         * Fires analytics events for all inline app notifications on initial page load
         * @private
         */
        _logNotificationAnalytics: function () {
            this.model.getApplications().each(function (app) {
                app.getNotifications().each(function (note) {
                    var metadata = {
                        appKey: app.getKey()
                    };
                    if (note.getHasOverride()) {
                        metadata.hasOverride = true;
                    }
                    switch (note.getType()) {
                        case 'renew-notification':
                        case 'evaluation-notification':
                            metadata.daysToExpiry = app.getLicense().timeUntilExpiry('days');
                            break;

                        case 'upgrade-notification':
                            metadata.remainingUsers = app.getAccessDetails().remainingUsers();
                            break;

                        case 'update-notification':
                            metadata.updateVrs = app.getUpdate().updateVersion;
                            break;

                        case 'renew-to-update-notification':
                            metadata.daysToExpiry = app.getLicense().timeUntilExpiry('days');
                            metadata.updateVrs = app.getUpdate().updateVersion;
                            break;
                    }
                    UpmAnalytics.logEvent('manageapps-' + note.getType(), metadata);
                });
            });
        }
    });
});