/**
 * Represents an Application object.
 */
UPM.define('ApplicationModel', [
    'brace',
    'backbone',
    'jquery',
    'underscore',
    'moment',
    'ManageApplicationsEnvironment',
    'UpmEnvironment',
    'UpmAjax',
    'UpmXsrfTokenState',
    'UpmInstaller',
    'LicenseModel',
    'AccessDetailsModel',
    'NotificationModel',
    'VersionInfoModel',
    'NotificationCollection',
    'ApplicationPluginCollection'
], function(Brace,
            Backbone,
            $,
            _,
            moment,
            ManageApplicationsEnvironment,
            UpmEnvironment,
            UpmAjax,
            UpmXsrfTokenState,
            UpmInstaller,
            LicenseModel,
            AccessDetailsModel,
            NotificationModel,
            VersionInfoModel,
            NotificationCollection,
            ApplicationPluginCollection) {


    'use strict';


    return Brace.Model.extend({

        namedAttributes: {
            key: 'string',
            name: 'string',
            version: false, // can be number or string
            versionInfo: VersionInfoModel,
            description: 'string',
            licensedUserCountDescription: 'string',
            platform: 'boolean',
            uninstallable: 'boolean',
            licensedButNotInstalled: 'boolean',
            license: LicenseModel,
            accessDetails: AccessDetailsModel,
            links: 'object',
            notifications: NotificationCollection,
            plugins: ApplicationPluginCollection,
            loading: 'boolean',
            progressMessage: 'string',
            progressValue: 'number',
            update: 'object'
        },

        initialize: function () {
            this._setNotifications();
            this.on('sync', this._setNotifications);
            this.on('change:links', this._setNotifications);
        },

        /*
         * Set the model's url for fetching from the server
         */
        url: function () {
            return this.has('links') && this.getLinks().self ? this.getLinks().self : UpmEnvironment.getResourceUrl('applications-rest') + '/installed/' + this.getKey();
        },

        licenseUrl: function () {
            return this.getLinks().license || this.url() + '/license';
        },

        configurationUrl: function () {
            return this.getLinks()['clear-configuration'];
        },

        /*
         * Called by `this.fetch` to prepare the model for setting with the ajax response.
         * Fetch does not unset existing values that are not in the server response, so we
         * have to manually clean out the model to properly sync it. Only does so if you
         * pass in a `reset` flag ala `fetch({ reset: true })`
         */
        parse: function (response, options) {
            var keys = _.keys(this.omit('loading', 'progressMessage', 'progressValue'));
            if (options.reset) {
                _.each(keys, _.bind(function (key) {
                    if (!_.has(response, key)) {
                        this.unset(key);
                    }
                }, this));
            }
            return response;
        },

        isPlatform: function () {
            return !!this.getPlatform();
        },

        isDataCenter: function () {
            return this.isLicensed() && this.get('license').get('dataCenter');
        },

        isLicensed: function() {
            return this.has('license') && this.get('license').has('rawLicense');
        },

        isLoading: function () {
            return this.has('loading') && this.getLoading();
        },

        isEvaluation: function () {
            return this.isLicensed() && this.get('license').get('evaluation');
        },

        isTryable: function () {
            return this.getLinks() && _.has(this.getLinks(), 'try');
        },

        isBuyable: function () {
            return this.getLinks() && _.has(this.getLinks(), 'buy');
        },

        isRenewable: function () {
            return this.getLinks() && _.has(this.getLinks(), 'renew');
        },

        isUpgradable: function () {
            return this.getLinks() && _.has(this.getLinks(), 'upgrade');
        },

        isInstallable: function () {
            return this.getLinks() && _.has(this.getLinks(), 'binary');
        },

        isUpdatable: function () {
            return !!this.getUpdate() && (!!this.getBinaryUpdateUri() || !!this.getDownloadsPageUri());
        },

        isIncompatible: function () {
            return this.has('versionInfo') ? !this.getVersionInfo().isCompatible() : false;
        },

        // When doing installations from the sidebar, we add temporary stub
        // applications just for the sake of a progressbar. This check looks
        // for the 'self' link as a way of determining whether this application
        // is a stub since all backend-provided representations will include this link
        isApplicationStub: function () {
            return !this.getLicensedButNotInstalled() && (!this.getLinks() || !_.has(this.getLinks(), 'self'));
        },

        getUpdateReleaseNotes: function () {
            return this.getUpdate() && this.getUpdate().links['release-notes'];
        },

        getBinaryInstallUri: function () {
            return this.isInstallable() ? this.getLinks().binary : undefined;
        },

        getBinaryUpdateUri: function () {
            return this.getUpdate() && this.getUpdate().links.binary;
        },

        getDownloadsPageUri: function () {
            return this.getLinks() && this.getLinks().downloads;
        },
        
        /**
         * Convenience method for getting raw license string from nested LicenseModel
         * @returns String rawLicense
         */
        getLicenseKey: function () {
            return this.isLicensed() ? this.get('license').get('rawLicense') : '';
        },

        /**
         * Unset the loading, progressValue and progressMessage
         */
        stopLoading: function () {
            if (this.isLoading()) {
                this.unset('loading');
                this.unset('progressMessage');
                this.unset('progressValue');
            }
            return this;
        },

        /**
         * Redirects the user to MAC to generate the application evaluation License
         * @returns {Promise} that is always rejected.
         */
        startEvaluation: function() {
            var $form = $('<form method="post" class="hidden"></form>'),
                licenseDetails = this.getLicense(),
                hostLicense = UpmEnvironment.getHostLicense(),
                orgName = hostLicense ? hostLicense.organizationName : undefined,
                productKey = this.getKey(),
                serverId = UpmEnvironment.getServerId(),
                values;
            $form.attr('action', this.getLinks()['try']);
            values = _.extend(
                {
                    callback: this.getLinks()['eval-license-callback'],
                    licensefieldname: 'license',
                    productKey: productKey,
                },
                serverId ? {sid: serverId} : {},
                orgName ? { organisation_name: orgName } : {},
                (licenseDetails && licenseDetails.contactEmail) ? { owner: licenseDetails.contactEmail } : {}
            );

            $form.append(_.map(values, function(value, key) {
                return $('<input type="hidden">').attr('name', key).val(value);
            }));

            $form.appendTo(document.body);
            $form.submit();
            $form.remove();
            return $.Deferred().reject().promise();
        },

        /**
         * Initiate license change with provided key. Sets progress message and value for
         * the view to subscribe to to update the ui
         * @param licenseKey
         * @returns Promise that will resolve when complete, or reject with XHR failure details
         */
        submitLicense: function (licenseKey) {
            var deferred = $.Deferred();
            this.set({
                progressMessage: AJS.I18n.getText('upm.applications.license.key.updating'),
                loading: true
            });

            this._updateLicenseKey(licenseKey).then(
                _.bind(function (resp) {
                    return this.fetch({ reset: true }).then(
                        function() {
                            deferred.resolve(resp);
                        }
                    );
                }, this)
            ).fail(
                function (xhr, status, message) {
                    var err = UpmAjax.getAjaxErrorMessage(xhr, status, message);
                    deferred.reject(xhr, err.status, err.message);
                }
            ).always(
                _.bind(function () {
                    this.stopLoading();
                }, this)
            );

            return deferred.promise();
        },

        /**
         * Delete a license key from the app. If the app happens to be already uninstalled,
         * then it will trigger the remove event to take it off the page
         * @param deleteConfig. boolean to determine whether or not to also delete app configuration
         * @returns Promise that will resolve when complete, or reject with XHR failure details
         */
        deleteLicenseAndConfiguration: function (deleteConfig) {
            var deferred = $.Deferred();
            this.set({
                progressMessage: AJS.I18n.getText('upm.applications.license.key.deleting'),
                loading: true
            });

            this._updateLicenseKey().then(
                _.bind(function (data) {
                    // You can only delete the configuration if the application is installed
                    if (deleteConfig && !this.getLicensedButNotInstalled()) {
                        return this._deleteConfiguration();
                    }
                    return data;
                }, this)
            ).then(
                _.bind(function (data) {
                    this.unset('license');
                    if (!this.getLicensedButNotInstalled()) {
                        return this.fetch({ reset: true });
                    }
                    return data;
                }, this)
            ).then(
                _.bind(function (data) {
                    this.trigger('unlicensed', this);
                    deferred.resolve(data);
                }, this)
            ).fail(
                function (xhr, status, message) {
                    var err = UpmAjax.getAjaxErrorMessage(xhr, status, message);
                    deferred.reject(xhr, err.status, err.message);
                }
            ).always(
                _.bind(function () {
                    this._setNotifications();
                }, this)
            );

            return deferred.promise();
        },

        /**
         * If this model is for the platform, and it has an available update, request a list of
         * other applications that could/should be updated along with the platform.
         * @return {Promise}  a Promise that will be resolved with the result representation
         *   (ApplicationUpdateCollectionRepresentation) or rejected with an error
         */
        findAppUpdatesForPlatform: function() {
            if (this.getPlatform() && this.getUpdate()) {
                return $.ajax({
                    method: 'get',
                    url: ManageApplicationsEnvironment.getAppUpdatesUrl(this.getUpdate().buildNumber),
                    dataType: 'json'
                });
            } else {
                return $.Deferred().reject();
            }
        },

        /**
         * Public method used to 'install' this application model. Requires the model to have
         * a 'binary' uri in it's links object. If not, throws an ajax error. Otherwise, passes
         * off to the _installFromBinaryUri method passing in the binary link
         */
        installFromBinaryUri: function () {
            var deferred = $.Deferred();
            UpmXsrfTokenState.refreshToken().then(
                _.bind(function () {
                    return this._installFromBinaryUri(this.getBinaryInstallUri());
                }, this)
            ).then(
                deferred.resolve
            ).fail(
                function (xhr, status, message) {
                    var err = UpmAjax.getAjaxErrorMessage(xhr, status, message);
                    deferred.reject(xhr, err.status, err.message);
                }
            );
            return deferred.promise();
        },

        scheduleAppUpdatesForPlatform: function(appsToUpdate) {
            if (this.getPlatform() && this.getUpdate()) {
                var url = ManageApplicationsEnvironment.getScheduledBundledUpdatesUrl(this.getUpdate().buildNumber);
                if (appsToUpdate && appsToUpdate.length) {
                    return $.ajax({
                        method: 'post',
                        url: url,
                        contentType: UpmEnvironment.getContentType(),
                        data: JSON.stringify({ updates: appsToUpdate })
                    });
                } else {
                    return $.ajax({
                        method: 'delete',
                        url: url                        
                    });
                }
            } else {
                return $.Deferred().reject();
            }
        },

        /**
         * Public method used to update this application model should it have an updateVersion
         * and an updateVersion binary uri. If not, will throw an ajax error. Otherwise, passes
         * off to the _installFromBinaryUri method passing in the update binary link
         */
        updateApplication: function () {
            var deferred = $.Deferred();
            this.trigger('startInstallation');
            this.setProgressMessage(AJS.I18n.getText('upm.applications.updating', this.getName()));
            UpmXsrfTokenState.refreshToken().then(
                _.bind(function () {
                    return this._installFromBinaryUri(this.getBinaryUpdateUri());
                }, this)
            ).then(
                deferred.resolve
            ).fail(
                function (xhr, status, message) {
                    var err = UpmAjax.getAjaxErrorMessage(xhr, status, message);
                    deferred.reject(xhr, err.status, err.message);
                }
            ).always(
                _.bind(function () {
                    this.trigger('stopInstallation');
                    this.stopLoading();
                }, this)
            );
            return deferred.promise();
        },

        /**
         * Use once you've initiated an installation. Polls the task resource for updates
         * and once finished, get the applicationKey from the result and sync the model from
         * the server.
         * @param taskResponse  Initial response from the install request
         * @param progressUpdateCallback. Fired with each poll with the status value (btw 0 and 1)
         * @returns {*}
         */
        pollStatusThenSync: function (taskResponse, progressUpdateCallback) {
            var deferred = $.Deferred();
            UpmAjax.pollTaskResource(taskResponse, progressUpdateCallback).then(
                _.bind(function (response) {
                    if (response.applicationKey) {
                        if (!this.has('key')) {
                            this.set('key', response.applicationKey);
                        }
                        return this.fetch({ reset: true })
                            .done(_.bind(function() {
                                    this.trigger('installed');
                                }, this));
                    } else {
                        // This is an add-on, not an application - bounce to UPM
                        progressUpdateCallback(1);
                        UPM.trace('plugin-uploaded');
                        setTimeout(function () {
                            window.location.href = response.links.manage;
                        }, 250);
                        return $.Deferred();
                    }
                }, this)
            ).then(
                deferred.resolve
            ).fail(
                function (xhr, status, message) {
                    var err = UpmAjax.getAjaxErrorMessage(xhr, status, message);
                    deferred.reject(xhr, err.status, err.message);
                }
            );
            return deferred.promise();
        },

        /**
         * Private method used to 'install' this application model using a provided uri.
         * Will run the UpmInstaller.installFromUrl method, then pass off the results to
         * the pollStatusThenSync method. Polling status is stored in this model's
         * progressValue property, and can be listened to to update a progressbar on the view.
         */
        _installFromBinaryUri: function (uri) {
            return UpmInstaller.installFromUrl(
                uri,
                _.bind(function () {
                    this.set({
                        loading: true,
                        progressValue: 0.0
                    });
                }, this)
            ).then(
                _.bind(function (response) {
                    return this.pollStatusThenSync(response,
                        _.bind(this.setProgressValue, this));
                }, this)
            );
        },

        /**
         * Calls out to Hamlet to get an evaluation license for this application
         * @param token. Atlassian ID XSRF token
         * @returns jqXHR
         */
        _getEvaluationLicense: function (token) {
            var data = { productKey: this.getKey() };

            if (UpmEnvironment.getServerId()) {
                data.serverId = UpmEnvironment.getServerId();
            }
            return UpmAjax.ajaxCorsOrJsonp({
                url: this.getLinks()['try'],
                type: 'post',
                contentType: 'application/json',
                data: JSON.stringify(data),
                dataType: 'json',
                headers: {
                    'ATL-XSRF-Token': token
                },
                timeout: 15000,
                xhrFields: {
                    withCredentials: true
                }
            });
        },
        
        startLoading: function(message) {
            this.set({
                loading: true,
                progressMessage: message,
                progressValue: 0
            });
        },
        
        /**
         * Update license key via REST API. If licenseKey is provided, will POST it to server.
         * If not, will do a DELETE request to the server to remove the existing license key
         * @param licenseKey (optional)
         * @returns Promise that will resolve when complete, or reject with XHR failure details
         */
        _updateLicenseKey: function(licenseKey) {
            return UpmAjax.ajax({
                url: this.licenseUrl(),
                contentType: UpmEnvironment.getContentType(),
                type: (licenseKey ? 'POST' : 'DELETE'),
                data: JSON.stringify({ licenseKey: licenseKey })
            });
        },

        _deleteConfiguration: function () {
            return UpmAjax.ajax({
                url: this.configurationUrl(),
                contentType: UpmEnvironment.getContentType(),
                type: 'DELETE'
            });
        },

        _uninstall: function (pluginKeysToUninstall, progressCallback) {
            return UpmAjax.ajax({
                url: this.getLinks()['delete'],
                type: 'POST',
                contentType: 'application/x-www-form-urlencoded',
                data: { 'pluginKey': pluginKeysToUninstall },
                traditional: true  // this switch suppresses JQuery's habit of adding "[]" to the name of a multi-valued parameter
            }).then(function(response) {
                return UpmAjax.pollTaskResource(response, progressCallback);
            });
        },

        /**
         * Asks the back end to uninstall all of the uninstallable plugins that make up this application.
         * @param {Function} progressCallback  will be called with the progress (0.0-1.0) while the
         *   request is in progress
         * @param {boolean} deleteLicense. Whether or not to remove the license when uninstalling the app
         * @returns {Promise}  a Promise that will be resolved if the entire request completes
         *   successfully, or rejected with XHR failure details
         */
        uninstall: function(progressCallback, deleteLicenseAndConfig) {
            var allPlugins = this.getPlugins(),
                primary = allPlugins.filter(function(p) { return p.isPrimary(); }),
                app = allPlugins.filter(function(p) { return p.isApplication(); }),
                pluginKeysToUninstall = _.map(_.flatten([primary, app]), function(p) { return p.getKey(); }),
                deleteMethod = this.isLicensed() ? 'deleteLicenseAndConfiguration' : '_deleteConfiguration';

            // clear out everything. Need to delete config/license BEFORE uninstalling application
            if (deleteLicenseAndConfig) {
                return this[deleteMethod](true).then(
                    _.bind(function () {
                        return this._uninstall(pluginKeysToUninstall, progressCallback);
                    }, this)
                ).done(
                    _.bind(function () {
                        this.setUninstallable(false);
                        this.unset('accessDetails');
                        this.unset('version');
                        this.trigger('uninstalled', this);
                    }, this)
                ).always(
                    _.bind(function () {
                        this.stopLoading();
                    }, this)
                ).promise();
            }

            // Opting in to licensed-but-uninstalled state
            return this._uninstall(pluginKeysToUninstall, progressCallback).done(
                _.bind(function () {
                    this.setLicensedButNotInstalled(true);
                    this.setUninstallable(false);
                    this.unset('accessDetails');
                    this.unset('version');
                    this.setLinks({
                        license: this.getLinks().license,
                        binary: this.getLinks().binary
                    });
                    this._setNotifications();
                    this.trigger('uninstalled', this);
                }, this)
            ).always(
                _.bind(function () {
                    this.stopLoading();
                }, this)
            ).promise();
        },

        /**
         * All logic relating to determining what notifications are appropriate for this application
         * and what they should say
         * @private
         */
        _setNotifications: function () {
            this.setNotifications(new NotificationCollection());
            if (!this.isLicensed() && this.isTryable() && this.isBuyable() && !this.isPlatform()) {
                if (ManageApplicationsEnvironment.isDataCenter()) {
                    this.getNotifications().add(this._createUnlicensedDataCenterNotification());
                } else {
                    this.getNotifications().add(this._createUnlicensedTryAndBuyNotification());
                }
            }
            if (this.isEvaluation() && this.isBuyable()) {
                this.getNotifications().add(this._createEvaluationBuyNotification());
            }
            if (this.isDataCenter()) {
                if (this.isRenewable()) {
                    this.getNotifications().add(this._createRenewNotification());
                }
                if (this.isUpdatable() && !this.getLicense().isExpired()) {
                    this.getNotifications().add(this.isIncompatible() ?
                        this._createIncompatibilityNotification() :
                        this._createUpdateNotification());
                }
            } else {
                if (this.isUpdatable() && this.isRenewable() && this.getLicense().isExpired()) {
                    this.getNotifications().add(this._createRenewToUpdateNotification());
                    if (this.isIncompatible()) {
                        this.getNotifications().add(this._createIncompatibilityNotification());
                    }
                } else {
                    if (this.isRenewable()) {
                        this.getNotifications().add(this._createRenewNotification());
                    }
                    if (this.isIncompatible() && this.isUpdatable()) {
                        this.getNotifications().add(this._createIncompatibilityNotification());
                    } else if (this.isUpdatable()) {
                        this.getNotifications().add(this._createUpdateNotification());
                    }
                }
            }
            if (this.isUpgradable()) {
                this.getNotifications().add(this._createUpgradeNotification());
            }
            if (this.getLicensedButNotInstalled() && this.isLicensed()) {
                this.getNotifications().add(this._createLicensedButNotInstalledNotification());
            }
            if (this.isIncompatible() && this.getVersionInfo().requiresPlatformUpdate()) {
                this.getNotifications().add(this._createIncompatibilityNotification());
            }

            try {
                var whisperMessages = require("atlassian-whisper/messages");
                var appKey = this.get("key");
                this.getNotifications().forEach(function (notification) {
                    if (whisperMessages.hasOverride("upm-" + notification.get("type")) ||
                        whisperMessages.hasOverride("upm-" + appKey + "-" + notification.get("type"))) {
                        notification.set("hasOverride", true);
                    }
                })
            } catch(e) {
                // whisper is not available, might be disabled
            }
        },

        /**
         * Unlicensed Notifications: https://extranet.atlassian.com/display/JIRADEV/Manually+upload+and+license+product
         */
        _createUnlicensedTryAndBuyNotification: function () {
            return new NotificationModel({
                contentHtml: AJS.I18n.getText(
                    'upm.applications.notifications.unlicensed',
                    this.getName(),
                    'trial-notification-action',
                    'buy-notification-action'
                ),
                type: 'unlicensed-notification'
            });
        },

        /**
         * Unlicensed Notifications for Data Center instances: https://ecosystem.atlassian.net/browse/UPM-5027
         */
        _createUnlicensedDataCenterNotification: function() {
            return new NotificationModel({
                contentHtml: AJS.I18n.getText(
                    'upm.applications.notifications.unlicensed.datacenter',
                    this.getName(),
                    AJS.I18n.getText('upm.applications.notifications.datacenter.suffix')
                ),
                type: 'unlicensed-notification-datacenter'
            });
        },
        
        /**
         * Evaluation Notifications: https://extranet.atlassian.com/display/JIRADEV/Purchase+product+from+eval+mode
         * Data Center variant: https://ecosystem.atlassian.net/browse/UPM-5027
         */
        _createEvaluationBuyNotification: function () {
            var license = this.get('license'),
                message,
                dataCenterSuffix = '',
                buttons =
                    [
                        {
                            'text': AJS.I18n.getText('upm.applications.notifications.actions.buy'),
                            'class': 'buy-notification-action'
                        }
                    ];
            
            if (this.isDataCenter()) {
                dataCenterSuffix = AJS.I18n.getText('upm.applications.notifications.datacenter.suffix');
                buttons = [];
            }

            switch (true) {
                case license.isExpiringTomorrow():
                    message = AJS.I18n.getText('upm.applications.notifications.evaluation.countdown.single', this.getName(), dataCenterSuffix);
                    break;

                case license.isExpiringToday():
                    message = AJS.I18n.getText('upm.applications.notifications.evaluation.countdown.today', this.getName(), dataCenterSuffix);
                    break;

                case license.isExpired():
                    message = AJS.I18n.getText('upm.applications.notifications.evaluation.expired', this.getName(), dataCenterSuffix);
                    break;

                default:
                    message = AJS.I18n.getText('upm.applications.notifications.evaluation.countdown', this.getName(), license.timeUntilExpiry('days'), dataCenterSuffix);
            }
            
            return new NotificationModel({
                contentHtml: message,
                warning: license.isExpiringThisWeek(),
                error: license.isExpired(),
                buttons: buttons,
                type: 'evaluation-notification'
            });
        },

        /**
         * Renewal Notifications: https://extranet.atlassian.com/display/JIRADEV/Product+license+renewal+is+due
         */
        _createRenewNotification: function () {
            var license = this.get('license'),
                note = new NotificationModel({
                    type: 'renew-notification'
                });
            if (this.isDataCenter()) {
                note.set({
                    contentHtml: (license.isExpired() ?
                        AJS.I18n.getText('upm.applications.notifications.renew.expired.datacenter', this.getName(), AJS.I18n.getText('upm.applications.notifications.datacenter.suffix')) :
                        AJS.I18n.getText('upm.applications.notifications.renew.countdown.datacenter', this.get('license').get('expiryDateString'), this.getName(), AJS.I18n.getText('upm.applications.notifications.datacenter.suffix'))),
                    warning: license.isExpiringThisWeek(),
                    error: license.isExpired()
                });

            } else {
                note.set({
                    contentHtml: (license.isExpired() ?
                        AJS.I18n.getText('upm.applications.notifications.renew.expired', this.getName()) :
                        AJS.I18n.getText('upm.applications.notifications.renew.countdown', this.getName(), this.get('license').get('expiryDateString'))),
                    warning: license.isExpiringThisWeek() || license.isExpired(),
                    buttons: [
                        {
                            'text': AJS.I18n.getText('upm.applications.notifications.actions.renew'),
                            'class': 'renew-notification-action'
                        }
                    ]
                });
            }

            return note;
        },

        /**
         * Renew to Update Notification. Only applicable for when a perpetually licensed application has an available
         * update AND expired maintenance that will need to be purchased before they can update. This will prevent
         * perpetually licensed instances from directly installing updates their license will not support -- resulting
         * in a borked instance
         */
        _createRenewToUpdateNotification: function () {
            var update = this.getUpdate();
            return new NotificationModel({
                contentHtml: AJS.I18n.getText('upm.applications.notifications.renew.to.update', this.getName(), update.updateVersion),
                warning: true,
                error: false,
                buttons: [
                    {
                        'text': AJS.I18n.getText('upm.applications.notifications.actions.renew'),
                        'class': 'renew-notification-action'
                    }
                ],
                type: 'renew-to-update-notification'
            });
        },

        /**
         * Upgrade Notifications:
         * - https://extranet.atlassian.com/display/JIRADEV/Upgrade+user+tier+-+Future+user+journey
         * - https://extranet.atlassian.com/display/JIRADEV/Exceeded+user+limit
         */
        _createUpgradeNotification: function () {
            var access = this.get('accessDetails'),
                uri = access.get('managementUri'),
                message,
                note = new NotificationModel({
                    warning: !access.hasExceededLimit(),
                    error: access.hasExceededLimit(),
                    type: 'upgrade-notification'
                });

            switch (true) {
                case access.hasOneUserLeft():
                    message = AJS.I18n.getText('upm.applications.notifications.upgrade.last');
                    break;

                case access.hasNoUsersLeft():
                    message = AJS.I18n.getText('upm.applications.notifications.upgrade.limit');
                    break;

                case access.hasExceededLimit():
                    message = AJS.I18n.getText('upm.applications.notifications.upgrade.exceeded');
                    break;

                default:
                    message = AJS.I18n.getText('upm.applications.notifications.upgrade.nearing', access.remainingUsers());
            }

            if (this.isDataCenter()) {
                note.set({
                    contentHtml: [ message, AJS.I18n.getText('upm.applications.notifications.upgrade.datacenter', uri)].join(' ')
                });
            } else {
                note.set({
                    contentHtml: [ message, AJS.I18n.getText('upm.applications.notifications.upgrade.manage.users', uri)].join(' '),
                    buttons: [
                        {
                            'text': AJS.I18n.getText('upm.applications.notifications.actions.upgrade'),
                            'class': 'upgrade-notification-action'
                        }
                    ]
                });
            }

            return note;
        },

        _createUpdateNotification: function () {
            var update = this.getUpdate(),
                canUpdateDirectly = !!this.getBinaryUpdateUri(),
                message = AJS.I18n.getText('upm.applications.notifications.update', this.getName(), update.updateVersion);

            if (this.getUpdateReleaseNotes()) {
                message = AJS.I18n.getText('upm.applications.notifications.update.releasenotes', this.getName(), update.updateVersion, this.getUpdateReleaseNotes());
            }

            return new NotificationModel({
                contentHtml: message,
                error: false,
                warning: false,
                buttons: [
                    {
                        'text': canUpdateDirectly ? AJS.I18n.getText('upm.applications.notifications.actions.update') :
                            AJS.I18n.getText('upm.applications.notifications.actions.download'),
                        'class': canUpdateDirectly ? 'update-notification-action' : 'download-platform-notification-action'
                    }
                ],
                type: 'update-notification'
            });
        },

        _createIncompatibilityNotification: function () {
            if (this.getVersionInfo().requiresPlatformUpdate()) {
                return new NotificationModel({
                    contentHtml: AJS.I18n.getText('upm.applications.notifications.requires.core.update', ManageApplicationsEnvironment.getPlatformApplicationName(), this.getName()),
                    warning: false,
                    error: true,
                    buttons: [],
                    type: 'incompatible-notification'
                });
            }

            var update = this.getUpdate(),
                canUpdateDirectly = !!this.getBinaryUpdateUri(),
                licenseAllowsUpdate = this.isLicensed() ? moment(update.releaseDate).isBefore(this.getLicense().getExpiryDate()) : true,
                buttons = [
                    {
                        'text': AJS.I18n.getText('upm.applications.notifications.actions.update'),
                        'class': 'update-notification-action'
                    }
                ];

            return new NotificationModel({
                contentHtml: AJS.I18n.getText('upm.applications.notifications.requires.update', this.getName(), update.updateVersion, ManageApplicationsEnvironment.getPlatformApplicationName()),
                error: true,
                warning: false,
                buttons: licenseAllowsUpdate && canUpdateDirectly ? buttons : [],
                type: 'incompatible-notification'
            });
        },

        _createLicensedButNotInstalledNotification: function () {
            return new NotificationModel({
                contentHtml: AJS.I18n.getText('upm.applications.notifications.uninstall.and.licensed', this.getName()),
                error: false,
                warning: true,
                buttons: [
                    {
                        'text': (this.isInstallable() ?
                            AJS.I18n.getText('upm.applications.notifications.actions.install') :
                            AJS.I18n.getText('upm.applications.notifications.actions.download')
                        ),
                        'class': 'install-notification-action'
                    }
                ],
                type: 'licensed-but-not-installed-notification'
            });
        }
    });
});
