/* jshint maxlen: 120 */
/* globals filesString, AJS*/

'use strict';

var webdriverio = require('webdriverio');
var chai = require('chai');
var chaiAsPromised = require('chai-as-promised');

var ExamplePage = require('./page_objects/example_page.js');

chai.should();
chai.use(chaiAsPromised);

var windowSize = {
  width: 1280,
  height: 960
};
var id = 0;
var options = {
  desiredCapabilities: {
    browserName: 'firefox'
  }
};
if (process.env.WEBDRIVER_IO_FIREFOX_BINARY) {
  options.desiredCapabilities.firefox_binary = process.env.WEBDRIVER_IO_FIREFOX_BINARY;
  process.stderr.write("WebdriverIO is using firefox binary at " + options.desiredCapabilities.firefox_binary);
}
var client = webdriverio.remote(options);

var _showToolbar = function (cb) {
  this.elements('#cp-file-body')
    .then(function (el) {
      return this.moveTo(el.value[0]['ELEMENT']);
    }.bind(this))
    .pause(600)
    .elements('.cp-toolbar')
    .then(function (el) {
      cb();
    });
};

function guardAssertions(done, func) {
  return function () {
    try {
      func.apply(this, arguments);
      done();
    } catch (e) {
      client.log('browser', function (err, messages) {
        messages = messages.value.filter(function (m) {
          return m.level === 'SEVERE';
        });
        process.stderr.write(JSON.stringify(messages, null, 4));
        id += 1;
        var name = 'testassets/image-' + id + '.png';
        process.stderr.write('saving image as ' + name + '\n');
        client.saveScreenshot(name, function (err) {
          done(e);
        });
      });
    }
  };
}

before(function () {
  client.init();
  client.setViewportSize(windowSize);
});

after(function () {
  return client.end();
});

describe('FileViewer', function () {

  var examplePage;

  before(function () {
    examplePage = new ExamplePage(client);
    return examplePage.open();
  });

  describe('Watermark Image', function () {

    beforeEach(function () {
      return examplePage.showImage();
    });

    it('should show when the FileViewer is opened', function (done) {
      return client
      .waitForExist('#cp-file-body')
      .pause(500)
      .isExisting('.cp-watermark-layer')
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

  describe('Image viewer', function () {

    beforeEach(function () {
      return examplePage.showImage();
    });

    it('is opened when an image url is clicked', function (done) {
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });

    it('is closed by pressing ESC key', function (done) {
      examplePage.fileViewer.escape();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('is closed by clicking the close button', function (done) {
      examplePage.fileViewer.clickCloseButton();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('is closed by clicking the background', function (done) {
      examplePage.fileViewer.clickBackground();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

  describe('Document viewer', function () {

    beforeEach(function () {
      return examplePage.showDocument().pause(1000);
    });

    it('is opened when a document url is clicked', function (done) {
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });

    it('is closed by pressing ESC key', function (done) {
      examplePage.fileViewer.escape();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('is closed by clicking the close button', function (done) {
      examplePage.fileViewer.clickCloseButton();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('is closed by clicking the background', function (done) {
      examplePage.fileViewer.clickBackground();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

  describe.skip('Video viewer', function () {

    beforeEach(function () {
      return examplePage.showVideoWebM();
    });

    it('is opened when a video url is clicked', function (done) {
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });

    it('is closed by pressing ESC key', function (done) {
      return examplePage.fileViewer.escape()
      .then(function () {
        return examplePage.fileViewer.exists();
      })
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('is closed by clicking the close button', function (done) {
      return examplePage.fileViewer.clickCloseButton()
      .then(function () {
        return examplePage.fileViewer.exists();
      })
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('is closed by clicking the background', function (done) {
      return examplePage.fileViewer.clickBackground()
      .then(function () {
        return examplePage.fileViewer.exists();
      })
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

  describe.skip('Audio viewer', function () {

    beforeEach(function () {
      return examplePage.showAudio();
    });

    it('is opened when an audio url is clicked', function (done) {
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });

    it('is closed by pressing ESC key', function (done) {
      examplePage.fileViewer.escape();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('is closed by clicking the close button', function (done) {
      examplePage.fileViewer.clickCloseButton();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('is closed by clicking the background', function (done) {
      examplePage.fileViewer.clickBackground();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

});

describe('Password protected PDFs', function () {
  var examplePage;

  before(function () {
    examplePage = new ExamplePage(client);
    examplePage.open();
    examplePage.showDocumentPassword();
  });

  it('show the password input dialog', function (done) {
    return client.waitForExist('#cp-preview-password', function () {
      return client.isExisting('#cp-preview-password')
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });
  });
});

describe('Error messages', function () {

  var examplePage;

  before(function () {
    examplePage = new ExamplePage(client);
    return examplePage.open();
  });

  describe('Image errors', function () {

    beforeEach(function () {
      return examplePage.showImage404();
    });

    it('are shown when invalid image url is clicked', function (done) {
      return examplePage.fileViewer.hasErrorMessage()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });

    it('are closed by pressing ESC key', function (done) {
      examplePage.fileViewer.escape();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('are closed by clicking the close button', function (done) {
      examplePage.fileViewer.clickCloseButton();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('are closed by clicking the background', function (done) {
      examplePage.fileViewer.clickBackground();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

  describe('Document errors', function () {

    beforeEach(function () {
      return examplePage.showDocument404();
    });

    it('are shown when invalid document url is clicked', function (done) {
      return client.waitForExist('.cp-error-layer', function () {
        examplePage.fileViewer.hasErrorMessage()
        .then(guardAssertions(done, function (result) {
          return result.should.equal(true);
        }));
      });
    });

    it('are closed by pressing ESC key', function (done) {
      return client.waitForExist('.cp-error-layer', function () {
        examplePage.fileViewer.escape();
        return examplePage.fileViewer.exists()
        .then(guardAssertions(done, function (result) {
          return result.should.equal(false);
        }));
      });
    });

    it('are closed by clicking the close button', function (done) {
      return client.waitForExist('.cp-error-layer', function () {
        examplePage.fileViewer.clickCloseButton();
        return examplePage.fileViewer.exists()
        .then(guardAssertions(done, function (result) {
          return result.should.equal(false);
        }));
      });
    });

    it('are closed by clicking the background', function (done) {
      return client.waitForExist('.cp-error-layer', function () {
        examplePage.fileViewer.clickBackground();
        return examplePage.fileViewer.exists()
        .then(guardAssertions(done, function (result) {
          return result.should.equal(false);
        }));
      });
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

  describe('PDFJS Error Messages', function () {

    var i18nErrorMessages = [];

    before(function () {
      examplePage = new ExamplePage(client);
      examplePage.open();
      return client.execute(function () {
        return {
          'general': AJS.I18n.keys['cp.error.pdf.missing.header'].replace("''", "'"),
          'invalid': AJS.I18n.keys['cp.error.pdf.invalid.header'].replace("''", "'"),
          'password': AJS.I18n.keys['cp.error.pdf.password.header'].replace("''", "'")
        };
      }, function(err, ret) {
        i18nErrorMessages = ret.value;
      });
    });

    it('show correct text for missing files', function (done) {
      examplePage.showDocument404();
      return client.waitForExist('.cp-error-layer')
      .getText('#cp-error-message .title')
      .then(function (result) {
        return result;
      })
      .then(guardAssertions(done, function (result) {
        return result.should.equal(i18nErrorMessages.general);
      }));
    });

    // @TODO: Find test file for this case
    it.skip('show correct text for invalid/corrupt files', function (done) {
      examplePage.showDocumentInvalid();
      return client.waitForExist('.cp-error-layer')
      .getText('#cp-error-message')
      .then(function (result) {
        return result;
      })
      .then(guardAssertions(done, function (result) {
        return result.should.equal(i18nErrorMessages.invalid);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

  describe.skip('Video errors', function () {

    beforeEach(function () {
      return examplePage.showVideo404();
    });

    it('are shown when invalid video url is clicked', function (done) {
      return examplePage.fileViewer.hasErrorMessage()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });

    it('are closed by pressing ESC key', function (done) {
      examplePage.fileViewer.escape();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('are closed by clicking the close button', function (done) {
      examplePage.fileViewer.clickCloseButton();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('are closed by clicking the background', function (done) {
      examplePage.fileViewer.clickBackground();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

  describe.skip('Audio errors', function () {

    beforeEach(function () {
      return examplePage.showAudio404();
    });

    it('are shown when invalid audio url is clicked', function (done) {
      return examplePage.fileViewer.hasErrorMessage()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(true);
      }));
    });

    it('are closed by pressing ESC key', function (done) {
      examplePage.fileViewer.escape();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('are closed by clicking the close button', function (done) {
      examplePage.fileViewer.clickCloseButton();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    it('are closed by clicking the background', function (done) {
      examplePage.fileViewer.clickBackground();
      return examplePage.fileViewer.exists()
      .then(guardAssertions(done, function (result) {
        return result.should.equal(false);
      }));
    });

    afterEach(function () {
      return examplePage.fileViewer.escape();
    });

  });

});

describe('Switching files', function () {

  var examplePage, files;

  before(function () {
    examplePage = new ExamplePage(client);
    examplePage.open();
    return client.execute(function () {
      return filesString;
    }, function(err, ret) {
      files = JSON.parse(ret.value);
    });
  });

  it('Next file by pressing arrow right', function (done) {
    examplePage.showImage();
    examplePage.fileViewer.arrowRight();
    return client.getText('#cp-title-container')
    .then(guardAssertions(done, function (result) {
      return result.should.equal(files[1].title);
    }));
  });

  it('Next file by clicking right arrow button', function (done) {
    client.click('#cp-nav-right');
    return client.getText('#cp-title-container')
    .then(guardAssertions(done, function (result) {
      return result.should.equal(files[2].title);
    }));
  });

  it('Previous file by pressing arrow left', function (done) {
    examplePage.fileViewer.arrowLeft();
    return client.getText('#cp-title-container')
    .then(guardAssertions(done, function (result) {
      return result.should.equal(files[1].title);
    }));
  });

  it('Previous file by clicking left arrow button', function (done) {
    client.click('#cp-nav-left');
    return client.getText('#cp-title-container')
    .then(guardAssertions(done, function (result) {
      return result.should.equal(files[0].title);
    }));
  });

});

describe('Minimode', function () {

  var examplePage, files;

  before(function () {
    examplePage = new ExamplePage(client);
    examplePage.open();
    return client.execute(function () {
      return filesString;
    }, function(err, ret) {
      files = JSON.parse(ret.value);
    });
  });

  it('Click "Show all files" should open mini-mode', function (done) {
    examplePage.showImage();
    examplePage.fileViewer.toggleMiniMode();
    return examplePage.fileViewer.hasOpenMinimode()
    .then(guardAssertions(done, function (result) {
      return result.should.equal(true);
    }));
  });

  it('Opening mini-mode should focus the thumbnail of the current file', function (done) {
    return client.getAttribute('#cp-thumbnails li', 'class')
    .then(guardAssertions(done, function (result) {
      return result[0].indexOf('selected').should.not.equal(-1);
    }));
  });

  it('Clicking on a thumbnail should switch to the corresponding file', function (done) {
    return client.elements('#cp-thumbnails li', function (err, elements) {
      client.elementIdClick(elements.value[1]['ELEMENT'], function (err, res) {
        client.getText('#cp-title-container')
        .then(guardAssertions(done, function (result) {
          result.should.equal(files[1].title);
        }));
      });
    });
  });

  it('Switching the file should set focus to the corresponding thumbnail', function (done) {
    client
    .click('#cp-nav-left')
    .pause(1500);
    return client.getText('#cp-title-container')
    .then(guardAssertions(done, function (result) {
      result.should.equal(files[0].title);
    }));
  });

  it('Clicking "Show all files" again should close mini-mode', function (done) {
    examplePage.fileViewer.toggleMiniMode();
    return examplePage.fileViewer.hasOpenMinimode()
    .then(guardAssertions(done, function (result) {
      return result.should.equal(false);
    }));
  });

});

describe('Toolbar', function () {

  var examplePage;

  before(function () {

    client.addCommand('showToolbar', _showToolbar);

    examplePage = new ExamplePage(client);
    examplePage.open();
    return examplePage.showImage();
  });

  it('Moving the mouse should show the toolbar', function (done) {

    return client
    .waitForExist('#cp-img')
    .pause(500)
    .showToolbar()
    .isVisible('.cp-toolbar')
    .then(guardAssertions(done, function (result) {
      return result.should.equal(true);
    }));
  });

   it('Zooming should show zoom scale info overlay', function (done) {

     return client
     .showToolbar()
     .click('.cp-toolbar-minus')
     .isVisible('.cp-scale-info')
     .then(guardAssertions(done, function (result) {
       return result.should.equal(true);
     }));
   });

  it('Clicking "Zoom in" button should increase the scale', function (done) {
    var sizeBefore, sizeAfter;

    return client
    .getElementSize('#cp-img')
    .then(function (size) {
      sizeBefore = size;
    })
    .pause(500)
    .showToolbar()
    .click('.cp-toolbar-plus')
    .getElementSize('#cp-img')
    .then(guardAssertions(done, function (size) {
      sizeAfter = size;
      return sizeAfter.width.should.be.above(sizeBefore.width);
    }));
  });

  it('Clicking "Zoom out" button should decrease the scale', function (done) {
    var sizeBefore, sizeAfter;

    return client
    .getElementSize('#cp-img')
    .then(function (size) {
      sizeBefore = size;
    })
    .pause(500)
    .showToolbar()
    .click('.cp-toolbar-minus')
    .getElementSize('#cp-img')
    .then(guardAssertions(done, function (size) {
      sizeAfter = size;
      return sizeAfter.width.should.be.below(sizeBefore.width);
    }));
  });

  it('Clicking "Fit page" button should fit the page', function (done) {
    var sizeContainer, sizeImage;

    return client
    .setViewportSize({
        width: 350,
        height: 500
    })
    .getElementSize('#cp-image-preview')
    .then(function (size) {
      sizeContainer = size;
    })
    .pause(500)
    .showToolbar()
    .click('.cp-toolbar-fit')
    .getElementSize('#cp-img')
    .then(guardAssertions(done, function (size) {
      sizeImage = size;
      client.setViewportSize(windowSize, function () {
        return sizeContainer.width.should.equal(sizeImage.width);
      });
    }));
  });

  after(function () {
    client.setViewportSize(windowSize);
  });

});

describe.skip('Video/Audio Player', function () {

  var examplePage;

  before(function () {
    examplePage = new ExamplePage(client);
    examplePage.open();
    return examplePage.showVideoWebM();
  });

  it('Open a video file should automatically start playing', function (done) {
    return client
    .waitForExist('.vjs-waiting', 10000, true)
    .getAttribute('.video-js', 'class')
    .then(guardAssertions(done, function (result) {
      var index = result.indexOf('vjs-playing');
      return index.should.be.above(-1);
    }));
  });

  it('If the user is idle the controlbar should fade out', function (done) {
    return client
    .pause(1500)
    .getCssProperty('.vjs-control-bar', 'opacity')
    .then(guardAssertions(done, function (result) {
      return result.value.should.equal(0);
    }));
  });

  it('Moving the mouse should fade the controlbar back in', function (done) {
    return client
    .elements('#cp-header')
    .elements('.vjs-progress-holder')
    .then(function (el) {
      return client.moveTo(el.value[0]['ELEMENT']);
    })
    .elements('#cp-video-player')
    .then(function (el) {
      return client.moveTo(el.value[0]['ELEMENT']);
    })
    .pause(500)
    .getCssProperty('.vjs-control-bar', 'opacity')
    .then(guardAssertions(done, function (result) {
      return result.value.should.equal(1);
    }));
  });

  it('Hovering over the progressbar should display a progress tooltip', function (done) {
    client
    .elements('.vjs-progress-holder')
    .then(function (el) {
      return client.moveTo(el.value[0]['ELEMENT']);
    })
    .pause(500)
    .getCssProperty('#vjs-tip', 'display')
    .then(guardAssertions(done, function (result) {
      result.value.should.equal('block');
    }));
  });

});

describe('Presentation mode', function () {

  var examplePage, files, viewportSize;

  before(function () {
    client.addCommand('showToolbar', _showToolbar);

    examplePage = new ExamplePage(client);
    examplePage.open();

    client.execute(function () {
      return filesString;
    }, function (err, ret) {
      files = JSON.parse(ret.value);
    });

    client.getViewportSize(function (err, size) {
      viewportSize = size;
    });

    return examplePage.showImage();
  });

  describe('Enter Presentation', function () {

    it('Clicking Start Presentation should expand file body', function (done) {
      client
        .waitForExist('#cp-img')
        .pause(750)
        .showToolbar()
        .click(".cp-toolbar-presentation")
        .pause(750)
        .getElementSize('#cp-file-body', function (err, size) {
          size.width.should.equal(viewportSize.width);
          size.height.should.equal(viewportSize.height);
          done();
        });
    });

    it.skip('image height should be equal to view port height', function () {
      return client
        .getElementSize('#cp-img', function (err, size) {
          return size.height.should.equal(viewportSize.height);
        });
    });

    it('should show Exit Presentation button in toolbar', function () {
      return client
        .pause(500)
        .showToolbar()
        .isVisible('.cp-toolbar-presentation-exit')
        .then(function (result) {
          return result.should.equal(true);
        });
    });

    it('should not show Next/Previous arrow buttons', function () {
      return client
        .isVisible('#cp-nav-right')
        .then(function (result) {
          return result.should.equal(false);
        })
        .isVisible('#cp-nav-left')
        .then(function (result) {
          return result.should.equal(false);
        });
    });

    it('Clicking the background should not close file viewer', function () {
      examplePage.fileViewer.clickBackground();
      return examplePage.fileViewer.exists()
      .then(function (result) {
        return result.should.equal(true);
      });
    });

    it('Pressing arrow right should go to next file', function () {
      examplePage.fileViewer.arrowRight();
      return client.pause(500).getText('#cp-title-container')
      .then(function (result) {
        return result.should.equal(files[1].title);
      });
    });

    it('Pressing arrow left should go to previous file', function () {
      examplePage.fileViewer.arrowLeft();
      return client.pause(500).getText('#cp-title-container')
      .then(function (result) {
        return result.should.equal(files[0].title);
      });
    });
  });

  describe('Presentation mode of document', function () {
    before(function () {
      return examplePage.showDocument()
      .pause(1000);
    });

    var _isShowingPage = function (page, cb) {
      client.isVisible('#pageContainer' + page, function (err, isVisible) {
        cb(true);
      });
    };

    it('should show page 1', function (done) {
      _isShowingPage(1, function (result) {
        result.should.equal(true);
        done();
      });
    });

    it('Pressing Down should go to next page', function (done) {
      examplePage.fileViewer.arrowDown();
      _isShowingPage(2, function (result) {
        result.should.equal(true);
        done();
      });
    });

    it('Pressing Up should go to previous page', function (done) {
      examplePage.fileViewer.arrowUp();
      _isShowingPage(1, function (result) {
        result.should.equal(true);
        done();
      });
    });

    it('Clicking Next page button should go to next page', function (done) {
      client.pause(500).showToolbar().pause(500);
      client.click('.cp-toolbar-next-page', function (err) {
        _isShowingPage(2, function (result) {
          result.should.equal(true);
          done();
        });
      });
    });

    it('Clicking Previous page button should go to previous page', function (done) {
      client.pause(500).showToolbar().pause(500);
      client.click('.cp-toolbar-next-page', function (err) {
        _isShowingPage(1, function (result) {
          result.should.equal(true);
          done();
        });
      });
    });
  });

  describe('Exit Presentation Mode', function () {
    it('Clicking Exit Presentation should collapse file body', function () {
      return client
        .pause(500)
        .showToolbar()
        .click(".cp-toolbar-presentation-exit")
        .getElementSize('#cp-file-body', function (err, size) {
          return size.height.should.below(viewportSize.height);
        });
    });

    it('should show Next/Previous arrow buttons', function () {
      return client
        .isVisible('#cp-nav-right')
        .then(function (result) {
          return result.should.equal(true);
        })
        .isVisible('#cp-nav-left')
        .then(function (result) {
          return result.should.equal(true);
        });
    });

    it('Clicking the background should close file viewer', function () {
      examplePage.fileViewer.clickBackground();
      return examplePage.fileViewer.exists()
      .then(function (result) {
        return result.should.equal(false);
      })
    });
  });

  describe('Unsupported file', function () {
    it("should not have Start Presentation button", function () {
      examplePage.showUnsupportedFile();
      return client
        .pause(500)
        .showToolbar()
        .pause(500)
        .isVisible('.cp-toolbar')
        .then(function (result) {
          return result.should.equal(false);
        });
    });
  });
});
