/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/text-layer-builder',
  [
    'pdf-viewer/viewer-properties'
  ],
  function (
    viewerProperties
    ) {

    var PDFJS = window.PDFJS;
    var CustomStyle = PDFJS.CustomStyle;

    /**
     * TextLayerBuilder provides text-selection functionality for the PDF.
     * It does this by creating overlay divs over the PDF text. These divs
     * contain text that matches the PDF text they are overlaying. This object
     * also provides a way to highlight text that is being searched for.
     */
    function TextLayerBuilder (options) {
      this.textLayerDiv = options.textLayerDiv;
      this.layoutDone = false;
      this.divContentDone = false;
      this.pageIdx = options.pageIndex;
      this.matches = [];
      this.lastScrollSource = options.lastScrollSource || null;
      this.viewport = options.viewport;
      this.isViewerInPresentationMode = options.isViewerInPresentationMode;
      this.textDivs = [];
      this.findController = window.PDFFindController || null;
    }

    TextLayerBuilder.prototype = {
      renderLayer: function TextLayerBuilder_renderLayer () {
        var textLayerFrag = document.createDocumentFragment();
        var textDivs = this.textDivs;
        var textDivsLength = textDivs.length;
        var canvas = document.createElement('canvas');
        var ctx = canvas.getContext('2d');

        // No point in rendering many divs as it would make the browser
        // unusable even after the divs are rendered.
        if (textDivsLength > viewerProperties.MAX_TEXT_DIVS_TO_RENDER) {
          return;
        }

        for (var i = 0; i < textDivsLength; i++) {
          var textDiv = textDivs[i];
          if (textDiv.dataset.isWhitespace !== undefined) {
            continue;
          }

          ctx.font = textDiv.style.fontSize + ' ' + textDiv.style.fontFamily;
          var width = ctx.measureText(textDiv.textContent).width;
          if (width > 0) {
            textLayerFrag.appendChild(textDiv);
            var textScale = textDiv.dataset.canvasWidth / width;
            var rotation = textDiv.dataset.angle;
            var transform = 'scale(' + textScale + ', 1)';
            transform = 'rotate(' + rotation + 'deg) ' + transform;
            CustomStyle.setProp('transform', textDiv, transform);
            CustomStyle.setProp('transformOrigin', textDiv, '0% 0%');
          }
        }

        this.textLayerDiv.appendChild(textLayerFrag);
        this.renderingDone = true;
      },

      setupRenderLayoutTimer:
        function TextLayerBuilder_setupRenderLayoutTimer () {
          // Schedule renderLayout() if the user has been scrolling,
          // otherwise run it right away.
          var self = this;
          var lastScroll = (this.lastScrollSource === null ?
            0 : this.lastScrollSource.lastScroll);

          if (Date.now() - lastScroll > viewerProperties.RENDER_DELAY) { // Render right away
            this.renderLayer();
          } else { // Schedule
            if (this.renderTimer) {
              clearTimeout(this.renderTimer);
            }
            this.renderTimer = setTimeout(function () {
              self.setupRenderLayoutTimer();
            }, viewerProperties.RENDER_DELAY);
          }
        },

      appendText: function TextLayerBuilder_appendText (geom, styles) {
        var style = styles[geom.fontName];
        var textDiv = document.createElement('div');
        this.textDivs.push(textDiv);
        if (!/\S/.test(geom.str)) {
          textDiv.dataset.isWhitespace = true;
          return;
        }
        var tx = PDFJS.Util.transform(this.viewport.transform, geom.transform);
        var angle = Math.atan2(tx[1], tx[0]);
        if (style.vertical) {
          angle += Math.PI / 2;
        }
        var fontHeight = Math.sqrt((tx[2] * tx[2]) + (tx[3] * tx[3]));
        var fontAscent = (style.ascent ? style.ascent * fontHeight :
          (style.descent ? (1 + style.descent) * fontHeight : fontHeight));

        textDiv.style.position = 'absolute';
        textDiv.style.left = (tx[4] + (fontAscent * Math.sin(angle))) + 'px';
        textDiv.style.top = (tx[5] - (fontAscent * Math.cos(angle))) + 'px';
        textDiv.style.fontSize = fontHeight + 'px';
        textDiv.style.fontFamily = style.fontFamily;

        textDiv.textContent = geom.str;
        textDiv.dataset.fontName = geom.fontName;
        textDiv.dataset.angle = angle * (180 / Math.PI);
        if (style.vertical) {
          textDiv.dataset.canvasWidth = geom.height * this.viewport.scale;
        } else {
          textDiv.dataset.canvasWidth = geom.width * this.viewport.scale;
        }
      },

      setTextContent: function TextLayerBuilder_setTextContent (textContent) {
        this.textContent = textContent;

        var textItems = textContent.items;
        for (var i = 0, len = textItems.length; i < len; i++) {
          this.appendText(textItems[i], textContent.styles);
        }
        this.divContentDone = true;
        this.setupRenderLayoutTimer();
      }

    };
    return TextLayerBuilder;

  }
);
