/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/page-view',
  [
    'pdf-viewer/utils',
    'pdf-viewer/viewer-properties',
    'pdf-viewer/rendering-states',
    'pdf-viewer/presentation-mode',
    'pdf-viewer/text-layer-builder',
    'pdf-viewer/cache'
  ],
  function (
    viewerUtils,
    viewerProperties,
    renderingStates,
    PresentationMode,
    TextLayerBuilder,
    cache
  ) {

    var PDFJS = window.PDFJS;
    var CustomStyle = PDFJS.CustomStyle;

    /**
     * Scrolls specified element into view of its parent.
     * element {Object} The element to be visible.
     * containerEl {Object} The elements scroll parent.
     * spot {Object} An object with optional top and left properties,
     *         specifying the offset from the top left edge.
     */
    function scrollIntoView (element, containerEl, spot) {
      // Assuming offsetParent is available (it's not available when viewer is in
      // hidden iframe or object). We have to scroll: if the offsetParent is not set
      // producing the error. See also animationStartedClosure.
      var parent = containerEl;
      var offsetY = element.offsetTop + element.clientTop;
      var offsetX = element.offsetLeft + element.clientLeft;
      if (!parent) {
        console.error('container element is not set -- cannot scroll');
        return;
      }

      while (parent.clientHeight === parent.scrollHeight) {
        if (parent.dataset._scaleY) {
          offsetY /= parent.dataset._scaleY;
          offsetX /= parent.dataset._scaleX;
        }
        offsetY += parent.offsetTop;
        offsetX += parent.offsetLeft;
        parent = parent.offsetParent;
        if (!parent) {
          return; // no need to scroll
        }
      }
      if (spot) {
        if (spot.top !== undefined) {
          offsetY += spot.top;
        }
        if (spot.left !== undefined) {
          offsetX += spot.left;
          parent.scrollLeft = offsetX;
        }
      }
      parent.scrollTop = offsetY;
    }

    var PageView = function pageView (container, id, scale, defaultViewport, parentViewer) {
      this.id = id;

      this.parentViewer = parentViewer;

      this.rotation = 0;
      this.scale = scale || 1.0;
      this.viewport = defaultViewport;
      this.pdfPageRotate = defaultViewport.rotation;

      this.renderingState = renderingStates.INITIAL;
      this.resume = null;

      this.textLayer = null;

      this.layers = [];

      this.zoomLayer = null;

      this.annotationLayer = null; // custom layer for Confluence annotations pins
      this.linkLayer = null; // contains links refered to as AnnotationLayer in PDFJS

      var anchor = document.createElement('a');
      anchor.name = '' + this.id;

      var div = this.el = document.createElement('div');
      div.id = 'pageContainer' + this.id;
      div.setAttribute('data-page-number', this.id);
      div.className = 'page';
      div.style.width = Math.floor(this.viewport.width) + 'px';
      div.style.height = Math.floor(this.viewport.height) + 'px';

      container.appendChild(anchor);
      container.appendChild(div);

      this.setPdfPage = function pageViewSetPdfPage (pdfPage) {
        this.pdfPage = pdfPage;
        this.pdfPageRotate = pdfPage.rotate;
        var totalRotation = (this.rotation + this.pdfPageRotate) % 360;
        this.viewport = pdfPage.getViewport(this.scale * viewerProperties.CSS_UNITS, totalRotation);
        this.stats = pdfPage.stats;
        this.reset();
      };

      this.destroy = function pageViewDestroy () {
        this.zoomLayer = null;
        this.reset();
        if (this.pdfPage) {
          this.pdfPage.destroy();
        }
      };

      this.reset = function pageViewReset (keepAnnotations, keepLinks) {
        if (this.renderTask) {
          this.renderTask.cancel();
        }
        this.resume = null;
        this.renderingState = renderingStates.INITIAL;

        div.style.width = Math.floor(this.viewport.width) + 'px';
        div.style.height = Math.floor(this.viewport.height) + 'px';

        var childNodes = div.childNodes;
        for (var i = div.childNodes.length - 1; i >= 0; i--) {
          var node = childNodes[i];
          if ((this.zoomLayer && this.zoomLayer === node) ||
            (keepAnnotations && this.annotationLayer === node) ||
            (keepLinks && this.linkLayer === node)) {
            continue;
          }
          div.removeChild(node);
        }
        div.removeAttribute('data-loaded');

        if (keepAnnotations) {
          if (this.annotationLayer) {
            // Hide annotationLayer until all elements are resized
            // so they are not displayed on the already-resized page
            this.annotationLayer.setAttribute('hidden', 'true');
          }
        } else {
          this.annotationLayer = null;
        }

        if (!keepLinks) {
          this.linkLayer = null;
        }

        delete this.canvas;
        this.loadingIconDiv = document.createElement('div');
        this.loadingIconDiv.className = 'loadingIcon';
        div.appendChild(this.loadingIconDiv);
      };

      this.update = function pageViewUpdate (scale, rotation) {
        this.scale = scale || this.scale;

        if (typeof rotation !== 'undefined') {
          this.rotation = rotation;
        }

        var totalRotation = (this.rotation + this.pdfPageRotate) % 360;
        this.viewport = this.viewport.clone({
          scale: this.scale * viewerProperties.CSS_UNITS,
          rotation: totalRotation
        });

        if (viewerProperties.USE_ONLY_CSS_ZOOM && this.canvas) {
          this.cssTransform(this.canvas);
          return;
        } else if (this.canvas && !this.zoomLayer) {
          this.zoomLayer = this.canvas.parentNode;
          this.zoomLayer.style.position = 'absolute';
        }
        if (this.zoomLayer) {
          this.cssTransform(this.zoomLayer.firstChild);
        }
        this.reset(true, true);
      };

      this.cssTransform = function pageCssTransform (canvas) {
        // Scale canvas, canvas wrapper, and page container.
        var width = this.viewport.width;
        var height = this.viewport.height;
        canvas.style.width = canvas.parentNode.style.width = div.style.width =
          Math.floor(width) + 'px';
        canvas.style.height = canvas.parentNode.style.height = div.style.height =
          Math.floor(height) + 'px';
        // The canvas may have been originally rotated, so rotate relative to that.
        var relativeRotation = this.viewport.rotation - canvas._viewport.rotation;
        var absRotation = Math.abs(relativeRotation);
        var scaleX = 1, scaleY = 1;
        if (absRotation === 90 || absRotation === 270) {
          // Scale x and y because of the rotation.
          scaleX = height / width;
          scaleY = width / height;
        }
        var cssTransform = 'rotate(' + relativeRotation + 'deg) ' +
          'scale(' + scaleX + ',' + scaleY + ')';
        CustomStyle.setProp('transform', canvas, cssTransform);

        if (this.textLayer) {
          // Rotating the text layer is more complicated since the divs inside the
          // the text layer are rotated.
          // TODO: This could probably be simplified by drawing the text layer in
          // one orientation then rotating overall.
          var textRelativeRotation = this.viewport.rotation -
            this.textLayer.viewport.rotation;
          var textAbsRotation = Math.abs(textRelativeRotation);
          var scale = (width / canvas.width);
          if (textAbsRotation === 90 || textAbsRotation === 270) {
            scale = width / canvas.height;
          }
          var textLayerDiv = this.textLayer.textLayerDiv;
          var transX, transY;
          switch (textAbsRotation) {
            case 0:
              transX = transY = 0;
              break;
            case 90:
              transX = 0;
              transY = '-' + textLayerDiv.style.height;
              break;
            case 180:
              transX = '-' + textLayerDiv.style.width;
              transY = '-' + textLayerDiv.style.height;
              break;
            case 270:
              transX = '-' + textLayerDiv.style.width;
              transY = 0;
              break;
            default:
              console.error('Bad rotation value.');
              break;
          }
          CustomStyle.setProp('transform', textLayerDiv,
            'rotate(' + textAbsRotation + 'deg) ' +
              'scale(' + scale + ', ' + scale + ') ' +
              'translate(' + transX + ', ' + transY + ')');
          CustomStyle.setProp('transformOrigin', textLayerDiv, '0% 0%');
        }

        setupInternalLinks(div, this.pdfPage, this.viewport);
      };

      Object.defineProperty(this, 'width', {
        get: function PageView_getWidth () {
          return this.viewport.width;
        },
        enumerable: true
      });

      Object.defineProperty(this, 'height', {
        get: function PageView_getHeight () {
          return this.viewport.height;
        },
        enumerable: true
      });

      this.getPagePoint = function pageViewGetPagePoint (x, y) {
        return this.viewport.convertToPdfPoint(x, y);
      };

      this.scrollIntoView = function pageViewScrollIntoView (dest) {
        if (PresentationMode.active) {
          if (this.parentViewer.page !== this.id) {
            // Avoid breaking parentViewer.getVisiblePages in presentation mode.
            this.parentViewer.page = this.id;
          }
          dest = null;
          this.parentViewer.setScale(this.parentViewer.currentScaleValue, true, true);
        }
        if (!dest) {
          scrollIntoView(div, this.parentViewer.el.container);
          return;
        }

        var x = 0, y = 0;
        var width = 0, height = 0, widthScale, heightScale;
        var changeOrientation = (this.rotation % 180 === 0 ? false : true);
        var pageWidth = (changeOrientation ? this.height : this.width) /
          this.scale / viewerProperties.CSS_UNITS;
        var pageHeight = (changeOrientation ? this.width : this.height) /
          this.scale / viewerProperties.CSS_UNITS;
        var scale = 0;
        if (!dest[1]) dest[1] = '';
        switch (dest[1].name) {
          case 'XYZ':
            x = dest[2];
            y = dest[3];
            scale = dest[4];
            // If x and/or y coordinates are not supplied, default to
            // _top_ left of the page (not the obvious bottom left,
            // since aligning the bottom of the intended page with the
            // top of the window is rarely helpful).
            x = x !== null ? x : 0;
            y = y !== null ? y : pageHeight;
            break;
          case 'Fit':
          case 'FitB':
            scale = 'page-fit';
            break;
          case 'FitH':
          case 'FitBH':
            y = dest[2];
            scale = 'page-width';
            break;
          case 'FitV':
          case 'FitBV':
            x = dest[2];
            width = pageWidth;
            height = pageHeight;
            scale = 'page-height';
            break;
          case 'FitR':
            x = dest[2];
            y = dest[3];
            width = dest[4] - x;
            height = dest[5] - y;
            widthScale = (this.parentViewer.viewer.clientWidth - viewerProperties.VERTICAL_PADDING) /
              width / viewerProperties.CSS_UNITS;
            heightScale = (this.parentViewer.viewer.clientHeight - viewerProperties.VERTICAL_PADDING) /
              height / viewerProperties.CSS_UNITS;
            scale = Math.min(Math.abs(widthScale), Math.abs(heightScale));
            break;
          default:
            return;
        }

        if (scale && scale !== this.parentViewer.currentScale) {
          this.parentViewer.setScale(scale, true, true);
        } else if (this.parentViewer.currentScale === viewerProperties.UNKNOWN_SCALE) {
          this.parentViewer.setScale(viewerProperties.DEFAULT_SCALE, true, true);
        }

        if (scale === 'page-fit' && !dest[4]) {
          scrollIntoView(div, this.parentViewer.el.container);
          return;
        }

        var boundingRect = [
          this.viewport.convertToViewportPoint(x, y),
          this.viewport.convertToViewportPoint(x + width, y + height)
        ];
        var left = Math.min(boundingRect[0][0], boundingRect[1][0]);
        var top = Math.min(boundingRect[0][1], boundingRect[1][1]);

        scrollIntoView(div, this.parentViewer.el.container, { left: left, top: top });
      };

      this.getTextContent = function pageviewGetTextContent () {
        return this.parentViewer.getPage(this.id).then(function (pdfPage) {
          return pdfPage.getTextContent();
        });
      };

      this.draw = function pageviewDraw (callback) {
        var pdfPage = this.pdfPage;
        // It's important that linkLayer and annotationLayer stay on top
        // of the other layers. In case they don't exist insertBefore with
        // referenceNode = null will append.
        var beforeLayer = this.annotationLayer || this.linkLayer || null;

        if (this.pagePdfPromise) {
          return;
        }
        if (!pdfPage) {
          var promise = this.parentViewer.getPage(this.id);
          promise.then(function (pdfPage) {
            delete this.pagePdfPromise;
            this.setPdfPage(pdfPage);
            this.draw(callback);
          }.bind(this));
          this.pagePdfPromise = promise;
          return;
        }

        if (this.renderingState !== renderingStates.INITIAL) {
          console.error('Must be in new state before drawing');
        }

        this.renderingState = renderingStates.RUNNING;

        var viewport = this.viewport;
        // Wrap the canvas so if it has a css transform for highdpi the overflow
        // will be hidden in FF.
        var canvasWrapper = document.createElement('div');
        canvasWrapper.style.width = div.style.width;
        canvasWrapper.style.height = div.style.height;
        canvasWrapper.classList.add('canvasWrapper');

        var canvas = document.createElement('canvas');
        canvas.id = 'page' + this.id;
        // Keep the canvas hidden until the first draw callback, or until drawing
        // is complete when `!this.renderingQueue`, to prevent black flickering.
        canvas.setAttribute('hidden', 'hidden');
        var isCanvasHidden = true;
        canvasWrapper.appendChild(canvas);

        div.insertBefore(canvasWrapper, beforeLayer);

        this.canvas = canvas;

        canvas.mozOpaque = true;
        var ctx = canvas.getContext('2d', {alpha: false});
        var outputScale = viewerUtils.getOutputScale(ctx);

        if (viewerProperties.USE_ONLY_CSS_ZOOM) {
          var actualSizeViewport = viewport.clone({ scale: viewerProperties.CSS_UNITS });
          // Use a scale that will make the canvas be the original intended size
          // of the page.
          outputScale.sx *= actualSizeViewport.width / viewport.width;
          outputScale.sy *= actualSizeViewport.height / viewport.height;
          outputScale.scaled = true;
        }

        var sfx = viewerUtils.approximateFraction(outputScale.sx);
        var sfy = viewerUtils.approximateFraction(outputScale.sy);
        canvas.width = viewerUtils.roundToDivide(viewport.width * outputScale.sx, sfx[0]);
        canvas.height = viewerUtils.roundToDivide(viewport.height * outputScale.sy, sfy[0]);
        canvas.style.width = viewerUtils.roundToDivide(viewport.width, sfx[1]) + 'px';
        canvas.style.height = viewerUtils.roundToDivide(viewport.height, sfy[1]) + 'px';
        // Add the viewport so it's known what it was originally drawn with.
        canvas._viewport = viewport;

        var textLayerDiv = null;
        if (!PDFJS.disableTextLayer) {
          textLayerDiv = document.createElement('div');
          textLayerDiv.className = 'textLayer';
          textLayerDiv.style.width = canvas.style.width;
          textLayerDiv.style.height = canvas.style.height;

          div.insertBefore(textLayerDiv, beforeLayer);
        }
        var textLayer = this.textLayer =
          textLayerDiv ? new TextLayerBuilder({
            textLayerDiv: textLayerDiv,
            pageIndex: this.id - 1,
            lastScrollSource: parentViewer,
            viewport: this.viewport,
            isViewerInPresentationMode: PresentationMode.active
          }) : null;

        if (!this.linkLayer) {
          // Adding linkLayer before layerBuilders to ensure proper
          // order of the layers.
          var linkLayerDiv = document.createElement('div');
          linkLayerDiv.className = 'linkLayer';
          div.appendChild(linkLayerDiv);
          this.linkLayer = linkLayerDiv;
        }

        // when in presentation mode, we don't want to render any layers (i.e annotation) on top of pdf pages
        if (parentViewer.layerBuilders && !PresentationMode.active) {
          for (var i = 0; i < parentViewer.layerBuilders.length; i++) {
            var layerDiv = document.createElement('div');
            layerDiv.style.width = canvas.style.width;
            layerDiv.style.height = canvas.style.height;

            var LayerBuilder = parentViewer.layerBuilders[i];
            var layer = new LayerBuilder({
              layerDiv: layerDiv,
              pageIndex: this.id - 1,
              lastScrollSource: parentViewer,
              viewport: this.viewport,
              isViewerInPresentationMode: PresentationMode.active
            });
            this.layers.push(layer);
            div.appendChild(layerDiv);
            layer.setupRenderLayoutTimer && layer.setupRenderLayoutTimer();
          }
        }

        // Rendering area

        var self = this;
        function pageViewDrawCallback (error) {
          // The renderTask may have been replaced by a new one, so only remove the
          // reference to the renderTask if it matches the one that is triggering
          // this callback.
          if (renderTask === self.renderTask) {
            self.renderTask = null;
          }

          if (error === 'cancelled') {
            return;
          }

          self.renderingState = renderingStates.FINISHED;

          if (isCanvasHidden) {
            self.canvas.removeAttribute('hidden');
            isCanvasHidden = false;
          }

          if (self.loadingIconDiv) {
            div.removeChild(self.loadingIconDiv);
            delete self.loadingIconDiv;
          }

          if (self.zoomLayer) {
            div.removeChild(self.zoomLayer);
            self.zoomLayer = null;
          }

          // @todo: send event specifying the error

          if (self.onAfterDraw) {
            self.onAfterDraw();
          }

          cache.push(self);

          var event = document.createEvent('CustomEvent');
          event.initCustomEvent('pagerender', true, true, {
            pageNumber: pdfPage.pageNumber
          });
          div.dispatchEvent(event);

          callback();
        }

        var transform = !outputScale.scaled ? null :
          [outputScale.sx, 0, 0, outputScale.sy, 0, 0];

        var renderContext = {
          canvasContext: ctx,
          transform: transform,
          viewport: this.viewport,
          textLayer: textLayer,
          // intent: 'default', // === 'display'
          continueCallback: function pdfViewcContinueCallback (cont) {
            if (self.parentViewer.highestPriorityPage !== 'page' + self.id) {
              self.renderingState = renderingStates.PAUSED;
              self.resume = function resumeCallback () {
                self.renderingState = renderingStates.RUNNING;
                cont();
              };
              return;
            }
            if (isCanvasHidden) {
              self.canvas.removeAttribute('hidden');
              isCanvasHidden = false;
            }
            cont();
          }
        };

        var renderTask = this.renderTask = this.pdfPage.render(renderContext);

        this.renderTask.promise.then(
          function pdfPageRenderCallback () {
            pageViewDrawCallback(null);
            if (textLayer) {
              self.getTextContent().then(
                function textContentResolved (textContent) {
                  textLayer.setTextContent(textContent);
                }
              );
            }
          },
          function pdfPageRenderError (error) {
            pageViewDrawCallback(error);
          }
        );

        setupInternalLinks(div, pdfPage, this.viewport);
        div.setAttribute('data-loaded', true);
      };

      var setupInternalLinks = function setupInternalLinks (pageDiv, pdfPage, viewport) {
        var self = this;
        var pdfViewer = self.parentViewer;

        pdfPage.getAnnotations().then(function (annotationsData) {
          var PDFJS = window.PDFJS;
          viewport = viewport.clone({ dontFlip: true });

          if (self.linkLayer && self.linkLayer.childElementCount > 0) {
            PDFJS.AnnotationLayer.update({
              viewport: viewport,
              div: self.linkLayer,
              page: pdfPage,
              annotations: annotationsData,
              linkService: pdfViewer.linkService
            });
          } else {
            if (annotationsData.length === 0) {
              return;
            }

            if (!self.linkLayer) {
              self.linkLayer = document.createElement('div');
              self.linkLayer.className = 'linkLayer';
              pageDiv.appendChild(self.linkLayer);
            }

            self.linkLayer.innerHTML = '';
            PDFJS.AnnotationLayer.render({
              viewport: viewport,
              div: self.linkLayer,
              page: pdfPage,
              annotations: annotationsData,
              linkService: pdfViewer.linkService
            });
          }
        });
      }.bind(this);
    };

    return PageView;
  });
