/* Copyright 2012 Mozilla Foundation
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Extracted from PDFJS viewer.js by Atlassian to be AMD compatible
 * and to remove unneeded features
 */
define('pdf-viewer/utils',
  [],
  function () {

    /**
     *  Approximates float number as a fraction using Farey sequence (max order
     *  of 8).
     *  @param {number} x - Positive float number.
     *  @returns {Array} Estimated fraction: the first array item is a numerator,
     *                   the second one is a denominator.
     */
    var approximateFraction = function roundToDivide (x) {
      // Fast paths for int numbers or their inversions.
      if (Math.floor(x) === x) {
        return [x, 1];
      }
      var xinv = 1 / x;
      var limit = 8;
      if (xinv > limit) {
        return [1, limit];
      } else  if (Math.floor(xinv) === xinv) {
        return [1, xinv];
      }

      var x_ = x > 1 ? xinv : x;
      // a/b and c/d are neighbours in Farey sequence.
      var a = 0, b = 1, c = 1, d = 1;
      // Limiting search to order 8.

      /* eslint-disable no-constant-condition */
      while (true) {
        // Generating next term in sequence (order of q).
        var p = a + c, q = b + d;
        if (q > limit) {
          break;
        }
        if (x_ <= p / q) {
          c = p; d = q;
        } else {
          a = p; b = q;
        }
      }
      /* eslint-enable no-constant-condition */

      // Select closest of the neighbours to x.
      if (x_ - a / b < c / d - x_) {
        return x_ === x ? [a, b] : [b, a];
      } else {
        return x_ === x ? [c, d] : [d, c];
      }
    };

    var roundToDivide = function roundToDivide (x, div) {
      var r = x % div;
      return r === 0 ? x : Math.round(x - r + div);
    };

    /**
     * Returns scale factor for the canvas. It makes sense for the HiDPI displays.
     * @return {Object} The object with horizontal (sx) and vertical (sy)
     scales. The scaled property is set to false if scaling is
     not required, true otherwise.
     */
    var getOutputScale = function getOutputScale (ctx) {
      var devicePixelRatio = window.devicePixelRatio || 1;
      var backingStoreRatio = ctx.webkitBackingStorePixelRatio ||
        ctx.mozBackingStorePixelRatio ||
        ctx.msBackingStorePixelRatio ||
        ctx.oBackingStorePixelRatio ||
        ctx.backingStorePixelRatio || 1;
      var pixelRatio = devicePixelRatio / backingStoreRatio;
      return {
        sx: pixelRatio,
        sy: pixelRatio,
        scaled: pixelRatio != 1
      };
    };

    return {
      approximateFraction: approximateFraction,
      roundToDivide: roundToDivide,
      getOutputScale: getOutputScale
    };
  });
