(function () {
  /*global sinon, test, equal, ok, THREE*/
  'use strict';

  var newFileViewerMock = function () {
    var mock = new FileViewer({});
    sinon.stub(mock._view.fileContentView, 'getLayerForName');
    return mock;
  };

  var new3DView = function (fileViewer) {
    return new ThreeDView({
      fileViewer: fileViewer,
      model: {
        get: function () { return ''; }
      }
    });
  };

  var FileViewer = require('file-viewer');
  var ThreeView = require('3d-viewer/three-viewer');
  var ThreeDView = require('3d-view');

  module('3DView');

  test('should set 3 new actions to the toolbar' , function () {

    // given
    var fileViewerMock = newFileViewerMock();
    var setActions = sinon.spy(function () {});
    var toolbar = { setActions: setActions, getActions: function() { return []; } };
    fileViewerMock._view.fileContentView.getLayerForName.returns(toolbar);
    var threeDView = new3DView(fileViewerMock);

    // when
    threeDView.setupMode('BASE');

    // then
    equal(3, setActions.getCall(0).args[0].length);
  });

  test('should set default camera mode to Orthographic, and should toggle on grid and axis', function () {
    // given
    var fileViewerMock = newFileViewerMock();
    fileViewerMock._view.fileContentView.getLayerForName.returns({ setActions: function () {} });
    var view = new3DView(fileViewerMock);
    ThreeView.loadModel = function () { return { file: { onComplete: function () {} } }; };

    // when
    view.render();

    // then
    equal(true, view.viewer.camera instanceof THREE.OrthographicCamera);
    equal(true, view.viewer.grid.visible);
    equal(true, view.viewer.axes.visible);

  });

  test('should toggle camera mode', function () {
    // given
    var fileViewerMock = newFileViewerMock();
    fileViewerMock._view.fileContentView.getLayerForName.returns({ setActions: function () {} });
    var view = new3DView(fileViewerMock);
    ThreeView.loadModel = function () { return { file: { onComplete: function () {} } }; };

    // when
    view.render();
    view.viewer.toggleCameraType();

    // then
    equal(true, view.viewer.camera instanceof THREE.PerspectiveCamera);

  });

  test('should toggle grid', function () {
    // given
    var fileViewerMock = newFileViewerMock();
    fileViewerMock._view.fileContentView.getLayerForName.returns({ setActions: function () {} });
    var view = new3DView(fileViewerMock);
    ThreeView.loadModel = function () { return { file: { onComplete: function () {} } }; };

    // when
    view.render();
    view.viewer.toggleGrid();

    // then
    equal(false, view.viewer.grid.visible);

  });

  test('should toggle axis', function () {
    // given
    var fileViewerMock = newFileViewerMock();
    fileViewerMock._view.fileContentView.getLayerForName.returns({ setActions: function () {} });
    var view = new3DView(fileViewerMock);
    ThreeView.loadModel = function () { return { file: { onComplete: function () {} } }; };

    // when
    view.render();
    view.viewer.toggleAxes();

    // then
    equal(false, view.viewer.axes.visible);

  });

  test('should have the same zoom in and zoom out factors', function () {
    // given
    var fileViewerMock = newFileViewerMock();
    fileViewerMock._view.fileContentView.getLayerForName.returns({ setActions: function () {} });
    var view = new3DView(fileViewerMock);
    ThreeView.loadModel = function () { return { file: { onComplete: function () {} } }; };
    var zoomSpy = sinon.spy(view.viewer, "zoom");

    // when
    view.render();
    view.zoomIn();
    view.zoomOut();

    // then
    equal(true, zoomSpy.withArgs(1.0 + ThreeDView._ZOOM_FACTOR).calledOnce);
    equal(true, zoomSpy.withArgs(1.0 - ThreeDView._ZOOM_FACTOR).calledOnce);

  });



}());