define('viewer-registry',
  [
    'underscore',
    'assert'
  ],
  function (
    _,
    assert
  ) {
    'use strict';

    var createMatchFn = function (expected) {
      return function (current) {
        return current === expected;
      };
    };

    /**
     * ViewerRegistry is responsible for mapping file types to content viewers.
     *
     * When FileViewer is asked to view a file, it uses the file's type and asks
     * its ViewerRegistry for the proper viewer. In addition, ViewerRegistry is
     * exposed to the outside world. Therefore viewers and plugins can register
     * themself without touching FileViwer core.
     *
     * A viewer is a backbone view and is registered via a function that wraps this
     * view into a promise.
     *
     * Multiple viewers for the same filetype are weighted and can thus be overriden.
     * The default weight is 10 with a lower weight meaning higher priority.
     */
    var ViewerRegistry = function () {
      this._handlers = [];
    };

    /**
     * Checks for a valid viewer (is a function).
     *
     * @param {*} previewer
     * @return {boolean}
     */
    ViewerRegistry.isValidPreviewer = function (previewer) {
      return _.isFunction(previewer);
    };

    /**
     * Checks for a valid weight (a number).
     *
     * @param {*} weight
     * @return {boolean}
     */
    ViewerRegistry.isValidWeight = function (weight) {
      return typeof weight === 'number' && !isNaN(weight);
    };

    /**
     * Register a new viewer for a given filetype with an optional weight.
     *
     * fileType can either be a string which is then directly matched or a
     * predicate function that get's handed the current file type and then
     * can return true / false.
     *
     * @param {string|function} fileType
     * @param {function} previewer
     * @param {integer} [weight=10]
     * @thors {Error}
     */
    ViewerRegistry.prototype.register = function (fileType, previewer, weight) {
      var matchesFileType = typeof fileType === 'function' ? fileType : createMatchFn(fileType);

      weight = weight || 10;

      assert(ViewerRegistry.isValidPreviewer(previewer), 'previewer is valid');
      assert(ViewerRegistry.isValidWeight(weight), 'weight is valid');

      this._handlers.push({
        matchesFileType: matchesFileType,
        previewer: previewer,
        weight: weight
      });

      this._updateWeighting();
    };

    /**
     * Get the viewer with the lowest weight for the given fileType.
     *
     * Returns undefined if no viewer is found.
     *
     * @param {string} fileType
     * @return {object} previewer
     */
    ViewerRegistry.prototype.get = function (fileType) {
      var handler = _.find(this._handlers, function (handler) {
        return handler.matchesFileType(fileType);
      });

      return handler && handler.previewer;
    };

    ViewerRegistry.prototype._updateWeighting = function () {
      // Sorts handlers by weight - needs to be called after a new handler is inserted.
      this._handlers = _.sortBy(this._handlers, function (handler) {
        return handler.weight;
      });
    };

    return ViewerRegistry;
  }
);
