define('template-store', [
  'assert',
  'underscore'
], function (
  assert,
  _
) {
  'use strict';

  /**
   * Provides templates by asking a previously configured backend.
   *
   * A backend is a function that accepts a template path and returns the
   * matched template. If no template is found, it returns undefined.
   *
   * @constructor
   */
  var TemplateStore = function () {
    this._backend = null;
  };

  /**
   * Checks if backend is a valid backend.
   * @param {*} backend
   * @return {bool}
   */
  TemplateStore.validBackend = function (backend) {
    return _.isFunction(backend);
  };

  /**
   * Asks its backend for the given templateUrl.
   * @param {string} templateUrl
   * @return {*}
   * @throws {Error} if backend is invalid
   */
  TemplateStore.prototype.get = function (templateUrl) {
    assert(TemplateStore.validBackend(this._backend), 'backend is valid');
    return this._backend(templateUrl);
  };

  /**
   * Sets a backend for this template store.
   * @param {function} backend
   * @throws {Error} if backend is invalid
   */
  TemplateStore.prototype.useBackend = function (backend) {
    assert(TemplateStore.validBackend(backend), 'backend is valid');
    this._backend = backend;
  };

  return TemplateStore;
});