(function () {
    'use strict';

    var Backbone = require('backbone');
    var Files = require('files');
    var File = require('file');
    var PanelContainerView = require('panel-container-view');

    module('PanelContainerView', {
        setup: function () {
            this.panelContainerView = new PanelContainerView({ collection: this.collection });
        }
    });

    test('#addPanelView() adds a PanelView for the given name', function () {
        this.panelContainerView.addPanelView('name', Backbone.View);
        ok(this.panelContainerView.hasPanelView('name'), 'panel view exists after adding');
    });

    test('#addPanelView() does not instanciate a panel', function () {
        var spy = sinon.spy();
        var View = Backbone.View.extend({ initialize: spy });
        this.panelContainerView.addPanelView('panel', View);
        ok(!spy.called, 'panel was not yet created');
    });

    test('#initializePanel() instanciates a panel', function () {
        var spy = sinon.spy();
        var View = Backbone.View.extend({ initialize: spy });
        this.panelContainerView.addPanelView('panel', View);
        this.panelContainerView.initializePanel('panel');
        ok(spy.called, 'panel is created');
    });

    test('#initializePanel() fires togglePanel', function () {
        var listener = sinon.spy();

        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.on('togglePanel', listener);
        this.panelContainerView.initializePanel('panelA');

        ok(listener.calledOnce, 'togglePanel event is triggered');
        equal(listener.args[0][0], 'panelA', 'togglePanel passes the proper params');
        equal(listener.args[0][1], true, 'togglePanel passes the proper params');
    });

    test('#initializePanel() calls render', function () {
        this.panelContainerView.addPanelView('panel', Backbone.View);

        sinon.spy(this.panelContainerView, 'render');
        this.panelContainerView.initializePanel('panel');
        ok(this.panelContainerView.render.calledOnce, 'render was called');
    });

    test('#initializePanel() throws if a panel is already initialized', function () {
        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.initializePanel('panelA');
        throws(function () {
            this.panelContainerView.initializePanel('panelA');
        }.bind(this), 'initializing panel a second time throws');
    });

    test('#initializePanel() throws if the panel name does not exist', function () {
        throws(function () {
            this.panelContainerView.initializePanel('panelA');
        }.bind(this), 'initializing an unknown panel throws');
    });

    test('#teardownPanel() calls teardown on an initialized panel', function () {
        var spy = sinon.spy();
        var View = Backbone.View.extend({ teardown: spy });
        this.panelContainerView.addPanelView('panel', View);

        this.panelContainerView.teardownPanel();
        ok(!spy.called, 'teardown is not called on an uninitiailzed panel');

        this.panelContainerView.initializePanel('panel');
        this.panelContainerView.teardownPanel();
        ok(spy.calledOnce, 'teardown is called on an initiailzed panel');
    });

    test('#teardownPanel() calls remove on an initialized panel', function () {
        var spy = sinon.spy();
        var View = Backbone.View.extend({ remove: spy });
        this.panelContainerView.addPanelView('panel', View);

        this.panelContainerView.teardownPanel();
        ok(!spy.called, 'remove is not called on an uninitiailzed panel');

        this.panelContainerView.initializePanel('panel');
        this.panelContainerView.teardownPanel();
        ok(spy.calledOnce, 'remove is called on an initiailzed panel');
    });

    test('#teardownPanel() calls remove on initialized panel', function () {
        this.panelContainerView.addPanelView('panel', Backbone.View);

        sinon.spy(this.panelContainerView, 'render');
        this.panelContainerView.teardownPanel('panel');
        ok(this.panelContainerView.render.calledOnce, 'render was called');
    });

    test('#teardownPanel() fires teardownPanel', function () {
        var listener = sinon.spy();

        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.initializePanel('panelA');
        this.panelContainerView.on('teardownPanel', listener);
        this.panelContainerView.teardownPanel();

        ok(listener.calledOnce, 'teardownPanel event is triggered');
        equal(listener.args[0][0], 'panelA', 'teardownPanel passes the proper params');
    });

    test('#teardownPanel() fires togglePanel', function () {
        var listener = sinon.spy();

        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.initializePanel('panelA');
        this.panelContainerView.on('togglePanel', listener);
        this.panelContainerView.teardownPanel();

        ok(listener.calledOnce, 'togglePanel event is triggered');
        equal(listener.args[0][0], 'panelA', 'togglePanel passes the proper params');
        equal(listener.args[0][1], false, 'togglePanel passes the proper params');
    });

    test('#teardownPanel() fires teardownPanel with a null panelId if no such panel exists', function () {
        var listener = sinon.spy();

        this.panelContainerView.on('teardownPanel', listener);
        this.panelContainerView.teardownPanel();

        ok(listener.calledOnce, 'teardownPanel event is triggered');
        equal(listener.args[0][0], null, 'teardownPanel passes the proper params');
    });

    test('#teardownPanel() removes the expaneded class from the container', function () {
        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.initializePanel('panelA');

        equal(this.panelContainerView.isAnyPanelInitialized(), true, 'class is set');

        this.panelContainerView.teardownPanel();

        equal(this.panelContainerView.isAnyPanelInitialized(), false, 'class is not set');
    });

    test('#teardownPanel() removes the element from the container', function () {
        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.initializePanel('panelA');
        this.panelContainerView.render();

        equal(this.panelContainerView.$el.children().length, 1, 'has children element');

        this.panelContainerView.teardownPanel();

        equal(this.panelContainerView.$el.children().length, 0, 'has no children element');
    });

    test('#teardownLayers() does not throw if layers are not initialized', function () {
        equal(this.panelContainerView.isAnyPanelInitialized(), false);
        this.panelContainerView.teardownPanel();
        // no exception thrown at this point
    });

    test('#reinitializePanel() recreates the current view', function () {
        var initializeSpy = sinon.spy();
        var teardownSpy = sinon.spy();
        var View = Backbone.View.extend({
            initialize: initializeSpy,
            teardown: teardownSpy
        });
        this.panelContainerView.addPanelView('panelA', View);
        this.panelContainerView.initializePanel('panelA');
        ok(initializeSpy.calledOnce, 'view is initialized');

        this.panelContainerView.reinitializePanel();

        ok(teardownSpy.calledOnce, 'view is torn down');
        ok(initializeSpy.calledTwice, 'view is re-initialized');
    });

    test('#reinitializePanel() creates no view if there was none before', function () {
        var initializeSpy = sinon.spy();
        var View = Backbone.View.extend({
            initialize: initializeSpy
        });
        this.panelContainerView.addPanelView('panelA', View);
        ok(!initializeSpy.called, 'view is not initialized');
        this.panelContainerView.reinitializePanel();
        ok(!initializeSpy.called, 'view is still not initialized');
    });

    test('#isAnyPanelInitialized() returns true if any panel is initialized', function () {
        this.panelContainerView.addPanelView('panelA', Backbone.View);

        equal(this.panelContainerView.isAnyPanelInitialized(), false, 'no panel is initialized');
        this.panelContainerView.initializePanel('panelA');
        equal(this.panelContainerView.isAnyPanelInitialized(), true, 'any panel is initialized');
    });

    test('#isPanelInitialized() returns true if panel with the given name is initialized', function () {
        this.panelContainerView.addPanelView('panelA', Backbone.View);

        equal(this.panelContainerView.isPanelInitialized(), false, 'no panel is initialized');
        this.panelContainerView.initializePanel('panelA');
        equal(this.panelContainerView.isPanelInitialized('panelA'), true, 'panelA is initialized');
        equal(this.panelContainerView.isPanelInitialized('panelB'), false, 'panelB is not initialized');
    });

    test('#getInitializedPanelName() returns the initialized panel name', function () {
        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.initializePanel('panelA');
        equal(this.panelContainerView.getInitializedPanelName(), 'panelA', 'returns panel name');
    });

    test('#getInitializedPanel() returns the initialized panel', function () {
        var value = {};
        var CustomView = Backbone.View.extend({
            initialize: function () { this.value = value; }
        });

        this.panelContainerView.addPanelView('panelA', CustomView);
        this.panelContainerView.initializePanel('panelA');
        equal(this.panelContainerView.getInitializedPanel().value, value, 'returns panel instance');
    });

    test('#render() renders the initialized panel', function () {
        var spyA = sinon.stub();
        spyA.returns(spyA);
        var spyB = sinon.stub();
        spyB.returns(spyB);
        var CustomViewA = Backbone.View.extend({ render: spyA });
        var CustomViewB = Backbone.View.extend({ render: spyA });

        this.panelContainerView.addPanelView('panelA', CustomViewA);
        this.panelContainerView.addPanelView('panelB', CustomViewB);
        this.panelContainerView.initializePanel('panelA');

        this.panelContainerView.render();

        ok(spyA.called, 'initialized panel is rendered');
        ok(!spyB.called, 'uninitialized panel is not rendered');
    });

    test('#render() fires renderPanel event', function () {
        var listener = sinon.spy();

        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.initializePanel('panelA');

        this.panelContainerView.on('renderPanel', listener);
        this.panelContainerView.render();

        ok(listener.calledOnce, 'event is triggered');
    });

    test('#render() empties the element and appends an initialized panel', function () {
        this.panelContainerView.addPanelView('panelA', Backbone.View);
        this.panelContainerView.initializePanel('panelA');
        this.panelContainerView.render();
        this.panelContainerView.$el.append('<span></span><span></span><span></span>');
        equal(this.panelContainerView.$el.children().length, 4, 'has 4 children before render');
        this.panelContainerView.render();
        equal(this.panelContainerView.$el.children().length, 1, 'has 1 children after render');
    });
}());
