define('panel-container-view', [
  'backbone',
  'assert',
  'constants-dictionary'
], function (
  Backbone,
  assert,
  ConstantsDictionary
) {
  'use strict';

  var PanelContainerView = Backbone.View.extend({

    className: 'panel-view',

    /**
     * @constructor
     * @param {object} options
     */
    initialize: function (options) {
      this._panelViewsByName = new ConstantsDictionary();
      this._currentPanel = null;
      this._currentPanelName = null;
      this._lastAddedPanelName = null;
      this._fileViewer = options.fileViewer;
    },

    /**
     * Checks if a panel with the given name exists.
     * @param {string} name
     * @return {bool}
     */
    hasPanelView: function (name) {
      return this._panelViewsByName.isDefined(name);
    },

    /**
     * Adds a View as a panel with a certain, unique name.
     * @param {string} name
     * @param {Backbone.View} PanelView
     * @throws Error if name is already used.
     */
    addPanelView: function (name, PanelView) {
      this._panelViewsByName.define(name, PanelView);
      this._lastAddedPanelName = name;
    },

    /**
     * Checks wether any panel is currently initialized.
     * @return {bool}
     */
    isAnyPanelInitialized: function () {
      return this.$el.is('.expanded');
    },

    /**
     * Checks wether a panel with the given name is currently initialized.
     * @param {string} name
     * @return {bool}
     */
    isPanelInitialized: function (name) {
      return this._currentPanelName === name;
    },

    /**
     * Initializes the panel with the given name. Then re-renders itself.
     * @param name {String} name of the panel to be initialized. If empty, then use the last added (using addPanelView) panel.
     * @fires initializePanel(panelName)
     * @fires togglePanel(panelName, isInitialized)
     * @throws Error if a panel is already initialized or the panel doesn't exist
     */
    initializePanel: function (name) {
      name = name || this._lastAddedPanelName;
      assert(this.isAnyPanelInitialized() === false, 'no panel is initialized');
      assert(this.hasPanelView(name) === true, 'panel exists');

      var PanelView = this._panelViewsByName.lookup(name);

      this._currentPanelName = name;
      this._currentPanel = new PanelView({
        collection: this.collection,
        fileViewer: this._fileViewer,
        panelView: this
      });

      this.$el.toggleClass('expanded', true);

      this.trigger('initializePanel', this._currentPanelName);
      this.trigger('togglePanel', this._currentPanelName, true);

      this.render();
    },

    /**
     * Tears the initialized panel down and removes it. Then re-renders itself.
     * Won't throw if there's no initialized panel.
     * @fires togglePanel(panelName, isInitialized)
     * @fires teardownPanel(panelName)
     */
    teardownPanel: function () {
      if (this._currentPanel) {
        if (this._currentPanel.teardown) {
          this._currentPanel.teardown();
        }
        this._currentPanel.remove();
      }

      this.$el.toggleClass('expanded', false);

      this.trigger('togglePanel', this._currentPanelName, false);
      this.trigger('teardownPanel', this._currentPanelName);

      this._currentPanelName = null;
      this._currentPanel = null;

      this.render();
    },

    /**
     * Utility method. Recreates the current PanelView (if there is one).
     */
    reinitializePanel: function () {
      if (!this.isAnyPanelInitialized()) { return; }

      var previousPanel = this.getInitializedPanelName();
      this.teardownPanel();
      this.initializePanel(previousPanel);
    },

    /**
     * Returns the name of the instanciated PanelView.
     * @return {string} panelName
     * @throws {Error} if no panel is initialized
     */
    getInitializedPanelName: function () {
      assert(this.isAnyPanelInitialized(), 'a panel is initialized');
      return this._currentPanelName;
    },

    /**
     * Returns the instanciated PanelView.
     * @return {PanelView}
     * @throws {Error} if no panel is initialized
     */
    getInitializedPanel: function () {
      return this._currentPanel;
    },

    /**
     * Renders the initialized panel.
     * Won't throw if no panel is initialized.
     * @fires renderPanel(panelName)
     */
    render: function () {
      this.$el.empty();

      if (this.isAnyPanelInitialized()) {
        this._currentPanel.render();
        this._currentPanel.$el.appendTo(this.$el);
      }
      this.trigger('renderPanel', this._currentPanelName);

      return this;
    }

  });

  return PanelContainerView;
});