define('focus-manager-impl', ['assert'], function (assert) {
  'use strict';

  var hasFocusOverride = function (element) {
    if (!element || !element.hasAttribute) { return false; }
    return element.hasAttribute('data-fv-allow-focus') || hasFocusOverride(element.parentElement);
  };

  /**
   * Trap focus for overlays and dialogs to enable proper keyboard navigation.
   * @constructor
   * @param {jQuery} $el - The outmost element within focus will be trapped
   */
  var FocusManager = function ($el) {
    assert($el, 'the first parameter to new FocusManager is a jQuery element');
    this.$container = $el;
    this._handler = this._focusHandler.bind(this);
  };

  /**
   * Trap element focus inside container element.
   */
  FocusManager.prototype.trapFocus = function () {
    this._unfocusParentPage();
    document.addEventListener('focus', this._handler, true);
  };

  /**
   * Stop trapping element focus inside container element.
   */
  FocusManager.prototype.releaseFocus = function () {
    document.removeEventListener('focus', this._handler, true);
    this._restoreFocus();
  };

  // Focus event handler.
  // On focus instantly blur the element if it is not a child of container
  // and set focus back to the container element.
  FocusManager.prototype._focusHandler = function (event) {
    if (
      !hasFocusOverride(event.target) &&
      this.$container &&
      !this.$container[0].contains(event.target)
    ) {
      event.stopPropagation();
      document.activeElement.blur();
      this.$container.attr('tabindex', 1);
      this.$container.focus();
    }
  };

  // Store active DOM element, unfocus it and then focus the container.
  FocusManager.prototype._unfocusParentPage = function () {
    this._originalActiveElement = document.activeElement;
    this._originalActiveElement.blur();
    this.$container.attr('tabindex', 1);
    this.$container.focus();
  };

  // Restore focus to the stored original active element.
  FocusManager.prototype._restoreFocus = function () {
    if (this._originalActiveElement) {
      this._originalActiveElement.focus();
    }
  };

  return FocusManager;
});
