define('constants-dictionary', [
  'assert'
], function (
  assert
) {
  'use strict';

  /**
   * A constants dictionary is a dictionary where you can only define a
   * given name once.
   */
  var ConstantsDictionary = function () {
    this._valuesByName = {};
    this._names = [];
  };

  /**
   * Define a name with the given value.
   * @param {string} name The name that is being defined.
   * @param {*} value Any value, even undefined.
   * @throws Error if name is already defined.
   */
  ConstantsDictionary.prototype.define = function (name, value) {
    assert(!this.isDefined(name), 'name is unique');
    this._valuesByName[name] = value;
    this._names.push(name);
  };

  /**
   * Lookup a name with the given value.
   * @param {string} name The name to lookup.
   * @throws Error if name is not yet defined.
   */
  ConstantsDictionary.prototype.lookup = function (name) {
    assert(this.isDefined(name), 'name is defined');
    return this._valuesByName[name];
  };

  /**
   * Check if a name is defined.
   * @param {string} name The name to check.
   * @return {boolean}
   */
  ConstantsDictionary.prototype.isDefined = function (name) {
    return name in this._valuesByName;
  };

  /**
   * Lists all definitions in the order they were defined. Returns an array of objects
   * with name and value properties.
   * @return {Array}
   */
  ConstantsDictionary.prototype.list = function () {
    return this._names.map(function (name) {
      return {
        name: name,
        value: this._valuesByName[name]
      };
    }, this);
  };

  return ConstantsDictionary;
});