(function () {
    'use strict';

    var File = require('file');
    var FileViewer = require('file-viewer');
    var Analytics = require('Analytics');

    module('Analytics');

    test('#send() forwards to the configured analytics backend', function () {
        // given
        var analyticsBackend = sinon.spy();
        var fileViewer = new FileViewer({});
        var noop = function () {};

        var data = { 'some': 'data' };

        var analytics = new Analytics(analyticsBackend, fileViewer, noop);

        // when
        analytics.send('analytics-key', data);

        // then
        ok(analyticsBackend.calledOnce);
        equal(analyticsBackend.args[0][0], 'analytics-key');
        equal(analyticsBackend.args[0][1].some, data.some);
    });

    test('#send() augments the data with file info', function () {
        // given
        var analyticsBackend = sinon.spy();
        var fileViewer = new FileViewer({});
        var hasher = function (x) { return x.toUpperCase(); };

        var data = {};
        var file = { type: 'unknown', src: 'http://localhost:8000' };

        var analytics = new Analytics(analyticsBackend, fileViewer, hasher);

        fileViewer.getCurrentFile = sinon.stub().returns(new File(file));
        fileViewer.getView().getViewState = sinon.stub().returns('success');

        // when
        analytics.send('analytics-key', data);

        // then
        var passedData = analyticsBackend.args[0][1];
        ok(analyticsBackend.calledOnce);
        equal(passedData.fileType, file.type);
        equal(passedData.fileId, hasher(file.src));
        equal(passedData.fileState, 'success');
    });

    test('#send() is a noop if no analytics backend is configured', function () {
        var analytics = new Analytics();
        analytics.send('analytics-key', { 'some': 'data' });
        expect(0); // no exception
    });

    test('#fn() returns a partialy applied send()', function () {
        // given
        var key = 'someKey';
        var data = { actiontype: 'button' };
        var analytics = new Analytics();

        var spy = analytics.send = sinon.spy();
        var fn = analytics.fn(key, data);

        // when
        fn();

        // then
        ok(spy.calledOnce);
        equal(spy.args[0][0], key);
        equal(spy.args[0][1].actiontype, data.actiontype);
    });

}());