define('Analytics', ['underscore'], function (_) {
  'use strict';

  /**
   * Augments data and forwards them to a configured analytics backend.
   *
   * An analytics backend is just a function that accepts a key and a data
   * object. It is also responsible for filtering incoming data.
   *
   * For all data send, this module fetches current file and view state
   * and adds them to the data. The added keys are
   * - fileType
   * - fileId (hashed version of the file src)
   * - fileState (value of view state)
   *
   * @param analyticsBackend
   * @param fileViewer
   * @param hasher
   * @returns {Analytics}
   * @constructor
   */
  var Analytics = function (backend, fileViewer, hasher) {
    this._backend = backend;
    this._fileViewer = fileViewer;
    this._hasher = hasher;
  };

  /**
   * Forwards an analytics event to the configured backend.
   * @param {string} key
   * @param {Object} data
   */
  Analytics.prototype.send = function (key, data) {
    if (!this._backend) { return; }
    var file = this._fileViewer.getCurrentFile();
    var attributes = (file && file.attributes) || {};
    var augmentedData = _.extend({
      fileType: attributes.type,
      fileId: this._hasher(attributes.src || ''),
      fileState: this._fileViewer.getView().getViewState()
    }, data);
    this._backend(key, augmentedData);
  };

  /**
   * Returns a partially applied analytics function for use inside of
   * promise handlers.
   * @param {string} key
   * @param {object} data
   * @returns {function}
   */
  Analytics.prototype.fn = function (key, data) {
    return this.send.bind(this, key, data);
  };

  return Analytics;
});
