(function () {
    'use strict';

    var $ = require('jquery');
    var _ = require('underscore');
    var djb2 = require('djb2');
    var FileViewer = require('file-viewer');
    var MainView = require('MainView');
    var ConstantsDictionary = require('constants-dictionary');

    var indexWhereId = function (collection, id) {
        var index;
        for (index = 0; index < collection.length; index++) {
            if (collection[index].id === id) { return index; }
        }
        return null;
    };

    module('FileViewer', {
        setup: function () {
            this.pluginA = function () {};
        },
        teardown: function () {
            // knowledge about internals necessary to reset registered plugins
            FileViewer._plugins = new ConstantsDictionary();
        }
    });

    test('registerPlugin() registers a plugin that gets the instance passed when instanciating FileViewer', function () {
        var spyPlugin = sinon.spy();
        FileViewer.registerPlugin('spyPlugin', spyPlugin);

        var fileViewer = new FileViewer({ usePlugins: ['spyPlugin'] });

        ok(spyPlugin.calledOnce, 'plugin is called');
        equal(spyPlugin.args[0][0], fileViewer, 'instance is passed in');
    });

    test('registerPlugin() throws when registering a plugin with the same name twice', function () {
        FileViewer.registerPlugin('pluginA', this.pluginA);
        throws(function () {
            FileViewer.registerPlugin('pluginA', this.pluginA);
        }.bind(this), 'throws when re-registering a plugin');
    });

    test('registerPlugin() throws when registering something that is not a plugin', function () {
        var noPlugin = null;

        equal(FileViewer.isPlugin(noPlugin), false, 'it is not a plugin');
        throws(function () {
            FileViewer.registerPlugin('pluginA', noPlugin);
        }.bind(this), 'throws when registering a non-plugin');
    });

    test('isPlugin() returns true for functions', function () {
        var noPlugin = {};
        equal(FileViewer.isPlugin(noPlugin), false, 'an object is not a plugin');
        equal(FileViewer.isPlugin(this.pluginA), true, 'a function is a plugin');
    });

    test('new FileViewer() invokes all plugins', function () {
        var pluginA = sinon.spy();
        var pluginB = sinon.spy();
        FileViewer.registerPlugin('pluginA', pluginA);
        FileViewer.registerPlugin('pluginB', pluginB);

        var viewer = new FileViewer({});

        ok(pluginA.calledOnce, 'plugin is called');
        ok(pluginB.calledOnce, 'plugin is called');
    });

    test('#getView() returns an instance of MainView', function () {
        var fileViewOptions = {};
        var fileViewer = new FileViewer({ fileViewOptions: fileViewOptions });
        ok(fileViewer.getView() instanceof MainView, 'is a MainView');
    });

    test('#getConfig() returns the specified config', function () {
        var cfg = {};
        var fileViewer = new FileViewer(cfg);
        equal(fileViewer.getConfig(), cfg, 'same config');
    });

    test('#getFiles() returns the specified files as plain json', function () {
        var files = [{id: 0}, {id: 1}];
        var fileViewer = new FileViewer();
        fileViewer.updateFiles(files);
        equal(fileViewer.getFiles()[0].id, files[0].id);
        equal(fileViewer.getFiles()[1].id, files[1].id);
    });

    test('#updateFiles() doesn‘t re-render', function () {
        // given
        var file1 = { id: 'a' };
        var fv = new FileViewer();
        var view = fv.getView();
        sinon.spy(view, 'showFile');

        // when
        fv.updateFiles([file1]);

        // then
        ok(!view.showFile.called);
    });

    test('#updateFiles() returns the updated list of files', function () {
        // given
        var file1 = { id: 'a' };
        var file2 = { id: 'b' };
        var fv = new FileViewer({ files: [file1] });

        // when
        fv.updateFiles([file1, file2]);

        // then
        var files = fv.getFiles();
        equal(files[0].id, file1.id);
        equal(files[1].id, file2.id);
    });

    test('#updateFiles() triggers a "fv.updateFiles" event', function () {
        // given
        var file1 = { id: 'a' };
        var fv = new FileViewer();
        var listener = sinon.spy();
        fv.on('fv.updateFiles', listener);

        // when
        fv.updateFiles([file1]);

        // then
        ok(listener.calledOnce);
    });

    test('#updateFiles() replaces all files when no matcher is given', function () {
        // given
        var file1 = { id: 'a' };
        var file2 = { id: 'b' };
        var file3 = { id: 'c' };
        var fv = new FileViewer({ files: [file1, file2] });

        // when
        fv.updateFiles([file3]);

        // then
        equal(fv.getFiles().length, 1);
    });

    test('#updateFiles() passes a copy of the file attributes to a given matcher', function () {
        // given
        var file1 = { id: 'a' };
        var file2 = { id: 'b' };
        var fv = new FileViewer({ files: [file1] });
        var matcher = sinon.spy();

        // when
        fv.updateFiles([file2], matcher);

        // then
        ok(matcher.calledTwice);
        equal(matcher.args[0][0].id, file1.id);
        equal(matcher.args[1][0].id, file2.id);
    });

    test('#updateFiles() maintains the current order when a matcher is given', function () {
        // given
        var file1 = { id: 'a', order: 10 };
        var file2 = { id: 'b', order: 20 };
        var file3 = { id: 'c', order: 30 };
        var files = [file2, file1, file3]; // files are out of order
        var sortedFiles = _.sortBy(files, 'order');
        var fv = new FileViewer({ files: files });

        fv._fileState.setCurrentFileIndex(0); // file2 is now selected
        equal(fv._fileState.getCurrent().id, 'b');

        // when
        fv.updateFiles(sortedFiles, function (x) { return x.id; });

        // then
        equal(indexWhereId(fv.getFiles(), 'a'), 1);
    });

    test('#updateFiles() appends new files at the end of the collection when a matcher is given', function () {
        // given
        var file1 = { id: 'a' };
        var file2 = { id: 'b' };
        var file3 = { id: 'c' };
        var files = [file1, file2];
        var fv = new FileViewer({ files: files });

        // when
        fv.updateFiles([file3, file1, file2], function (x) { return x.id; });

        // then
        equal(indexWhereId(fv.getFiles(), 'c'), 2);
    });

    test('#updateFiles() doesn‘t remove existing, but unmatched files when a matcher is given', function () {
        // given
        var file1 = { id: 'a' };
        var file2 = { id: 'b' };
        var file3 = { id: 'c' };
        var files = [file1, file2];
        var fv = new FileViewer({ files: files });

        // when
        fv.updateFiles([file3], function (x) { return x.id; });

        // then
        equal(fv.getFiles().length, 3);
    });

    test('#analytics returns an instance of Analytics that is configured by FileViewer', function () {
        var backend = sinon.spy();
        var fileViewer = new FileViewer({analyticsBackend: backend});
        fileViewer.analytics.send('a', {});
        ok(backend.calledOnce);
    });

    test('#open() just opens the FileViewer when no file is given', function () {
        var fileViewer = new FileViewer({appendTo: $('<div>')});
        var openListener = sinon.spy();
        var showFileListener = sinon.spy();
        fileViewer.on('fv.open', openListener);
        fileViewer.on('fv.showFile', showFileListener);

        fileViewer.open();

        ok(fileViewer.isOpen());
        ok(openListener.calledOnce);
        ok(!showFileListener.called);
    });

    test('#open() opens FileViewer and shows the file that matches the query', function () {
        // given
        var file = {src: 'test'};
        var fileViewer = new FileViewer({appendTo: $('<div>')});
        var openListener = sinon.spy();
        var showFileListener = sinon.spy();
        fileViewer.on('fv.open', openListener);
        fileViewer.on('fv.showFile', showFileListener);

        // when
        fileViewer.updateFiles([file]);
        fileViewer.open({src: file.src});

        // then
        ok(fileViewer.isOpen());
        ok(openListener.calledOnce);
        ok(showFileListener.calledOnce);
    });

    test('#open() opens FileViewer and show an error if no file matches the query', function () {
        // given
        var fileViewer = new FileViewer({appendTo: $('<div>')});
        var openListener = sinon.spy();
        var showFileErrorListener = sinon.spy();
        fileViewer.on('fv.open', openListener);
        fileViewer.on('fv.showFileError', showFileErrorListener);

        // when
        fileViewer.open({src: 'something'});

        // then
        ok(fileViewer.isOpen());
        ok(openListener.calledOnce);
        ok(showFileErrorListener.calledOnce);
    });

    test('#open() triggers no analytics event if no file is given', function () {
        var backend = sinon.spy();
        var fileViewer = new FileViewer({
            appendTo: $('<div>'),
            analyticsBackend: backend
        });
        fileViewer.open();
        ok(!backend.called);
    });

    test('#open() triggers an analytics event if a file is given', function () {
        // given
        var backend = sinon.spy();
        var fileViewer = new FileViewer({
            appendTo: $('<div>'),
            analyticsBackend: backend
        });
        var file = {
            src: 'test',
            type: 'test/example'
        };

        // when
        fileViewer.updateFiles([file]);
        fileViewer.open({src: file.src}, 'exampleSource');

        // then
        ok(backend.calledOnce);
        var key = backend.args[0][0];
        var data = backend.args[0][1];

        equal(key, 'files.fileviewer-web.opened');
        equal(data.source, 'exampleSource');
        equal(data.fileType, file.type);
        equal(data.fileId, djb2(file.src));
        equal(data.fileState, 'success');
    });

    test('#setFiles() replaces the current collection', function () {
        // given
        var file1 = { id: 'a' };
        var file2 = { id: 'b' };
        var fileViewer = new FileViewer({ files: [file1] });

        // when
        fileViewer.setFiles([file2]);

        // then
        equal(fileViewer.getFiles().length, 1);
        equal(fileViewer.getFiles()[0].id, 'b');
    });

    test('#setFiles() calls MainView#showFile() with the first file matched by the given query when open', function () {
        // given
        var file1 = { id: 'a' };
        var viewer = new FileViewer();
        var view = viewer.getView();
        sinon.spy(view, 'showFile');

        // when
        viewer.open();
        viewer.setFiles([file1], { id: 'a' });

        // then
        ok(view.showFile.calledOnce);
        equal(view.showFile.args[0][0].id, 'a');
    });

    test('#setFiles() calls MainView#showFile() with null when open and the query does‘t match', function () {
        // given
        var file1 = { id: 'a' };
        var viewer = new FileViewer();
        var view = viewer.getView();
        sinon.spy(view, 'showFile');

        // when
        viewer.open();
        viewer.setFiles([file1]);

        // then
        ok(view.showFile.calledOnce);
        equal(view.showFile.args[0][0], null);
    });

    test('#setFiles() doesn‘t call MainView#showFile() when closed', function () {
        // given
        var file1 = { id: 'a' };
        var viewer = new FileViewer();
        var view = viewer.getView();
        sinon.spy(view, 'showFile');

        // when
        viewer.setFiles([file1], { id: 'a' });

        // then
        ok(!view.showFile.called);
    });

    test('#setFiles() triggers fv.setFiles', function () {
        // given
        var file1 = { id: 'a' };
        var viewer = new FileViewer();
        var listener = sinon.spy();
        viewer.on('fv.setFiles', listener);

        // when
        viewer.setFiles([file1]);

        // then
        ok(listener.calledOnce);
    });

}());
