define('file-viewer', [
  'jquery',
  'underscore',
  'backbone',
  'assert',
  'constants-dictionary',
  'MainView',
  'file-state',
  'files',
  'file',
  'soy-template-backend',
  'asset-module-backend',
  'template-store-singleton',
  'module-store-singleton',
  'viewer-registry',
  'file-types',
  'defaultConfig',
  'storage',
  'image-view-provider',
  'pdf-view-provider',
  'video-view-provider',
  '3d-view-provider',
  'unknown-file-type-view-provider',
  'Analytics',
  'djb2'
],
function (
  $,
  _,
  Backbone,
  assert,
  ConstantsDictionary,
  MainView,
  FileState,
  Files,
  File,
  soyTemplateBackend,
  assetModuleBackend,
  templateStore,
  moduleStore,
  ViewerRegistry,
  fileTypes,
  defaultConfig,
  Storage,
  imageViewProvider,
  pdfViewProvider,
  videoViewProvider,
  threeDViewProvider,
  unknownFileTypeViewProvider,
  Analytics,
  djb2
) {
  'use strict';

  /**
   * Core API to integrate FileViewer into a project.
   *
   * @class
   * @alias FileViewer
   * @param {Object} config
   * @throws {Error} if config is invalid
   */
  var FileViewer = function (config) {
    config = _.defaults(config || {}, defaultConfig);
    config.appendTo = config.appendTo || $('body');
    FileViewer._instanceCount += 1;
    config.instanceId = FileViewer._instanceCount;

    templateStore.useBackend(config.templateBackend || soyTemplateBackend(this));
    moduleStore.useBackend(config.moduleBackend || assetModuleBackend(this));
    
    this._config = config;
    this._properties = new Backbone.Model();

    this._fileState = new FileState();
    this._viewerRegistry = new ViewerRegistry();
    this._analytics = new Analytics(config.analyticsBackend, this, djb2);

    if (config.viewers.indexOf('image') !== -1) {
      this._viewerRegistry.register(fileTypes.isImageBrowserSupported, imageViewProvider, 0);
    }
    if (config.viewers.indexOf('document') !== -1) {
      this._viewerRegistry.register(fileTypes.isPDF, pdfViewProvider, 0);
    }
    if (config.viewers.indexOf('video') !== -1) {
      this._viewerRegistry.register(fileTypes.isMultimediaBrowserSupported, videoViewProvider, 0);
    }
    if (config.viewers.indexOf('3d') !== -1) {
      this._viewerRegistry.register(fileTypes.is3D, threeDViewProvider, 0);
    }

    // set fallback viewer
    this._viewerRegistry.register(fileTypes.matchAll, unknownFileTypeViewProvider, 100);

    this._files = new Files(config.files || [], {
      service: config.commentService
    });
    this._fileState.setCollection(this._files);

    this._view = new MainView({
      model: new Backbone.Model({
        fileViewer: this,
        instanceId: config.instanceId,
        embedded: config.embedded
      })
    });

    this._isOpen = false;

    this._storage = new Storage(this.getConfig().customStorage, 'fileViewer.');

    FileViewer._plugins.list()
      .map(function (definition) { return definition.value; })
      .forEach(function (plugin) { plugin(this); }, this);
  };

  // internal nondecreasing counter
  FileViewer._instanceCount = 0;

  // privately expose available modules for debugging purposes
  // _modules is defined support/require.js
  /* eslint-disable no-undef */
  FileViewer._modules = _modules;
  /* eslint-enable no-undef*/

  // keeps track of registered plugins
  FileViewer._plugins = new ConstantsDictionary();

  /**
   * Current version of this build.
   *
   * @type {String}
   */
  FileViewer.VERSION = '@@__FILEVIEWER_VERSION__'; // replaced during build

  /**
   * Define a new module for the use with FileViewer.require().
   *
   * Be careful with the naming, because module names can be overwritten.
   *
   * @method
   * @param {String} moduleName
   * @param {Array} dependencies
   * @param {Function} factory
   */
  FileViewer.define = define;

  /**
   * Require a previously defined module by name.
   *
   * @method
   * @param {String} moduleName
   * @returns {*}
   */
  FileViewer.require = require;

  /**
   * Register a new plugin for use with FileViewer.
   *
   * @param {String} name
   * @param {Function} plugin
   * @throws {Error} if plugin is invalid or name already exists.
   */
  FileViewer.registerPlugin = function (name, plugin) {
    assert(this.isPlugin(plugin), 'is a plugin');
    this._plugins.define(name, plugin);
  };

  /**
   * Returns if a plugin is enabled for use with FileViewer.
   *
   * @param {String} name
   */
  FileViewer.isPluginEnabled = function (name) {
    return this._plugins.isDefined(name);
  };

  /**
   * Gets a plugin registered for use with FileViewer.
   *
   * @param {String} name
   * @throws {Error} if plugin is invalid or name does not already exists.
   */
  FileViewer.getPlugin = function (name) {
    return this._plugins.lookup(name);
  };

  /**
   * Checks if the given object is a valid plugin.
   *
   * @param {*} potentialPlugin
   * @returns {Boolean}
   */
  FileViewer.isPlugin = function (potentialPlugin) {
    return _.isFunction(potentialPlugin);
  };

  /**
   * Support .on(), .off() and .trigger().
   */
  _.extend(FileViewer.prototype, Backbone.Events);

  /**
   * Instance of the analytics module, use this to send analytics from your plugins.
   */
  Object.defineProperty(FileViewer.prototype, 'analytics', {
    get: function () { return this._analytics; }
  });

  /**
   * The instance of FileViewer was opened.
   *
   * @event FileViewer~'fv.open'
   */

  /**
   * Open this instance of FileViewer by appending it to the configured element. This needs to be called
   * before showing a file.
   *
   * When a fileQuery object is passed in, this file is shown and a special analytics event is triggered.
   * When you want to record, where this interaction is comming from, pass in
   * an additional analyticsSource
   *
   * @param {Object} [fileQuery]
   * @param {String} [analyticsSource]
   * @fires {FileViewer~'fv.open'}
   */
  FileViewer.prototype.open = function (fileQuery, analyticsSource) {
    this._view.render().show().$el.appendTo(this._config.appendTo);
    this._view.delegateEvents();

    this._isOpen = true;
    this.trigger('fv.open');

    if (fileQuery) {
      this.showFileWithQuery(fileQuery).always(
        this.analytics.fn('files.fileviewer-web.opened', {source: analyticsSource})
      );
    }
  };

  /**
   * The instance of FileViewer was closed.
   *
   * @event FileViewer~'fv.close'
   */

  /**
   * Shut down this instance of FileViewer by removing it from the configured element. Reset current file.
   *
   * @fires {FileViewer~'fv.close'}
   */
  FileViewer.prototype.close = function () {
    this._view._currentFile = null;
    this._view.undelegateEvents();
    this._view
      .hide()
      .$el.remove();

    this._isOpen = false;
    this.trigger('fv.close');
  };

  /**
   * Check if FileViewer is currently open.
   *
   * @returns {Boolean}
   */
  FileViewer.prototype.isOpen = function () {
    return this._isOpen;
  };

  /**
   * The current file was changed and is about to be shown.
   *
   * @event FileViewer~'fv.changeFile'
   */

  /**
   * The current file has been rendered successfully.
   *
   * @event FileViewer~'fv.showFile'
   */

  /**
   * The current file has **not** been rendered successfully.
   *
   * @event FileViewer~'fv.showFileError'
   */

  /**
   * Show file that matches the given attribute query.
   *
   * A query is basically a set of values for certain keys that you want to match on. To match the file with id='a'
   * and src='test'.
   *
   *     {
   *     id: 'a',
   *     src: 'test'
   *     }
   *
   * @param {Object} query
   * @returns {Promise.<File>}
   * @fires {FileViewer~'fv.changeFile'}
   * @fires {FileViewer~'fv.showFile'}
   * @fires {FileViewer~'fv.showFileError'}
   */
  FileViewer.prototype.showFileWithQuery = function (query) {
    this._fileState.setCurrentWithQuery(query);
    var file = this._fileState.getCurrent();
    return this.showFile(file);
  };

  /**
   * Show the next file in the collection.
   *
   * @returns {Promise.<File>}
   * @fires {FileViewer~'fv.changeFile'}
   * @fires {FileViewer~'fv.showFile'}
   * @fires {FileViewer~'fv.showFileError'}
   */
  FileViewer.prototype.showFileNext = function () {
    if (this.isShowingLastFile() && !this.getConfig().enableListLoop) {
      return $.when();
    }
    this._fileState.setNext();
    return this.showFile(this._fileState.getCurrent());
  };

  /**
   * Show the previous file in the collection.
   *
   * @returns {Promise.<File>}
   * @fires {FileViewer~'fv.changeFile'}
   * @fires {FileViewer~'fv.showFile'}
   * @fires {FileViewer~'fv.showFileError'}
   */
  FileViewer.prototype.showFilePrev = function () {
    if (this.isShowingFirstFile() && !this.getConfig().enableListLoop) {
      return $.when();
    }
    this._fileState.setPrev();
    return this.showFile(this._fileState.getCurrent());
  };

  /**
   * FileViewer#setFiles() was called.
   * @event FileViewer~'fv.setFiles'
   */

  /**
   * Set both the list of files as well as the current file.
   *
   * Ensures that the updated current file is shown if the viewer is open. Viewer caching might prevent a full
   * re-render.
   *
   * If no query is given or the query doesn't match a file in the collection, the current file is set to `null`,
   * causing an error message to be shown if the viewer is open.
   *
   * @param {Array.<Object>} newFiles
   * @param {Object} [nextFileQuery=null]
   * @fires {FileViewer~'fv.setFiles'}
   */
  FileViewer.prototype.setFiles = function (newFiles, nextFileQuery) {
    this._files.reset(newFiles);
    this._fileState.setCurrentWithQuery(nextFileQuery);

    this.trigger('fv.setFiles');

    if (this.isOpen()) {
      this.showFile(this._fileState.getCurrent());
    }
  };

  /**
   * Returns the file being shown in this viewer.
   *
   * @returns {Object} the file being shown
   */
  FileViewer.prototype.getCurrent = function () {
    var currentFile = this._view.getCurrentFile();
    return currentFile && currentFile.toJSON();
  };

  /**
   * Returns the file being shown in this viewer as a backbone model.
   *
   * **Note**: This method is deprecated, because it exposes a backbone model. See #getCurrent() instead.
   *
   * @returns {File} the file being shown
   * @deprecated
   */
  FileViewer.prototype.getCurrentFile = function () {
    return this._view.getCurrentFile();
  };

  /**
   * Returns the current files collection.
   *
   * @returns {Array.<Object>}
   */
  FileViewer.prototype.getFiles = function () {
    return this._files.toJSON();
  };

  /**
   * Check if current file is the first one in the files collection.
   *
   * @returns {Boolean}
   */
  FileViewer.prototype.isShowingFirstFile = function () {
    return this._fileState.attributes.currentFileIndex === 0;
  };

  /**
   * Check if current file is the last one in the files collection.
   *
   * @returns {Boolean}
   */
  FileViewer.prototype.isShowingLastFile = function () {
    return this._fileState.collection.length ===
      this._fileState.attributes.currentFileIndex + 1;
  };

  /**
   * The view mode was changed.
   *
   * @event FileViewer~'fv.changeMode'
   */

  /**
   * Change current view mode to the given mode.
   *
   * @param {String} mode - either 'BASE' or 'PRESENTATION'
   * @fires {FileViewer~'fv.changeMode'}
   */
  FileViewer.prototype.changeMode = function (mode) {
    this._view.setupMode(mode);
    this.trigger('fv.changeMode', mode);
  };

  /**
   * Return the current mode.
   *
   * @returns {String}
   */
  FileViewer.prototype.getMode = function () {
    return this._view.getMode();
  };

  /**
   * Check if FileViewer is in the given mode.
   *
   * @param {String} mode - either 'BASE' or 'PRESENTATION'
   * @returns {Boolean}
   */
  FileViewer.prototype.isInMode = function (mode) {
    return this._view.isInMode(mode);
  };

  /**
   * A file action was called.
   *
   * @callback FileViewer~fileActionCallback
   * @param {File} file
   */

  /**
   * Add a file action to the viewer.
   *
   * Actions can be registered asynchronously, and are reset when the user navigates to a new file.
   *
   * Commonly, a plugin will listen to the change file event and register a file action
   * conditionally for the displayed file. If a file action shares a key with a file
   * action that currently exists, addFileAction will replace the old action with the
   * new action.
   *
   * @param {Object} opts
   * @param {String} opts.key - a unique identifier for the file action
   * @param {String} opts.text - the text display in the menu item
   * @param {FileViewer~fileActionCallback} opts.callback - a callback to be called when the menu item is selected
   * @throws Error if config is invalid or if no file is currently being viewed
   */
  FileViewer.prototype.addFileAction = function (opts) {
    assert(opts.key, 'has key');
    assert(opts.text, 'has text');
    assert(opts.callback, 'has a callback');
    this._view.fileControlsView.getLayerForName('moreButton').addFileAction(opts);
  };

  /**
   * Remove a file action from the viewer based on the key sent in the parameter.
   *
   * @param {Object} opts
   * @param {String} opts.key - a unique identifier for the file action you want to remove
   * @throws {Error} if no key is provided or if no file is currently being viewed
   */
  FileViewer.prototype.removeFileAction = function (opts) {
    assert(opts.key, 'has key');
    this._view.fileControlsView.getLayerForName('moreButton').removeFileAction(opts);
  };

  /**
   * Check if the FileViewer supports native rendering of a given content type.
   *
   * @param {String} contentType the content type to check if supported
   * @returns {Boolean}
   */
  FileViewer.prototype.supports = function (contentType) {
    var previewer = this._viewerRegistry.get(contentType);
    return previewer && previewer !== unknownFileTypeViewProvider;
  };

  /**
   * Allows non-core code to get and set their own values on an instance of FileViewer.
   *
   * @param {String} key
   * @param {*} value
   */
  FileViewer.prototype.set = function (key, value) {
    this._properties.set(key, value);
  };

  /**
   * Access plugin level properties.
   *
   * @param {String} key
   * @returns {*}
   */
  FileViewer.prototype.get = function (key) {
    return this._properties.get(key);
  };

  /**
   * Returns the specified config.
   *
   * @returns {Object}
   */
  FileViewer.prototype.getConfig = function () {
    return this._config;
  };

  /**
   * Returns the central view of FileViewer.
   *
   * @returns {MainView}
   */
  FileViewer.prototype.getView = function () {
    return this._view;
  };

  /**
   * Returns key-value storage of FileViewer.
   *
   * @returns {Object}
   */
  FileViewer.prototype.getStorage = function () {
    return this._storage;
  };

  /**
   * **DEPRECATED** Show the given file.
   *
   * **Carefull**: This method doesn't set the current file from the collection and thus allows to set a file that
   * is not in the files array, thus causing buggy behaviour with next/prev file.
   *
   * Use #showFileWithQuery() instead.
   *
   * @param {File} file
   * @returns {Promise.<File>}
   * @fires {FileViewer~'fv.changeFile'}
   * @fires {FileViewer~'fv.showFile'}
   * @fires {FileViewer~'fv.showFileError'}
   * @deprecated
   */
  FileViewer.prototype.showFile = function (file) {
    return this._showFile(file);
  };

  /**
   * **DEPRECATED** Show the file matching the given backbone object id.
   *
   * Use #showFileWithQuery() instead.
   *
   * @param {String} cid
   * @returns {Promise.<File>}
   * @fires {FileViewer~'fv.changeFile'}
   * @fires {FileViewer~'fv.showFile'}
   * @fires {FileViewer~'fv.showFileError'}
   * @deprecated
   */
  FileViewer.prototype.showFileWithCID = function (cid) {
    this._fileState.setCurrentWithCID(cid);
    return this.showFile(this._fileState.getCurrent());
  };

  /**
   * **DEPRECATED** Show the file matching the given id.
   *
   * Use #showFileWithQuery() instead.
   *
   * @param {String} id
   * @param {String} [ownerId]
   * @returns {Promise.<File>}
   * @fires {FileViewer~'fv.changeFile'}
   * @fires {FileViewer~'fv.showFile'}
   * @fires {FileViewer~'fv.showFileError'}
   * @deprecated
   */
  FileViewer.prototype.showFileWithId = function (id, ownerId) {
    var fileQuery = { id: id };

    if (ownerId) { fileQuery.ownerId = ownerId; }

    return this.showFileWithQuery(fileQuery);
  };

  /**
   * **DEPRECATED** Show file with matching src.
   *
   * Use #showFileWithQuery() instead.
   *
   * @param {String} src
   * @returns {Promise.<File>}
   * @fires {FileViewer~'fv.changeFile'}
   * @fires {FileViewer~'fv.showFile'}
   * @fires {FileViewer~'fv.showFileError'}
   * @deprecated
   */
  FileViewer.prototype.showFileWithSrc = function (src) {
    var fileQuery = { src: src };

    return this.showFileWithQuery(fileQuery);
  };

  /**
   * **DEPRECATED** Show the first file matching the given selector. If selector is falsy, the first file in the
   * collection is shown.
   *
   * Use #showFileWithQuery() instead.
   *
   * @param {Object} selector
   * @returns {Promise.<File>}
   * @fires {FileViewer~'fv.changeFile'}
   * @fires {FileViewer~'fv.showFile'}
   * @fires {FileViewer~'fv.showFileError'}
   * @deprecated
   */
  FileViewer.prototype.showFileWhere = function (selector) {
    this._fileState.selectWhere(selector);
    return this.showFile(this._fileState.getCurrent());
  };

  /**
   * **DEPRECATED** Maps a file to some value which is used for strict equality checks (`===`).
   *
   * @callback FileViewer~updateFilesMapFn
   * @param {Object} file
   * @returns {*}
   * @deprecated
   */

  /**
   * **DEPRECATED** FileViewer#updateFiles() was called.
   *
   * @event FileViewer~'fv.updateFiles'
   * @deprecated
   */

  /**
   * **DEPRECATED** Update the files collection without updating the view state.
   *
   * **Carefull**: A call to `updateFiles()` neither changes the file currently shown nor causes a re-rendering. This
   * can cause state and view to get out of sync and is likely to introduce bugs. Therefore, this method should no
   * longer be used. See FileViewer#setFiles() instead.
   *
   * This method operates in two modes based on it's input.
   *
   * When invoked with nothing but `files`, the existing file collection is simply replaced with the new one.
   *
   * When an optional `mapFn` function is given, the behaviour changes drastically. Each file in the given `files`
   * array is compared with the already existing ones using `mapFn`.
   *
   * - If a file existed before, but isn't matched by a new file, it won't be updated, but stays in the collection.
   * - If a file existed before and is matched by a new file, it is updated and stays in the collection.
   * - If a file didn't exist before, it is appended to the collection
   *
   * @param {Array.<Object>} files
   * @param {FileViewer~updateFilesMapFn} [mapFn]
   * @returns {Array.<Object>}
   * @fires {FileViewer~'fv.updateFiles'}
   * @deprecated
   */
  FileViewer.prototype.updateFiles = function (files, mapFn) {
    if (!(mapFn && _.isFunction(mapFn))) {
      this._files.reset(files);
    } else {
      var newModels = _.chain(files)
        .map(function (file) {
          var matchedModel = this._files.find(function (collectionModel) {
            return mapFn(collectionModel.toJSON()) === mapFn(file);
          });
          if (matchedModel) {
            matchedModel.set(file);
          } else {
            return new File(file);
          }
        }.bind(this))
        .compact()
        .value();

      this._files.add(newModels, {silent: true});
      this._files.trigger('reset', this._files);
    }

    this.trigger('fv.updateFiles');

    return this._files.toJSON();
  };

  // shows the given backbone file model, triggers an event and returns a promise
  // @todo should resolve with a json description of the passed-in file
  FileViewer.prototype._showFile = function (file) {
    assert(this._isOpen, 'FileViewer is open');
    var triggerEvent = function (event) {
      return function () {
        this.trigger(event, file);
      }.bind(this);
    }.bind(this);
    this.trigger('fv.changeFile', file);
    return this._view.showFile(file)
      .done(triggerEvent('fv.showFile'))
      .fail(triggerEvent('fv.showFileError'));
  };

  return FileViewer;
});
