define('PasswordLayer', [
  'ajs',
  'backbone',
  'jquery',
  'keyboard',
  'template-store-singleton'
], function (
  AJS,
  Backbone,
  $,
  keyboard,
  templateStore
) {
  'use strict';

  var pdfjsPasswordResponses = {
    NEED_PASSWORD: 1,
    INCORRECT_PASSWORD: 2
  };

  var fullscreenEvents = [
    'fullscreenchange',
    'webkitfullscreenchange',
    'mozfullscreenchange',
    'MSFullscreenChange'
  ].join(' ');

  var isFullscreen = function () {
    return (document.fullscreenElement ||
      document.mozFullScreen ||
      document.webkitIsFullScreen ||
      document.msFullscreenElement);
  };

  var PasswordLayer = Backbone.View.extend({

    className: 'cp-password-layer',

    events: {
      'keydown .cp-password-input': '_handleKeyDown',
      'click .cp-password-button': '_handleClick',
      'focus .cp-password-input': '_lockNavigation',
      'blur .cp-password-input': '_unlockNavigation'
    },

    initialize: function (options) {
      this._fileViewer = options.fileViewer;
      this.$el.hide();
    },

    teardown: function () {
      $(document).off(fullscreenEvents, this.updatePasswordLayer.bind(this));
    },

    /**
     * Show the password input layer
     * @param  {Number}   reason     Reason PDFJS why needs the password
     * @param  {Callback} updatePassword
     */
    showPasswordInput: function (reason, updatePassword) {
      $(document).on(fullscreenEvents, this.updatePasswordLayer.bind(this));
      this.updatePassword = updatePassword;
      this._fileViewer._view.fileContentView.getLayerForName('spinner').stopSpin();
      this.$el.show().html(templateStore.get('passwordLayer')({
        prompt: this._getPromptTitle(reason)
      }));
      this.updatePasswordLayer();
      this._showToolbar();
    },

    hidePasswordInput: function () {
      $(document).off(fullscreenEvents, this.updatePasswordLayer.bind(this));
      this.$el.empty();
      this.$el.hide();
    },

    /**
     * Update the passwordLayer depending on fullsccren/no fullscreen
     * Safari/Firefox can't handle keyboard inputs in fullscreen
     */
    updatePasswordLayer: function () {
      if (isFullscreen()) {
        this.$el.find('.cp-password-base').hide();
        this.$el.find('.cp-password-fullscreen').show();
      } else {
        this.$el.find('.cp-password-fullscreen').hide();
        this.$el.find('.cp-password-base').show();
      }
    },

    /**
     * Get i18n string for password prompt based on reason
     * @param  {Number} reason Reason PDFJS why needs the password
     * @return {String}
     */
    _getPromptTitle: function (reason) {
      var title = AJS.I18n.getText('cp.password.needed');
      if (reason === pdfjsPasswordResponses.INCORRECT_PASSWORD) {
        title = AJS.I18n.getText('cp.password.incorrect');
      }
      return title;
    },

    /**
     * Show passwordLayer and toolbar
     */
    _showToolbar: function () {
      var view  = this._fileViewer._view;
      var toolbar = view.fileContentView.getLayerForName('toolbar');
      var mode  = view._modes[view._mode];
      toolbar.setActions(mode.toolbarActions);
      toolbar.render();
    },

    /**
     * Check if password was given and call `updatePassword()`
     */
    _updatePassword: function () {
      var password = this.$el.find('.cp-password-input').val();
      if (password && password.length > 0) {
        this.hidePasswordInput();
        this.updatePassword(password);
      }
    },

    /**
     * Lock navigation keys
     */
    _lockNavigation: function () {
      this._fileViewer._view._navigationKeyLockCount++;
    },

    /**
     * Unlock navigation keys
     */
    _unlockNavigation: function () {
      this._fileViewer._view._navigationKeyLockCount--;
    },

    _handleClick: function (ev) {
      ev.preventDefault();
      this._updatePassword();
    },

    _handleKeyDown: function (ev) {
      if (ev.which === keyboard.keys.RETURN) {
        ev.preventDefault();
        return this._updatePassword();
      }
      if (ev.which === keyboard.keys.ESCAPE) {
        ev.preventDefault();
        return this._fileViewer.close();
      }
    }

  });

  return PasswordLayer;
});
