define('ArrowLayer', [
  'backbone', 'template-store-singleton'
], function (
  Backbone, templateStore
) {
  'use strict';

  var AJS = window.AJS;

  /**
   * Controls to switch between multiple files in a collection.
   * @constructor
   */
  var ArrowLayer = Backbone.View.extend({

    className: 'cp-arrow-layer',

    events: {
      'click #cp-nav-left:not(.disabled)': 'gotoPrevious',
      'click #cp-nav-right:not(.disabled)': 'gotoNext'
    },

    initialize: function (options) {
      this._fileViewer = options.fileViewer;
      this.listenTo(this._fileViewer._fileState.collection, 'add reset', this.checkAndHideNavigation);
      this._visibleArrows = false;
      this._updateElements();
    },

    render: function () {
      this.$el.html(templateStore.get('fileBodyArrows')()).hide();
      this._updateElements();
      this.checkAndHideNavigation();

      return this;
    },

    /**
     * Show next file in collection.
     */
    gotoNext: function () {
      this._fileViewer.showFileNext().always(
        this._fileViewer.analytics.fn('files.fileviewer-web.next', {
          actionType: 'button',
          mode: this._fileViewer.getMode()
        })
      );
    },

    /**
     * Show previous file in collection.
     */
    gotoPrevious: function () {
      this._fileViewer.showFilePrev().always(
        this._fileViewer.analytics.fn('files.fileviewer-web.prev', {
          actionType: 'button',
          mode: this._fileViewer.getMode()
        })
      );
    },

    /**
     * Enable disabled navigation arrow again and remove tooltip
     */
    enableArrow: function ($arrow) {
      $arrow.removeClass('disabled');
      $arrow.removeAttr('original-title');
    },

    /**
     * Disable navigation arrow and add tooltip
     * @param  {Object} options
     * @param  {Object} options.arrow
     * @param  {String} options.tooltipText
     * @param  {String} options.tooltipGravity
     */
    disableArrow: function (options) {
      options.arrow.addClass('disabled');
      options.arrow.attr('original-title', options.tooltipText);
      options.arrow.tooltip({ gravity: options.tooltipGravity });
    },

    /**
     * Returns true if either the left or right navigation arrow is shown
     * @return {Boolean}
     */
    showsArrow: function () {
      return this._visibleArrows;
    },

    /**
     * Check if controls should appear.
     * If there's only a single file in the collection, we don't show them.
     */
    checkAndHideNavigation: function () {
      if (this._fileViewer._fileState.collection.length <= 1) {
        this._visibleArrows = false;
        return this.$arrows.hide();
      }
      this.$arrows.show();
      this._visibleArrows = true;
      if (this._fileViewer.getConfig().enableListLoop) {
        return;
      }
      if (this._fileViewer.isShowingLastFile()) {
        this.disableArrow({
          arrow: this.$arrowRight,
          tooltipText: AJS.I18n.getText('cp.arrow.right.disabled'),
          tooltipGravity: 'e'
        });
      } else if (this._fileViewer.isShowingFirstFile()) {
        this.disableArrow({
          arrow: this.$arrowLeft,
          tooltipText: AJS.I18n.getText('cp.arrow.left.disabled'),
          tooltipGravity: 'w'
        });
      }
    },

    _updateElements: function () {
      this.$arrows = this.$el.find('.cp-nav');
      this.$arrowLeft = this.$el.find('#cp-nav-left');
      this.$arrowRight = this.$el.find('#cp-nav-right');
    }

  });

  return ArrowLayer;
});
