# FileViewer Integration Guide

Integrating FileViewer is a matter of adding a couple of script tags and a few lines of JavaScript to your page. This guide will first walk you through the initial setup and then mention a few other topics:

- Setup
- Enabling Plugins
- Disabling Viewers
- Custom Builds
- Lazy Loading Viewers
- Translations

## Setup

It's 5 simple steps to set everything up.

### 1. Check out FileViewer

You can install FileViewer via bower:

  bower install ssh://git@stash.atlassian.com:7997/dv/dt-fileviewer.git

### 2. Add Dependencies

First of all, make sure that you have the following libraries on your page:


    <link rel="stylesheet" href="node_modules/aui-dist/aui/css/aui.css"/>
    <script src="node_modules/aui-dist/aui/js/aui-all.js"></script>
    <script src="node_modules/backbone/node_modules/underscore/underscore.js"></script>
    <script src="node_modules/backbone/backbone.js"></script>

Some of the viewers have other dependencies:

    <!--// document viewer-->
    <script src="node_modules/atlassian-fileviewer/vendor/pdfjs/pdf.js"></script>

    <!--// audio / video viewer-->
    <link href="node_modules/video.js/dist/cdn/video-js.css" rel="stylesheet"/>
    <script src="node_modules/video.js/dist/cdn/video.js"></script>
    <script src="node_modules/videojs-youtube/src/youtube.js"></script>

*Note*: Most versions of `AUI` are supported. However, make sure that you always include `AUI` before `backbone`.

### 3. Add FileViewer

Add 3 files to your page and you're ready to go:

    <link rel="stylesheet" href="node_modules/atlassian-fileviewer/dist/fileviewer.all.css"/>
    <script src="node_modules/atlassian-fileviewer/dist/fileviewer.all.js"></script>
    <script src="node_modules/atlassian-fileviewer/dist/fileviewer-templates.js"></script>

These files include FileViewer core, all viewers, all plugins, english translations, precompiled templates and styles. Custom builds will be covered later.

*NOTE*: Those files are meant to be used in development. In production you should use the matching `.min.js` files.

### 4. Create an Instance of FileViewer

To create an instance of FileViewer, all you have to do is the following:

    var viewer = new FileViewer({
        // enable minimode plugin
        enableMiniMode: true,

        // configuration necessary for pdf / document viewer
        assets: {
          'pdf-config': {
                workerSrc: 'node_modules/atlassian-fileviewer/vendor/pdfjs/pdf.worker.js',
                viewer: 'pdfjs/'
          }
        }
    });

There's other configuration options as well, check out the source code of `file-viewer.js` for now.

### 5. Show a File

First, tell FileViewer about the files it should be aware of:

    viewer.updateFiles([
        {
            id: 0,
            src: 'image.png',
            type: 'image/png',
            title: 'image'
        },
        {
            id: 1,
            src: 'document.pdf',
            type: 'application/pdf',
            title: 'document'
        }
    ]);

Now open the viewer and show a file:

    viewer.open();
    viewer.showFileWithId(0);

*NOTE*: There's a couple of additional methods to showFiles, for now check the source code.

et voilà - you are now showing the file in your browser.

## Enabling Plugins

As soon as plugins are included on the page, they register themselves with FileViewer. However, some of them need to be activated with certain configuration first. Here's an example:

    // enable minimode plugin
    var viewer = new FileViewer({ enableMiniMode: true });

## Disabling Viewers

By default, all viewers are enabled - even though might not yet be included on the page. To completely disable viewers, you need to overwrite the default configuration and mention all the viewers that you want to use instead:

    // disable document viewer
    var viewer = new FileViewer({
      viewers: ['image', 'video']
    });

Just make sure that the enabled viewers are either on the page or that you setup lazy loading for the viewers.

## Custom Builds

If you only need certain parts of FileViewer, you should avoid using the `.all` files and instead include the components you need directly.

FileViewer ships one core component and then one component for every plugin and viewer. Each of those components consists of three files - `component.js`, `component-templates.js` and `component-styles.css`. Make sure that you include all of them.

## Lazy Loading Viewers

To lazy load viewers, you need to pass a loader function at configuration:

    var viewer = new FileViewer({
        moduleStore: {
          get: function (title) {
            var d = new $.Deferred();
            // load code, then resolve deferred
            return d.promise;
          }
        }
    });

## Translations

Right now, FileViewer only supports english translations via `AJS.I18n.getText`. You can either let the server handle the replacement or include the file `translations.en.js` in the page.

## Generating Previews

When FileViewer can't preview a file, it can ask a different service to provide it
with a converted, viewable version of the file.

To do that, you have to configure a custom conversion service like this:

1. Make sure that FileViewer doesn't support the given mime type natively.

        new File({ type: 'not/supported' });

    **Note: FileViewer will only call the conversion service, if it doesn't support the given file type. Here's a [list of supported types](https://extranet.atlassian.com/display/FIL/Supported+filetypes?src=contextnavpagetreemode).**

2. Provide the two functions `isPreviewGenerated()` and `generatePreview()` in the configuration.

        var viewer = new FileViewer({
            isPreviewGenerated: isPreviewGenerated,
            generatePreview: generatePreview,
            // ....
        });

3. Implement `isPreviewGenerated()`:

        // This function gets the file model passed in and
        // has to return a promise (you can use jQuery promises).
        var isPreviewGenerated = function (file) {

            // if there is a preview, resolve with:
            // true, url of the converted file and
            // type of the converted file and optionally
            // additional overwrites for the converted file
            var overwrites = { src_hd: 'urlToHdVersion' };
            return $.when(true, urlOfConvertedFile, typeOfConvertedFile, overwrites);

            // if there is no converted file yet, simply resolve with false
            // to trigger the conversion
            return $.when(false);

            // if there is a failure, reject the promise
            return $.reject(new Error('conversion service not available'));
        };

4. Implement `generatePreview()`:

        // This function, too, gets the file model passed in and
        // has to return a promise (you can use jQuery promises).
        var generatePreview = function (file) {

            // once you're done, resolve with
            // the url of the converted file and
            // the type of the converted file and optionally
            // additional overwrites for the converted file
            var overwrites = { src_hd: 'urlToHdVersion' };
            return $.when(urlOfConvertedFile, typeOfConvertedFile, overwrites);

            // if there is a failure, reject the promise
            return $.reject(new Error('conversion service not available'));
        };

When isPreviewGenerated() returns false, a spinner will show up and wait for
generatePrevew() to resolve. Afterwards, the converted file will be shown.
