(function (FileViewer) {
    'use strict';

    // use FileViewer's internal module system
    var define  = FileViewer.define;
    var require = FileViewer.require;

define('video-view/button-presentation-mode', [], function () {
  'use strict';

  var videojs = window.videojs;

  var ButtonPresentationMode = videojs.Button.extend({
    init: function (opts) {
      this._fileViewer = opts.fileViewer;
      this._player = opts.player;

      videojs.Button.call(this, this._player, {
        el: videojs.Component.prototype.createEl(null, {
          className: 'vjs-presentation-control vjs-control',
          innerHTML: [
            '<div class="vjs-control-content">',
            '<span class="vjs-control-text">',
            'Presentation Mode',
            '</span>',
            '</div>'
          ].join(''),
          role: 'button',
          tabindex: 0,
          'aria-live': 'polite'
        })
      });

      this.on('click', this.changeMode);
    }
  });

  ButtonPresentationMode.asPlugin = function (fileViewer) {
    return function (options) {
      var player = this;
      var button = new ButtonPresentationMode({
        player: player,
        fileViewer: fileViewer
      });

      button.options = options;

      player.ready(function () {
        player.controlBar.addChild(button);
      });
    };
  };

  ButtonPresentationMode.prototype.changeMode = function () {
    if (this._fileViewer.isInMode('PRESENTATION')) {
      this._fileViewer.changeMode('BASE');
    } else {
      this._fileViewer.changeMode('PRESENTATION');
    }
  };

  return ButtonPresentationMode;
});

define('video-view/button-toggle-hd', [
  'template-store-singleton',
  'jquery'
], function (
  templateStore,
  $
) {
  'use strict';

  var videojs = window.videojs;

  var ButtonToggleHd = videojs.Button.extend({
    init: function (opts) {
      this._fileViewer = opts.fileViewer;
      this._player = opts.player;
      this._isPlayingHd = opts.isPlayingHd;

      videojs.Button.call(this, this._player, {
        el: videojs.Component.prototype.createEl(null, {
          className: 'vjs-hd-control vjs-control',
          innerHTML: templateStore.get('avPlayer.customControlBarItem')({
            descr: 'Toggle HD/SD Quality'
          }),
          role: 'button',
          tabindex: 0,
          'aria-live': 'polite'
        })
      });

      this.on('click', this.toggleSource);
    }
  });

  ButtonToggleHd.asPlugin = function (fileViewer) {
    return function (options) {
      if (!options.src_hd) {
        return;
      }

      var player = this;
      var button = new ButtonToggleHd({
        player: player,
        fileViewer: fileViewer,
        isPlayingHd: options.hd_active
      });

      button.options = options;

      player.ready(function () {
        $(this.el()).addClass('vjs-hd-source');
        if (options.hd_active) {
          $(this.el()).addClass('vjs-hd-playing');
        }
        player.controlBar.addChild(button);
      });
    };
  };

  /**
   * Set preload attribute to make sure `loadedmetadata` is triggered
   */
  ButtonToggleHd.prototype.fixPreload = function () {
    var player = this._player;
    if (player.el().firstChild.preload === 'none') {
      player.el().firstChild.preload = 'metadata';
    }
  };

  /**
   * Update the players video and poster src and seek to position if given
   * @param {Object} options
   * @param {Number} options.position Time in seconds to start playing
   * @param {String} options.src Url of video source
   * @param {String} options.poster Url of video poster source
   */
  ButtonToggleHd.prototype.switchSource = function (options) {
    var player = this._player;
    var playerEl = $(player.el());
    var wasPaused = player.paused();
    var hasStarted = playerEl.hasClass('vjs-has-started');

    this.fixPreload();
    // Remove <source> to prevent playing both streams
    playerEl.find('source').remove();
    // Remove poster attribute so it doesn't show up when switching streams
    player.poster('');
    if (options.position) {
      // Hide <video> tag to prevent showing the new stream starting
      // at the beginning
      $(player.tag).css('display', 'none');
    }
    player.src(options.src);
    player.ready(function () {
      player.one('loadedmetadata', function () {
        player.poster(options.poster);
        if (hasStarted) {
          playerEl.addClass('vjs-has-started');
        }
      }.bind(player));
      if (options.position) {
        player.one('loadeddata', function () {
          player.currentTime(options.position);
          if (wasPaused) {
            player.pause();
          } else {
            player.play();
          }
        });
      }
      // Show <video> again when seeking is done
      player.one('seeked', function () {
        $(player.tag).css('display', 'block');
      });
    });
  };

  ButtonToggleHd.prototype.toggleSource = function () {
    var player = this._player;
    var playerEl = $(player.el());
    var options = {
      position: player.tech.currentTime(),
      src: '',
      poster: ''
    };

    if (this._isPlayingHd) {
      this._isPlayingHd = false;
      this._fileViewer.getStorage().setItem('videoQualityHd', false);
      options.src = this.options.src;
      options.poster = this.options.poster;
      playerEl.removeClass('vjs-hd-playing');
    } else {
      this._isPlayingHd = true;
      this._fileViewer.getStorage().setItem('videoQualityHd', true);
      options.src = this.options.src_hd;
      options.poster = this.options.poster_hd;
      playerEl.addClass('vjs-hd-playing');
    }
    this.switchSource(options);
  };

  return ButtonToggleHd;
});

define('video-view/button-youtube', [
  'template-store-singleton',
  'url'
], function (
  templateStore,
  urlHelper
) {
  'use strict';

  var videojs = window.videojs;

  var ButtonYoutube = videojs.Button.extend({
    init: function (opts) {
      this._fileViewer = opts.fileViewer;
      this._player = opts.player;
      videojs.Button.call(this, this._player, {
        el: videojs.Component.prototype.createEl(null, {
          className: 'vjs-youtube-control vjs-control',
          innerHTML: templateStore.get('avPlayer.customControlBarItem')({descr: 'Youtube'}),
          role: 'button',
          tabindex: 0,
          'aria-live': 'polite'
        })
      });

      this.on('click', this.openYoutubePage);
    }
  });

  ButtonYoutube.asPlugin = function (fileViewer) {
    return function (options) {
      var player = this;
      var button = new ButtonYoutube({
        player: player,
        fileViewer: fileViewer
      });
      player.ready(function () { player.controlBar.addChild(button); });
    };
  };

  ButtonYoutube.prototype.openYoutubePage = function () {
    var srcYoutube;
    this._fileViewer.analytics.send('files.fileviewer-web.file.gotoyoutube');
    this._player.pause();
    srcYoutube = urlHelper.addQueryParamToUrl(this._player.src(), {
      start: Math.round(this._player.currentTime()),
      autoplay: 1
    });
    window.open(srcYoutube, '_blank');
  };

  return ButtonYoutube;
});

define('video-view/plugin-progress-tooltips', [
  'template-store-singleton',
  'jquery'
], function (
  templateStore,
  $
) {
  'use strict';

  var PluginProgressTooltips = function (options) {
    var player = this;
    var init = function () {
      $('#vjs-tip').remove();
      var tipHtml = templateStore.get('avPlayer.progressTooltip')();
      $('.vjs-progress-control').after(tipHtml);

      $('.vjs-progress-holder').on('mousemove', function (event) {
        var minutes,
          seconds,
          seekBar,
          timeInSeconds,
          mousePosition;

        seekBar = player.controlBar.progressControl.seekBar;
        mousePosition = (event.pageX - $(seekBar.el()).offset().left) / seekBar.width();
        timeInSeconds = mousePosition * player.duration();

        if (timeInSeconds === player.duration()) {
          timeInSeconds = timeInSeconds - 0.1;
        }

        minutes = Math.floor(timeInSeconds / 60);
        seconds = Math.floor(timeInSeconds - minutes * 60);

        if (seconds < 10) {
          seconds = '0' + seconds;
        }

        $('#vjs-tip-inner').text([minutes, ':', seconds].join(''));

        var $el = $('.vjs-control-bar');
        var tooltipWidth = $('#vjs-tip').outerWidth();
        var left = event.pageX - $el.offset().left - Math.round(tooltipWidth / 2);
        $('#vjs-tip').css('left', left + 'px').css('display', 'block');
      });

      $('.vjs-progress-holder, .vjs-play-control').on('mouseout', function () {
        $('#vjs-tip').css('display', 'none');
      });
    };
    this.on('loadedmetadata', init);
  };

  return PluginProgressTooltips;
});

define('video-view', [
  'ajs',
  'backbone',
  'underscore',
  'jquery',
  'file',
  'BaseViewer',
  'instance-manager',
  'file-types',
  'template-store-singleton',
  'keyboard',
  'video-view/button-youtube',
  'video-view/button-toggle-hd',
  'video-view/button-presentation-mode',
  'video-view/plugin-progress-tooltips'
], function (
  AJS,
  Backbone,
  _,
  $,
  File,
  BaseViewer,
  InstanceManager,
  fileTypes,
  templateStore,
  keyboard,
  buttonYoutube,
  buttonToggleHd,
  buttonPresentationMode,
  pluginProgressTooltips
) {
  /*global videojs*/
  'use strict';

  var playerManager = new InstanceManager(videojs, function (player) {
    /*
    videojs players created for files with unsupported codecs
    throw an uncaught error when calling pause()
    see HC-19832 https://jira.atlassian.com/browse/HC-19832

    As soon as PR on video.js is merged we can update and removed the
    `player.tech` check here
    see https://github.com/videojs/video.js/pull/2676
    */
    player.tech && player.pause();
    player.dispose();
  });

  var VideoView = BaseViewer.extend({

    id: 'cp-video-preview',

    playerId: 'cp-video-player',

    events: {
      'click .vjs-poster' : 'togglePlay'
    },

    initialize: function () {
      BaseViewer.prototype.initialize.apply(this, arguments);
      var type     = this.model.get('type');

      this._paddingVertical  = 20;
      this._paddingScrollbar = 40;
      this._paddingNavArrows = 160;

      this._isPaused    = true;
      this._isVideo   = fileTypes.isVideo(type);
      this._isAudio   = fileTypes.isAudio(type);
      this._isYoutube   = fileTypes.isYoutube(type);
      this._isAutoPlay  = this._fileViewer.get('videoViewerAutoplay');
      this._maxWidth    = this._isVideo ? 1280 : 640;
      this._maxHeight   = this._isVideo ? 720 : 640;
      this._minWidth    = this._isVideo ? 160 : 120;
      this._minHeight   = this._isVideo ? 90  : 120;
      this._hasHdSource = this.model.get('src_hd') ? true : false;

      $(window).on(
        'resize.videojs',
        _.throttle(this._handleResize.bind(this), 50)
      );
      $(document).on(
        'keydown.videoView',
        keyboard.createConditionalKeyHandler(this._handleKeyboardControl.bind(this))
      );
    },

    teardown: function () {
      $(window).off('resize.videojs', this._handleResize);
      $(document).off('keydown.videoView');
      playerManager.destroy();
    },

    clickHitBackground: function (e) {
      var hitBackground = e.target.getAttribute('id') === this.id;
      return hitBackground;
    },

    setupMode: function (mode) {
      $('.cp-toolbar-layer').hide();
    },

    _handleKeyboardControl: function (e) {
      if (e.which === keyboard.keys.SPACE || e.which === keyboard.keys.ENTER) {
        this.togglePlay();
        e.preventDefault();
      }
    },

    render: function () {
      var type = this.model.get('type');

      this.$el.html(templateStore.get('avPlayer.playerBody')({
        src: this._getVideoSrc(),
        poster: this._getPosterSrc(),
        type: type,
        kind: this._isVideo ? 'video' : 'audio'
      }));

      playerManager.create(this.playerId, {
        plugins: this._registerPlugins(),
        controls: true,
        ytFullScreenControls: false,
        textTrackDisplay: false,
        errorDisplay: false,
        controlBar: {
          timeDivider: false,
          remainingTimeDisplay: false,
          liveDisplay: false,
          fullscreenToggle: false,
          muteToggle: false,
          volumeMenuButton: false,
          playbackRateMenuButton: true,
          subtitlesButton: false,
          captionsButton: false,
          chaptersButton: false
        },
        autoplay: this._isAutoPlay,
        preload: true,
        width: this._maxWidth,
        height: this._maxHeight
      }).then(this._setupPlayer.bind(this));

      return this;
    },

    play: function () {
      this._videoPlayer.play();
    },

    togglePlay: function () {
      if (this._videoPlayer && this._isPaused) {
        this._videoPlayer.play();
      } else if (this._videoPlayer) {
        this._videoPlayer.pause();
      }
    },

    // Set focus back to our main element if the active element
    // is the iFrame. This is to prevent losing focus to the youtube
    // embedded player.
    _refocusFileViewer: function () {
      if (document.activeElement.tagName === 'IFRAME') {
        var $container = $('#cp-container');
        $container.attr('tabindex', -1);
        $container.focus();
      }
    },

    handleResize: function () {
      this._handleResize();
    },

    _handleResize: function () {
      var containerWidth  = Math.max(
        (this.$el.width() - this._paddingNavArrows),
        this._minWidth
      );
      var containerHeight = Math.max(
        (this.$el.height() - this._paddingVertical),
        this._minHeight
      );

      var $videoEl  = $(this._videoPlayer.el());
      var videoWidth  = $videoEl.width();
      var videoHeight = $videoEl.height();

      var ratio  = Math.min(
        (containerWidth / videoWidth),
        (containerHeight / videoHeight)
      );
      var newWidth  = Math.min(this._maxWidth, (videoWidth * ratio));
      var newHeight = Math.min(this._maxHeight, (videoHeight * ratio));

      $videoEl.css('width', newWidth).css('height', newHeight);
    },

    _videoError: function () {
      var err = new Error('Media failed loading');

      err.title = AJS.I18n.getText('cp.error.media.default.header');
      err.description = this.model.get('src');
      err.icon = 'cp-multimedia-icon';

      if (!this.viewerReady) {
        $('#' + this.playerId).remove();
        this.trigger('viewerFail', err);
      }
    },

    _videoLoadedMetaData: function () {
      $('#' + this.playerId).addClass('vjs-ready');
      this.trigger('viewerReady');
      this.viewerReady = true;
    },

    _videoVolumeChange: function () {
      var storage = this._fileViewer.getStorage();
      storage.setItem('videoVolume', this._videoPlayer.volume());
    },

    _videoPlay: function () {
      this._refocusFileViewer();
      this._isPaused = false;
      $(this._videoPlayer.el()).removeClass('vjs-has-ended');
    },

    _videoPause: function () {
      this._refocusFileViewer();
      this._isPaused = true;
    },

    _videoEnded: function () {
      $(this._videoPlayer.el()).addClass('vjs-has-ended');
    },

    _restoreVolumeSetting: function () {
      var storage   = this._fileViewer.getStorage();
      var volumeLevel = parseFloat(storage.getItem('videoVolume'));
      if (!isNaN(volumeLevel)) {
        this._videoPlayer.volume(volumeLevel);
      }
    },

    _restoreVideoPosterIfNotSet: function () {
      var currentPoster = this._videoPlayer.poster();
      var posterSrc   = this._getPosterForCurrentQuality();
      if (this._isVideo && !currentPoster) {
        this._videoPlayer.poster(posterSrc);
      }
    },

    _handlePlayerReady: function () {
      this.viewerReady = false;
      this._restoreVolumeSetting();

      // VideoJS 4.12.15 adds tabindex=0 to control bar play button
      // Tabindex needs to be removed to prevent focus staying on the play button
      // and breaking keyboard control (SPACE) to play/pause.
      $(this._videoPlayer.el()).find('.vjs-control').each(function (idx, el) {
        $(el).removeAttr('tabindex');
      });

      // In case the poster was removed for autoplaying videos
      // it now is added again to be displayed after the video ended.
      this._videoPlayer.one('timeupdate',
        this._restoreVideoPosterIfNotSet.bind(this)
      );
      this._videoPlayer.on('loadedmetadata',
        this._videoLoadedMetaData.bind(this)
      );
      this._videoPlayer.on('volumechange',
        this._videoVolumeChange.bind(this)
      );
      this._videoPlayer.on('error',
        this._videoError.bind(this)
      );
      this._videoPlayer.on('play',
        this._videoPlay.bind(this)
      );
      this._videoPlayer.on('pause',
        this._videoPause.bind(this)
      );
      this._videoPlayer.on('ended',
        this._videoEnded.bind(this)
      );
    },

    _registerPlugins: function () {
      var pluginsObject = {};

      videojs.plugin('pluginProgressTooltips', pluginProgressTooltips);
      pluginsObject.pluginProgressTooltips = {};

      videojs.plugin('presentation', buttonPresentationMode.asPlugin(this._fileViewer));
      pluginsObject.presentation = {viewer: this};

      videojs.plugin('buttonToggleHd', buttonToggleHd.asPlugin(this._fileViewer));
      pluginsObject.buttonToggleHd = {
        src: this.model.get('src'),
        src_hd: this.model.get('src_hd') || '',
        poster: this.model.get('poster') || '',
        poster_hd: this.model.get('poster_hd') || this.model.get('poster') || '',
        hd_active: this._shouldUseHdSource()
      };

      if (this._isYoutube) {
        videojs.plugin('buttonYoutube', buttonYoutube.asPlugin(this._fileViewer));
        pluginsObject.buttonYoutube = {};
      }

      return pluginsObject;
    },

    _setUpAudio: function ($player) {
      $player.addClass('vjs-audio');
      if (!this.model.get('poster')) {
        // Set default poster for audio if none is provided
        $player.find('.vjs-poster').addClass('vjs-default-coverart');
      }
    },

    _setUpYoutube: function ($player) {
      // Disable the youtube-videojs' iframe blocker
      // to re-enable clicks to elements inside the youtube player.
      // Removing it completely is no option because the youtube plugin
      // fails on trying to remove the layer on player destruction.
      $('.iframeblocker').css('display', 'none');
    },

    _setupPlayer: function (player) {
      var $player = $('#' + this.playerId);
      this._videoPlayer = player;
      this._handleResize();

      if (this._isAudio) { this._setUpAudio($player); }
      if (this._isYoutube) { this._setUpYoutube($player); }

      player.ready(this._handlePlayerReady.bind(this));
    },

    _shouldUseHdSource: function () {
      if (!this._hasHdSource) { return false; }

      var playHdDefault = this._fileViewer.getConfig().videoDefaultQualityHd;
      var playHdUser    = this._fileViewer.getStorage().getItem('videoQualityHd');
      var playHdEnabled = typeof playHdUser === 'undefined' || playHdUser;

      var playHdSettings = (playHdDefault && playHdEnabled) !== false;

      return this._hasHdSource && playHdSettings;
    },

    _getPosterForCurrentQuality: function () {
      if (this._shouldUseHdSource()) {
        return this.model.get('poster_hd') || this.model.get('poster') || '';
      }
      return this.model.get('poster') || '';
    },

    _getPosterSrc: function () {
      // No poster for autoplaying videos to prevent flickering
      if (this._isAutoPlay && this._isVideo) {
        return '';
      }
      return this._getPosterForCurrentQuality();
    },

    _getVideoSrc: function () {
      if (this._shouldUseHdSource()) {
        return this.model.get('src_hd');
      }
      return this.model.get('src');
    }

  });

  return VideoView;
});

(function () {
  'use strict';
  var FileViewer = require('file-viewer');
  FileViewer.registerPlugin('autoplay-videos', function (fileViewer) {
    var totalFilesShown;
    fileViewer.on('fv.open', function () { totalFilesShown = 0; });
    fileViewer.on('fv.changeFile', function () {
      totalFilesShown++;
      fileViewer.set('videoViewerAutoplay', totalFilesShown === 1);
    });
  });
}());

}(function () {
  var FileViewer;

    if (typeof module !== "undefined" && ('exports' in module)) {
      FileViewer = require('./fileviewer.js');
    } else if (window.require) {
      FileViewer = window.FileViewer;
    } else {
      FileViewer = window.FileViewer;
    }

    return FileViewer;
}()));
