/**
 * @class JIRA.Triggers.SelectTriggerDialogPageView
 * View for the wizard page which allows selecting a trigger as part of adding a trigger.
 */
Backbone.define("JIRA.Triggers.SelectTriggerDialogPageView", Backbone.View.extend({
    events: {
        "click .trigger-type-info-box": "_openFeedbackDialog"
    },

    /**
     * Initialise the view.
     *
     * @classdesc
     * The representation of the trigger selection page in the Add Trigger Dialog.
     *
     * @constructs
     * @extends Backbone.view
     * @param {object} options.model an instance of JIRA.Triggers.SelectTriggerDialogPageModel.
     * @param {object} [options.dialog] an instance of JIRA.Triggers.AddTriggerDialogView.
     */
    initialize: function (options) {
        this.model = options.model;
        this.dialog = options.dialog;

        this._initFeedbackDialog(options);
        this._initFetchHandlers();
    },

    _initFeedbackDialog: function (options) {
        this.feedbackDialog = options.feedbackDialog || new JIRA.DevStatus.FeedbackDialog({
                summary: AJS.I18n.getText("admin.triggers.config.add-dialog.feedback.text")
            });
    },

    _initFetchHandlers: function () {
        this.listenTo(this.model, "webSudoWillShow", this._handleWebSudoWillShow);
        this.listenTo(this.model, "webSudoCancel", this._handleWebSudoCancel);
        this.listenTo(this.model, "selectionChanged", this._handleSelectionChanged);
    },

    show: function () {
        this.model.fetchTriggerTypeList()
            .done(_.bind(this._renderTriggerList, this))
            .fail(_.bind(this._handleFetchFailed, this));

        this.submitButton = this.dialog.addSubmitButton(
            AJS.I18n.getText('admin.triggers.config.add-dialog.add'),
            _.bind(this._handleTemplateSelectButtonClicked, this));
        this.dialog.addCancelLink();
    },

    getHeader: function () {
        return AJS.I18n.getText('admin.triggers.config.add-dialog.header');
    },

    _handleTemplateSelectButtonClicked: function () {
        if (this.model.get("selectedTrigger").configuration != null) {
            var configurationView = new JIRA.Triggers.ConfigurationDialogPageView({
                dialog: this.dialog,
                model: this.model,
                el: this.$el
            });
            this.dialog.goToNextPage(configurationView, null);

            return AJS.$.Deferred().resolve();
        } else if (this.model.get("selectedTrigger").diagnostics != null) {
            var diagnosticsView = new JIRA.Triggers.DiagnosticsDialogPageView({
                dialog: this.dialog,
                model: this.model,
                el: this.$el
            });
            this.dialog.goToNextPage(diagnosticsView, null);

            return AJS.$.Deferred().resolve();
        } else {
            return this.model.addTriggerType()
                .done(_.bind(this._handleTriggerAdded, this))
                .fail(_.bind(this._handleAddingTriggerFailed, this));
        }
    },

    _renderTriggerList: function (triggerTypes) {
        var self = this;
        this.$el.html(JIRA.Templates.Triggers.Add.renderTriggerSelectionPage({triggerTypes: triggerTypes}));
        if (triggerTypes.length == 0) {
            this.dialog.disableSubmitButton();
        }

        this._truncateDescriptions();
        this.$(".trigger-type").click(function () {
            var $this = AJS.$(this);
            self.model.changeSelectedTriggerType($this.data('moduleKey'));
        }).dblclick(function () {
            var $this = AJS.$(this);
            self.model.changeSelectedTriggerType($this.data('moduleKey'));
            self._handleTemplateSelectButtonClicked();
        });
        this._bindKeyboardEvents();
        this.$(".trigger-type:first-child").click();
        _.defer(function () {
            self.$(".trigger-types").focus();
        });
    },

    _handleSelectionChanged: function (selectedTriggerType) {
        var $newSelection = this.$("[data-module-key='" + selectedTriggerType.key + "']");
        $newSelection.addClass("selected").siblings().removeClass("selected");

        // if the is a diagnostics or configuration page, the submit button should say 'Next'
        if (selectedTriggerType.diagnostics != null || selectedTriggerType.configuration != null) {
            this.submitButton.text(AJS.I18n.getText('admin.triggers.config.add-dialog.next'));
        } else {
            this.submitButton.text(AJS.I18n.getText('admin.triggers.config.add-dialog.add'));
        }
    },

    _handleWebSudoWillShow: function () {
        this.dialog.hide();
    },

    _handleWebSudoCancel: function () {
        // We give up adding a trigger.
        this.dialog.closeDialog();
    },

    _handleFetchFailed: function () {
        this.dialog.disableSubmitButton();
        this.$el.html(
            AJS.messages.error({
                body: AJS.I18n.getText('admin.trigger.config.add-dialog.fetching.error.title'),
                closeable: false
            }));
    },

    _handleTriggerAdded: function () {
        this.dialog.closeDialog();
        this.remove();
    },

    _handleAddingTriggerFailed: function (xhr) {
        var hadError = JIRA.Triggers.AddTriggerCommon.handleAddingTriggerFailed(xhr, this.$el);

        var triggersList = this.$(".trigger-type-list");
        var errorContainer = this.$("#trigger-error-container");

        // If there was no previous error we should update the length of the triggers list so the scrolling keeps working nicely.
        if (!hadError) {
            triggersList.css("height", triggersList.height() - errorContainer.outerHeight());
        }

        this.model.changeSelectedTriggerType(this.$(".trigger-type:first-child").data('moduleKey'));
        this.$(".dialog-panel-body").scrollTop(0);

        this.dialog.enableSubmitButton();
    },

    _truncateDescriptions: function () {
        var locale = AJS.Meta.get("user-locale");
        var wrap = "word";
        if (locale === "ja_JP") {
            wrap = "letter";
        }
        this.$(".trigger-type-description").dotdotdot({
            'wrap': wrap,
            'lastCharacter': {
                'remove': [' ', ',', ';', '.', '!', '?', '。'],
                'noEllipsis': []
            }
        });
    },

    _bindKeyboardEvents: function () {
        var self = this;
        var $container = this.$(".trigger-types");

        $container.bind("keydown", function (e) {
            var enterKey = 13;
            if (e.keyCode == enterKey) {
                e.preventDefault();
                self._handleTemplateSelectButtonClicked();
            }

            var delta = self._getMoveDeltaForKey(e.which);
            if (delta) {
                self._moveSelection($container, delta);
                e.stopPropagation();
            }
        });
    },

    /**
     * Converts arrow directions into sibling movement amounts in a grid.
     * @param which - the code of the key pressed
     * @return {Number} the amount to move, or 0 if no move should occur for this key code
     */
    _getMoveDeltaForKey: function (which) {
        var COLUMNS = +2,
            ARROW_UP = (-1) * COLUMNS,
            ARROW_RIGHT = +1,
            ARROW_DOWN = +COLUMNS,
            ARROW_LEFT = -1;

        switch (which) {
            case 37:
                return ARROW_LEFT;
            case 39:
                return ARROW_RIGHT;
            case 38:
                return ARROW_UP;
            case 40:
                return ARROW_DOWN;
        }

        return 0;
    },

    /**
     * Move the selection in the template container forward or backward by a specified number of templates
     * @param container the template container jQuery object
     * @param delta - positive to move forward, negative to move backward
     */
    _moveSelection: function (container, delta) {
        var $results = container.find(".trigger-type");
        var $selected = $results.filter(".selected");
        var index = $results.index($selected) + delta;
        if (index < $results.length && index >= 0) {
            var $next = $results.eq(index);
            this.model.changeSelectedTriggerType($next.data('moduleKey'));

            this._scrollToSelectedElement($next);
        }
    },

    _scrollToSelectedElement: function ($selected) {
        var $itemContainer = AJS.$(".dialog-panel-body");
        var itemContainerTopPosition = $itemContainer.offset().top;
        var nextTopPosition = $selected.offset().top;
        var nextBottomPosition = nextTopPosition + $selected.outerHeight();
        var itemContainerBottomPosition = itemContainerTopPosition + $itemContainer.height();

        if (nextTopPosition < itemContainerTopPosition) {
            $itemContainer.scrollTop($itemContainer.scrollTop() - (itemContainerTopPosition - nextTopPosition));
        } else if (nextBottomPosition > (itemContainerTopPosition + $itemContainer.height())) {
            $itemContainer.scrollTop($itemContainer.scrollTop() + nextBottomPosition - itemContainerBottomPosition);
        }
    },

    _openFeedbackDialog: function (e) {
        e.preventDefault();
        this.feedbackDialog.show();
    }
}));
