/**
 * @class JIRA.Triggers.SelectTriggerDialogPageModel
 * Model used for the pages of the wizard for adding a trigger.
 */
Backbone.define('JIRA.Triggers.SelectTriggerDialogPageModel', Backbone.Model.extend({

    namedEvents: [
        "webSudoWillShow",
        "webSudoCancel",
        "selectionChanged"
    ],

    /**
     * Initialise the model.
     *
     * @classdesc
     * The Add Trigger Dialog model.
     *
     * @constructs
     * @extends Backbone.view
     * @param {object} options
     * @param {object} [options.listModel] the model of the configured transition triggers.
     * @param {string} [options.workflowName] the name of the current workflow.
     * @param {number} [options.actionId] the id of the current workflow transition.
     * @param {string} [options.workflowMode] the current workflow mode.
     */
    initialize: function (options) {
        this.listModel = options.listModel;
        this.workflowName = options.workflowName;
        this.actionId = options.actionId;
        this.workflowMode = options.workflowMode;
        this.set('definitionConfig', {});
    },

    /**
     * Retrieve the list of trigger types.
     */
    fetchTriggerTypeList: function () {
        var self = this;
        return AJS.$.ajax({
            url: this._getTriggerTypeListUrl()
        }).then(function (triggerTypes) {
            self.set('triggerTypes', triggerTypes);
        }).promise();
    },

    /**
     * Add an instance of the currently selected trigger to the current transition.
     */
    addTriggerType: function () {
        var self = this;

        var addTriggerPromise = JIRA.SmartAjax.makeWebSudoRequest({
            url: this._getTriggerConfigUrl(),
            type: "PUT",
            contentType: "application/json",
            data: JSON.stringify({
                triggerDefinitionKey: self.get("selectedTrigger").key,
                definitionConfig: self.get("definitionConfig")
            })
        }, {
            cancel: function (e) {
                e.preventDefault();
                self.trigger("webSudoCancel");
            },
            beforeShow: function () {
                self.trigger("webSudoWillShow");
            }
        }).promise();

        addTriggerPromise.done(function () {
            self.listModel.updateData();
        });

        return addTriggerPromise;
    },

    _getTriggerConfigUrl: function () {
        return AJS.format("{0}/rest/triggers/1.0/workflow/config?workflowName={1}&actionId={2}",
            AJS.contextPath(), encodeURIComponent(this.workflowName), encodeURIComponent(this.actionId));
    },

    _getTriggerTypeListUrl: function () {
        return AJS.format("{0}/rest/triggers/1.0/workflow/types?workflowName={1}&actionId={2}&workflowMode={3}",
            AJS.contextPath(),
            encodeURIComponent(this.workflowName),
            encodeURIComponent(this.actionId),
            encodeURIComponent(this.workflowMode));
    },

    changeSelectedTriggerType: function (triggerType) {
        var triggerTypes = this.get("triggerTypes");
        var selectedType = _.find(triggerTypes, function (toTest) {
            return triggerType === toTest.key;
        });
        selectedType.diagnostics = JIRA.Triggers.Diagnostics.getDiagnosticsFor(triggerType);
        selectedType.configuration = JIRA.Triggers.Configuration.getConfigurationFor(triggerType);

        this.set("selectedTrigger", selectedType);
        this.trigger("selectionChanged", selectedType);
    },

    setDefinitionConfig: function (config) {
        if (config === undefined) {
            config = {};
        }
        this.set('definitionConfig', config);
    }
}));
