/**
 * @class JIRA.Triggers.ConfigurationDialogPageView
 * View to represent the configuration page in the dialog wizard used to add a trigger.
 */
Backbone.define("JIRA.Triggers.ConfigurationDialogPageView", Backbone.View.extend({

    /**
     * Initialise the view.
     *
     * @classdesc
     * The representation of the Configuration page in the Add Trigger Dialog.
     *
     * @constructs
     * @extends Backbone.view
     * @param {object} options
     * @param {object} options.dialog an instance of JIRA.Triggers.AddTriggerDialogView.
     */
    initialize: function (options) {
        this.dialog = options.dialog;
    },

    show: function () {
        var selectedTrigger = this.model.get("selectedTrigger");

        this.$el.html(JIRA.Templates.Triggers.Add.renderConfigurationPage());
        // We checked before whether the configurationCallback is defined, so no need to check it here
        var triggerConfig = new selectedTrigger.configuration({
            name: selectedTrigger.name,
            key: selectedTrigger.key,
            description: selectedTrigger.description,
            icon: selectedTrigger.icon,
            container: this.$el.find("#configuration-content").get(),
            triggerId: this.model.get('triggerId'),
            workflowName: this.model.get('workflowName'),
            workflowMode: this.model.get('workflowMode'),
            actionId: this.model.get('actionId')
        });

        var submitBtnText = AJS.I18n.getText('admin.triggers.config.add-dialog.add');
        if (selectedTrigger.diagnostics != null) {
            submitBtnText = AJS.I18n.getText('admin.triggers.config.add-dialog.next');
        }
        this.submitButton = this.dialog.addSubmitButton(
            submitBtnText,
            _.bind(this._handleSubmitButtonClicked, this, triggerConfig.submitCallback, selectedTrigger));

        this.dialog.addCancelLink();

        triggerConfig.showConfig();
    },

    getHeader: function () {
        return AJS.I18n.getText('admin.trigger.config.viewdiagnostics.configurationfor',
            this.model.get("selectedTrigger").name);
    },

    _handleSubmitButtonClicked: function (callback, selectedTrigger) {
        var self = this;
        var finalResultPromise = new AJS.$.Deferred();
        var configSubmitPromise = callback();

        configSubmitPromise.fail(function () {
            finalResultPromise.fail();
        });
        configSubmitPromise.done(_.bind(function (definitionConfig) {
            self.model.setDefinitionConfig(definitionConfig);
            if (selectedTrigger.diagnostics != null) {
                var diagnosticsView = new JIRA.Triggers.DiagnosticsDialogPageView({
                    dialog: self.dialog,
                    model: self.model,
                    el: self.$el
                });
                this.dialog.goToNextPage(diagnosticsView, null);

                finalResultPromise.resolve();
            } else {
                finalResultPromise = self.model.addTriggerType()
                    .done(_.bind(self._handleTriggerAdded, self))
                    .fail(_.bind(self._handleAddingTriggerFailed, self));
            }
        }, this));

        return finalResultPromise.promise();
    },

    _handleAddTriggerButtonClicked: function () {
        return this.model.addTriggerType()
            .done(_.bind(this._handleTriggerAdded, this))
            .fail(_.bind(this._handleAddingTriggerFailed, this));
    },

    _handleTriggerAdded: function () {
        this.dialog.closeDialog();
        this.remove();
    },

    _handleAddingTriggerFailed: function (xhr) {
        JIRA.Triggers.AddTriggerCommon.handleAddingTriggerFailed(xhr, this.$el);

        this.dialog.enableSubmitButton();
    }
}));
