/**
 * @class JIRA.Triggers.AddTriggerDialogView
 * View to represent the dialog wizard to add a trigger.
 * This view controls the dialog itself as well as the functionality to navigate back and forth between dialog pages.
 */
Backbone.define("JIRA.Triggers.AddTriggerDialogView", Backbone.View.extend({

    /**
     * Initialise the view.
     *
     * @classdesc
     * The representation of the Add Trigger Dialog.
     *
     * @constructs
     * @extends Backbone.view
     * @param {object} options
     * @param {object} options.listModel the model of the configured transition triggers.
     * @param {object} options.model the model for this view.
     * @param {object} [options.firstPage] the first page of the dialog wizard.
     */
    initialize: function (options) {
        this.listModel = options.listModel;
        this.model = options.model || new JIRA.Triggers.AddTriggerDialogModel({
                workflowName: this.listModel.get('workflowName'),
                actionId: this.listModel.get('actionId'),
                workflowMode: this.listModel.get('workflowMode')
            });

        this._renderDialog();
        var firstPage = options.firstPage || this._getSelectTriggerPage();
        // pageStack represents the pages of the wizard which are added as soon a page is requested for the first time.
        // When going back to the previous page, the current page will be removed from the stack.
        this.pageStack = [];
        this._stackPage(firstPage);
        this._showPage();
    },

    _renderDialog: function () {
        this.dialog = this._createDialog();
        this.$dialog = this.dialog.popup.element;
        this.setElement(this.$dialog);

        this.dialog.addPanel("addTriggerPanel", "", "panel-body");
        this.dialog.show();
    },

    _hasPreviousPage: function () {
        return this.pageStack.length > 1;
    },

    _getCurrentPage: function () {
        return this.pageStack[this.pageStack.length - 1];
    },

    _stackPage: function (page) {
        this.pageStack.push(page);
    },

    _popPage: function () {
        this.pageStack.pop();
    },

    _createDialog: function () {
        return new AJS.Dialog({
            id: 'add-trigger-dialog',
            width: 800
        });
    },

    _getPane: function () {
        return this.dialog.getPanel(0);
    },

    _getSelectTriggerPage: function () {
        var selectTriggerPageModel = new JIRA.Triggers.SelectTriggerDialogPageModel({
            workflowName: this.listModel.get('workflowName'),
            actionId: this.listModel.get('actionId'),
            workflowMode: this.listModel.get('workflowMode'),
            listModel: this.listModel
        });
        var selectTriggerPageView = new JIRA.Triggers.SelectTriggerDialogPageView({
            model: selectTriggerPageModel,
            el: this.$dialog.find(".dialog-panel-body"),
            dialog: this
        });

        return {
            view: selectTriggerPageView,
            model: selectTriggerPageModel
        }
    },

    _showPage: function () {
        this._showLoadingIndicator();
        var newPage = this._getCurrentPage();
        this.setHeader(newPage.view.getHeader() || AJS.I18n.getText('admin.triggers.config.add-dialog.header'));

        // clean up previous buttons and links in the footer
        if (this.$submitButton) {
            this.$submitButton.remove();
        }
        if (this.$cancelLink) {
            this.$cancelLink.remove();
        }
        if (this.$backButton) {
            this.$backButton.remove();
        }

        if (this._hasPreviousPage()) {
            this.dialog.addButton(
                AJS.I18n.getText('admin.triggers.config.add-dialog.back'),
                _.bind(this._goToPreviousPage, this), "wizard-back-button");
            this.$backButton = this.$dialog.find(".wizard-back-button");
        }
        newPage.view.show();
        JIRA.trace("addtrigger.dialog.ready");
    },

    /**
     * Dictate the dialog to go to the next page for which a view and model are given.
     *
     * @param {JIRA.Triggers.WizardPage} nextView the view of the new dialog page.
     * @param {Object} nextModel the model of the next dialog page.
     */
    goToNextPage: function (nextView, nextModel) {
        this.pageStack.push({
            view: nextView,
            model: nextModel
        });
        this._showPage();
    },

    _goToPreviousPage: function () {
        if (this._hasPreviousPage()) {
            this._popPage();
            this._showPage();
        }
    },

    /**
     * Set the header of the dialog.
     *
     * @param title {string} the new title for the header of the dialog.
     */
    setHeader: function (title) {
        this.dialog.addHeader(title);
    },

    /**
     * Add a cancel link to the dialog.
     */
    addCancelLink: function () {
        var self = this;

        this.dialog.addCancel(AJS.I18n.getText('admin.triggers.config.add-dialog.cancel'), function (dialog) {
            dialog.hide();
            self.closeDialog();
        });
        this.$cancelLink = this.$dialog.find(".button-panel-cancel-link");
        this.$cancelLink.attr('tabindex', 102);
    },

    /**
     * Add a submit button to the dialog.
     *
     * @param title {string} the title of the new submit button.
     * @param callback {requestCallback} The callback that will be called when the new submit button is clicked.
     *      This callback should return a promise. When the button is clicked, a spinner will be shown next to the button until the promise is resolved.
     * @returns {Object} the newly added submit button.
     */
    addSubmitButton: function (title, callback) {
        var self = this;

        this.dialog.addButton(title, function () {
            var promise = callback();
            self._showSubmitSpinner();
            promise.always(function () {
                self._hideSubmitSpinner();
            });
        }, "select-trigger-button");
        this.$submitButton = this.$dialog.find(".select-trigger-button");
        this.$submitButton.attr('tabindex', 101);
        this.$submitButton.removeClass("button-panel-button").addClass("aui-button aui-button-primary");
        this.$submitButton.focus();

        return this.$submitButton;
    },

    /**
     * Hide the dialog.
     */
    hide: function () {
        this.dialog.hide();
    },

    _showSubmitSpinner: function () {
        if (!this.$submitButton) {
            return;
        }

        this.$submitButton.before(
            JIRA.Templates.Triggers.Common.renderSmallSpinner({
                id: "add-trigger-loading"
            })
        );
        this.disableSubmitButton();
    },

    _hideSubmitSpinner: function () {
        this.$("#add-trigger-loading").remove();
        this.enableSubmitButton();
    },

    /**
     * Disable the submit button on the dialog (if there is one).
     */
    disableSubmitButton: function () {
        if (this.$submitButton) {
            this.$submitButton.attr('disabled', 'disabled');
        }
    },

    /**
     * Enable the submit button on the dialog (if there is one).
     */
    enableSubmitButton: function () {
        if (this.$submitButton) {
            this.$submitButton.removeAttr('disabled');
            this.$("#add-trigger-loading").remove();
        }
    },

    _showLoadingIndicator: function () {
        this.disableSubmitButton();
        var targetPane = this._getPane();
        targetPane.html(JIRA.Templates.Triggers.Common.renderBigSpinner());
        var loadingIndicator = this.$dialog.find(".status-loading");
        loadingIndicator.spin('large');
        loadingIndicator.show();
    },

    /**
     * Close the dialog.
     * This will also trigger the list model of the trigger configuration page to update.
     */
    closeDialog: function () {
        this.listModel.updateData();
        this.dialog && this.dialog.remove();
        this.remove();
    }
}));
