/**
 * @class JIRA.Triggers.TriggersListView
 */
Backbone.define("JIRA.Triggers.TriggersListView", Backbone.View.extend({
    template: JIRA.Templates.Triggers.List.triggersList,
    loading: JIRA.Templates.Triggers.Common.renderBigSpinner,
    connError: JIRA.Templates.Triggers.List.connError,
    authError: JIRA.Templates.Triggers.List.authError,

    events: {
        "click button.open-trigger-selection-dialog": "_createAddTriggerView",
        "click .remove-trigger": "_handleRemove",
        "click .view-diagnostics": "_handleViewDiagnostics",
        "click #triggers-list-feedback-trigger": "_handleFeedback"
    },

    /**
     * @constructs
     * @extends Backbone.View
     * @param {Object} options options for setting up the view.
     */
    initialize: function (options) {
        this.options = _.defaults({}, options);
        this.model = this.options.model;
        this.listElement = this.options.listElement;

        this._initFeedbackDialog(options);
        this._initFetchHandlers();
    },

    /**
     * Renders a loading state (spinner)
     */
    renderLoadingState: function () {
        var container = this.listElement;
        container.html(this.loading());
        var loadingIndicator = this.listElement.find(".status-loading");
        loadingIndicator.spin('large');
        loadingIndicator.show();
        return this;
    },

    /**
     * Renders a successful result
     * @param {Object} result the result to render
     * @return {Object} self for chaining
     */
    renderSuccess: function (result) {
        var container = this.listElement;
        container.html(this.template({
            triggers: result.triggers,
            editable: this.model.get('editable')
        }));
        // Use tipsy on the warnings.
        AJS.$(container).find(".aui-icon-warning").tooltip({aria: true});
        // Update the count.
        AJS.$(container).parents("#workflow-transition-info").find("#triggers").find(".aui-badge").text(result.triggers.length);
        return this;
    },

    /**
     * Renders an authentication error
     * @param {Object} result the authorisation result to display
     * @return {Object} self for chaining
     */
    renderAuth: function (result) {
        var container = this.listElement;
        container.html(this.authError(result));
        return this;
    },

    /**
     * Renders when there's no connection
     * @return {Object} self for chaining
     */
    renderNoConnection: function () {
        var container = this.listElement;
        container.html(this.connError());
        return this;
    },

    /**
     * Checks if the element is present.
     * @return {boolean} if the element is present.
     */
    isPresent: function () {
        return !_.isUndefined(this.$el);
    },

    _initFeedbackDialog: function (options) {
        this.feedbackDialog = options.feedbackDialog || new JIRA.DevStatus.FeedbackDialog({
                summary: AJS.I18n.getText("admin.triggers.config.feedback.summary")
            });
    },

    _handleFeedback: function (e) {
        e.preventDefault();
        this.feedbackDialog.show();
    },

    _handleFetchSucceeded: function (result) {
        return this.renderSuccess(result);
    },

    _handleFetchFailed: function (result) {
        if (result.error.status === 401) {
            this.renderAuth(result);
        } else {
            this.renderNoConnection(); // Failure cases handled by soy.
        }
    },

    _handleRemove: function (e) {
        e.preventDefault();
        var $triggerRow = AJS.$(e.target).closest('.trigger-row');
        this.model.removeTrigger($triggerRow.attr('data-trigger-id'));
    },

    _handleViewDiagnostics: function (e) {
        e.preventDefault();
        var $triggerRow = AJS.$(e.target).closest('.trigger-row');
        var triggerId = $triggerRow.attr('data-trigger-id');

        var triggerInfo = this.model.getTriggerInfo(parseInt(triggerId)) || {};

        var diagnosticModel = new JIRA.Triggers.DiagnosticsPanelModel({
            triggerId: triggerId,
            name: triggerInfo.name,
            key: triggerInfo.triggerKey,
            icon: triggerInfo.icon,
            description: triggerInfo.description,
            actionId: this.model.get('actionId'),
            workflowName: this.model.get('workflowName'),
            workflowMode: this.model.get('workflowMode')
        });
        var dialogView = new JIRA.Triggers.DiagnosticsDialogView({model: diagnosticModel});
        dialogView.render();
    },

    _initFetchHandlers: function () {
        this.listenTo(this.model, "fetchRequested", this.renderLoadingState);
        this.listenTo(this.model, "fetchSucceeded", this._handleFetchSucceeded);
        this.listenTo(this.model, "fetchFailed", this._handleFetchFailed);
        this.listenTo(this.model, "removeFailed", this._handleFetchFailed);
        this.listenTo(this.model, "triggerTypeAvailabilityUpdatedSucceeded", this._handleTriggerTypeAvailabilityUpdateSucceeded);
        this.listenTo(this.model, "triggerTypeAvailabilityUpdatedFailed", this._handleTriggerTypeAvailabilityUpdateFailed)
    },

    _handleTriggerTypeAvailabilityUpdateSucceeded: function (result) {
        this._getAddTriggerButton().attr("aria-disabled", result != true);
    },

    _handleTriggerTypeAvailabilityUpdateFailed: function () {
        // If retrieving the update failed for some reason, we'll just enable to Add Trigger button anyway.
        this._getAddTriggerButton().removeAttr('aria-disabled');
    },

    _createAddTriggerView: function () {
        if (this._getAddTriggerButton().attr('aria-disabled') !== "true") {
            new JIRA.Triggers.AddTriggerDialogView({listModel: this.model});
        }
    },

    _getAddTriggerButton: function () {
        return this.$el.find("button.open-trigger-selection-dialog");
    }
}));
