/**
 * @class JIRA.Triggers.TriggersListModel
 * Model for the triggers list. Note that this isn't a simple list model. It
 * will treat the entire list as a "single item", and the view will also render
 * the entire list as a single item.
 */
Backbone.define('JIRA.Triggers.TriggersListModel', Backbone.Model.extend({
    typeId: undefined,
    properties: ['data', 'actionId', 'workflowName', 'workflowMode', 'editable'],

    namedEvents: [
        // events related to detailed data rest call
        "fetchRequested",
        "fetchFailed",
        "fetchSucceeded",
        // events related to remove data rest call
        "removeRequested",
        "removeFailed",
        "removeSucceeded",
        // events related to auth status rest call
        "fetchAuthRequested",
        "fetchAuthFailed",
        "fetchAuthSucceeded",
        // events related to trigger types rest call
        "triggerTypeAvailabilityUpdatedSucceeded",
        "triggerTypeAvailabilityUpdatedFailed"
    ],

    /**
     * Updates the data from the back-end.
     */
    updateData: function () {
        this.trigger("fetchRequested", this.get('workflowName'), this.get('actionId'));

        this._handleFetch(AJS.$.ajax({
            url: this._getTriggerUrl(),
            data: {
                workflowName: this.get('workflowName'),
                actionId: this.get('actionId'),
                workflowMode: this.get('workflowMode')
            }
        }).promise());

        this._fetchTriggerTypes();
    },

    /**
     * Removes the trigger from the triggers list on the current transition.
     * @param {number} triggerId the trigger ID to remove.
     */
    removeTrigger: function (triggerId) {
        this._handleRemove(AJS.$.ajax({
            url: this._getTriggerConfigurationUrl(triggerId),
            type: 'DELETE'
        }).promise());
    },

    _fetchTriggerTypes: function () {
        this._handleFetchTriggerTypes(AJS.$.ajax({
            url: this._getTriggerTypeListUrl()
        }).promise());
    },

    /**
     * Gets the trigger information for the given trigger ID.
     * @param {number} triggerId the trigger ID to get the key for.
     */
    getTriggerInfo: function (triggerId) {
        var trigger = _.find(this.get('data').triggers, function (item) {
            return item.id === triggerId;
        });
        if (!_.isUndefined(trigger)) {
            return trigger;
        }
        return undefined;
    },

    _getTriggerUrl: function () {
        return AJS.contextPath() + '/rest/triggers/1.0/workflow/info';
    },

    _getTriggerConfigurationUrl: function (triggerId) {
        var url = AJS.contextPath() + '/rest/triggers/1.0/workflow/config?';
        url += 'workflowName=' + encodeURIComponent(this._getWorkflowName());
        url += '&actionId=' + encodeURIComponent(this._getActionId());
        url += '&workflowMode=' + encodeURIComponent(this.get('workflowMode'));
        if (triggerId) {
            url += '&triggerId=' + encodeURIComponent(triggerId);
        }
        return url;
    },

    _getTriggerTypeListUrl: function () {
        return AJS.format("{0}/rest/triggers/1.0/workflow/types?workflowName={1}&actionId={2}&workflowMode={3}",
            AJS.contextPath(),
            encodeURIComponent(this.get('workflowName')),
            encodeURIComponent(this.get('actionId')),
            encodeURIComponent(this.get('workflowMode')));
    },

    _handleFetch: function (promise) {
        var self = this;

        promise.done(function (result) {
            self._fireSuccessResult(result);
        }).fail(function (xhr, status) {
            self._fireFailResult(xhr);
        });
    },

    _handleRemove: function (promise) {
        var self = this;

        promise.done(function (result) {
            self._handleSuccessfulRemove();
        }).fail(function (xhr, status) {
            if (xhr.status !== 404) {
                self._fireFailResult(xhr, 'removeFailed');
            } else {
                self._handleSuccessfulRemove();
            }
        });
    },

    _handleFetchTriggerTypes: function (promise) {
        var self = this;

        promise.done(function (result) {
            self.trigger("triggerTypeAvailabilityUpdatedSucceeded", result.length > 0);
        }).fail(function () {
            self.trigger("triggerTypeAvailabilityUpdatedFailed");
        });
    },

    _handleSuccessfulRemove: function () {
        this.updateData();
        this._fireSuccessResult(null, 'removeSucceeded');
    },

    _getCurrentUrl: function () {
        return encodeURIComponent(window.location.pathname + window.location.search);
    },

    _fireFailResult: function (xhr, eventName) {
        eventName = eventName || 'fetchFailed';
        var mdl = {error: xhr, destination: this._getCurrentUrl()};
        this.set('data', mdl);
        this.trigger(eventName, mdl);
    },

    _fireSuccessResult: function (result, eventName) {
        eventName = eventName || 'fetchSucceeded';
        if (result) {
            var newResult = _.map(result, function (item) {
                return _.extend({}, item, {
                    diagnosticsModule: JIRA.Triggers.Diagnostics.getDiagnosticsFor(item.key)
                });
            });
            var data = {triggers: newResult};
            this.set("data", data);
            this.trigger(eventName, data);
        } else {
            this.trigger(eventName);
        }
    },

    _getWorkflowName: function () {
        return this.get('workflowName');
    },

    _getActionId: function () {
        return this.get('actionId');
    }
}));
