AJS.test.require("com.atlassian.jira.plugins.jira-transition-triggers-plugin:triggers-list-common-resources");

(function () {

    module("JIRA.Triggers.TriggersListModel", {
        setup: function () {
            this.sandbox = sinon.sandbox.create();
            this.fixture = jQuery("#qunit-fixture");
            this.stubEverything();
        },
        teardown: function () {
            this.sandbox.restore();
        },
        stubEverything: function () {
            this.sandbox.stub(AJS.$, "ajax");
        },
        createModel: function (workflowName, actionId, workflowMode) {
            var model = new JIRA.Triggers.TriggersListModel({
                workflowName: workflowName,
                actionId: actionId,
                workflowMode: workflowMode
            });
            this.triggerSpy = this.sandbox.spy(model, "trigger");
            this.updateDataSpy = this.sandbox.spy(model, "updateData");
            return model;
        },
        mockAjax: function () {
            var deferred = AJS.$.Deferred();
            AJS.$.ajax.returns(deferred);
            return deferred;
        }
    });

    test("Model initialises with no data", function () {
        var model = this.createModel("workflow", 1, "live");
        var deferred = this.mockAjax();
        model.updateData();
        deferred.resolve([]);
        equal(model.get("data").triggers.length, 0, "Empty triggers");
        ok(this.triggerSpy.withArgs("fetchSucceeded").calledOnce, "fetchSucceeded was called");
    });

    test("Model initialises with no data", function () {
        var model = this.createModel("workflow", 1, "live");
        var deferred = this.mockAjax();
        model.updateData();
        deferred.resolve([{things: "stuff"}]);
        equal(model.get("data").triggers.length, 1, "One trigger");
        equal(model.get("data").triggers[0].things, "stuff", "Trigger data matches response");
        ok(this.triggerSpy.withArgs("fetchSucceeded").calledOnce, "fetchSucceeded was called");
    });

    test("Model asks for live workflow", function () {
        var model = this.createModel("workflow", 1, "live");
        var deferred = this.mockAjax();
        model.updateData();
        deferred.resolve([]);
        var ajaxCall = AJS.$.ajax.firstCall;
        equal(ajaxCall.args[0].data.workflowName, "workflow", "workflow name was correct");
        equal(ajaxCall.args[0].data.actionId, 1, "action ID was correct");
        equal(ajaxCall.args[0].data.workflowMode, "live", "workflow mode was correct");
        ok(this.triggerSpy.withArgs("triggerTypeAvailabilityUpdatedSucceeded").calledOnce, "triggerTypeAvailabilityUpdatedSucceeded was called");
    });

    test("Model asks for draft workflow", function () {
        var model = this.createModel("workflow", 1, "draft");
        var deferred = this.mockAjax();
        model.updateData();
        deferred.resolve([]);
        var ajaxCall = AJS.$.ajax.firstCall;
        equal(ajaxCall.args[0].data.workflowName, "workflow", "workflow name was correct");
        equal(ajaxCall.args[0].data.actionId, 1, "action ID was correct");
        equal(ajaxCall.args[0].data.workflowMode, "draft", "workflow mode was correct");
        ok(this.triggerSpy.withArgs("triggerTypeAvailabilityUpdatedSucceeded").calledOnce, "triggerTypeAvailabilityUpdatedSucceeded was called");
    });

    test("Model cannot contact the server", function () {
        var model = this.createModel("workflow", 1, "live");
        var deferred = this.mockAjax();
        model.updateData();
        deferred.reject({status: 404});
        equal(model.get("data").error.status, 404, "Empty triggers");
        ok(this.triggerSpy.withArgs("fetchFailed").calledOnce, "fetchFailed was called");
    });

    test("Auth exception when contacting server", function () {
        var model = this.createModel("workflow", 1, "live");
        var deferred = this.mockAjax();
        model.updateData();
        deferred.reject({status: 401});
        equal(model.get("data").error.status, 401, "Empty triggers");
        ok(this.triggerSpy.withArgs("fetchFailed").calledOnce, "fetchFailed was called");
    });

    test('Removing triggers successful triggers data update', function () {
        var model = this.createModel('workflow', 1, "live");
        var removeDeferred = this.mockAjax();
        model.removeTrigger(1); // removeDeferred provided to the model here

        // need to stub updateDeferred _after_ the removeDeferred has been provided to the model code, as mockAjax will
        // override the previous stub. The proper way to solve it is to upgrade sinon and use onCall() to stub
        // subsequent calls to ajax
        var updateDeferred = this.mockAjax();
        removeDeferred.resolve('OK'); // remove AJAX answer
        updateDeferred.resolve([{name: 'Some trigger'}]); // update AJAX answer

        equal(model.get('data').triggers.length, 1, 'One trigger in data');
        equal(model.get("data").triggers[0].name, 'Some trigger', 'Trigger data matches response');
        ok(this.updateDataSpy.calledOnce, 'updateData was called');
        ok(this.triggerSpy.withArgs('fetchSucceeded').calledOnce, 'fetchSucceeded was called');
    });

    test('Removing triggers successful triggers data update, which fails', function () {
        var model = this.createModel('workflow', 1, "live");
        var removeDeferred = this.mockAjax();
        model.removeTrigger(1); // removeDeferred provided to the model here

        // need to stub updateDeferred _after_ the removeDeferred has been provided to the model code, as mockAjax will
        // override the previous stub. The proper way to solve it is to upgrade sinon and use onCall() to stub
        // subsequent calls to ajax
        var updateDeferred = this.mockAjax();
        removeDeferred.resolve('OK'); // remove AJAX answer
        updateDeferred.reject({status: 400}); // update AJAX answer

        equal(model.get("data").error.status, 400, "No triggers because of error response");
        ok(this.updateDataSpy.calledOnce, 'updateData was called');
        ok(this.triggerSpy.withArgs('fetchFailed').calledOnce, 'fetchFailed was called');
    });

    test('Removing triggers fails', function () {
        var model = this.createModel('workflow', 1, "live");
        var removeDeferred = this.mockAjax();
        model.removeTrigger(1);

        // remove request fails
        removeDeferred.reject({status: 400});

        equal(model.get("data").error.status, 400, "No triggers because of error response");
        equal(this.updateDataSpy.callCount, 0, 'updateData was never called');
        ok(this.triggerSpy.withArgs('removeFailed').calledOnce, 'removeFailed was called');
    });

    test('Removing non existing trigger is successful', function () {
        var model = this.createModel('workflow', 1, "live");
        var removeDeferred = this.mockAjax();
        model.removeTrigger(1);

        // remove request fails
        removeDeferred.reject({status: 404});

        equal(model.get("data").error.status, 404, "No triggers because of error response");
        ok(this.updateDataSpy.calledOnce, 'updateData was called');
        ok(this.triggerSpy.withArgs('removeSucceeded').calledOnce, 'removeSucceeded was called');
    });

    test("Trigger type list initialises with data", function () {
        var model = this.createModel("workflow", 1);
        var deferred = this.mockAjax();
        model._fetchTriggerTypes();
        deferred.resolve(["1", "2"]);
        ok(this.triggerSpy.withArgs("triggerTypeAvailabilityUpdatedSucceeded", true).calledOnce, "triggerTypeAvailabilityUpdatedSucceeded was called");
    });

    test("Trigger type list initialises without data", function () {
        var model = this.createModel("workflow", 1);
        var deferred = this.mockAjax();
        model._fetchTriggerTypes();
        deferred.resolve([]);
        ok(this.triggerSpy.withArgs("triggerTypeAvailabilityUpdatedSucceeded", false).calledOnce, "triggerTypeAvailabilityUpdatedSucceeded was called");
    });

    test("Trigger type list can not be retrieved", function () {
        var model = this.createModel("workflow", 1);
        var deferred = this.mockAjax();
        model._fetchTriggerTypes();
        deferred.reject({status: 404});
        ok(this.triggerSpy.withArgs("triggerTypeAvailabilityUpdatedFailed").calledOnce, "triggerTypeAvailabilityUpdatedFailed was called");
    });

    test("Trigger Info gets appropriate information about a trigger", function () {
        var model = this.createModel("workflow", 1);
        model.set('data', {
            triggers: [{
                id: "key-1",
                val: "val-1"
            }, {
                id: "key-2",
                val: "val-2"
            }]
        });
        equal(model.getTriggerInfo("key-2").val, "val-2");
        equal(model.getTriggerInfo("key-3"), undefined);
    });
})();
