AJS.test.require([
    "jira.webresources:userprofile",
    "jira.webresources:jira-fields",
    "com.atlassian.jira.jira-tzdetect-plugin:tzdetect-userprofile-component"
], function(){
    "use strict";

    var $ = require('jquery');
    var Events = require('jira/util/events');
    var Types = require('jira/util/events/types');
    var contextPath = require('wrm/context-path')();

    var tzTestUtils = {
        saveProps: function (obj) {
            var propList = Array.prototype.slice.call(arguments, 1);
            if (!propList.length) {
                return;
            }
            var props = {};
            for (var i = 0, ii = propList.length; i < ii; i++) {
                props[propList[i]] = obj[propList[i]];
            }
            obj.__tzSavedProps = props;
        },
        restoreProps: function (obj) {
            if (!obj.__tzSavedProps) {
                return;
            }
            for (var prop in obj.__tzSavedProps) {
                obj[prop] = obj.__tzSavedProps[prop];
            }
            delete obj.__tzSavedProps;
        }
    };

    module("tzdetect-userprofile", {
        setup: function () {
            this.context = AJS.test.mockableModuleContext();
            this.checker = this.context.require('jira-tzdetect/userprofile-checker');
            tzTestUtils.saveProps(this.checker, "hash", "zoneId", "zoneParts");
            this.$link = $("<a/>", {id: "edit_prefs_lnk"}).appendTo("#qunit-fixture");
        },
        teardown: function () {
            tzTestUtils.restoreProps(this.checker);
        }
    });

    /** checkZoneId() **/

    test("checker does nothing when hash is missing or invalid", function () {
        var stubTracking = this.stub(this.checker, "track");
        var stubPrefs = this.stub(this.checker, "openPrefsDialog");

        this.checker.hash = "";
        this.checker.checkZoneId();
        equal(stubPrefs.callCount, 0, "Missing hash does nothing");

        this.checker.hash = "unrelated";
        this.checker.checkZoneId();
        equal(stubTracking.callCount, 0, "Unrelated hash doesn't track");
        equal(stubPrefs.callCount, 0, "Unrelated hash does nothing");
    });

    test("'zone-ignore' tracks but does nothing else", function () {
        var stubTracking = this.stub(this.checker, "track");
        var stubPrefs = this.stub(this.checker, "openPrefsDialog");

        this.checker.hash = "zone-ignore";
        this.checker.checkZoneId();
        equal(stubTracking.callCount, 1, "'Ignore' call is tracked");
        ok(stubTracking.calledWith("ignore"), "'Ignore' call is tracked correctly");
        equal(stubPrefs.callCount, 0, "Preferences dialog doesn't open");
    });

    test("'zone-other' tracks and opens preferences", function () {
        var stubTracking = this.stub(this.checker, "track");
        var stubPrefs = this.stub(this.checker, "openPrefsDialog");

        this.checker.hash = "zone-other";
        this.checker.checkZoneId();
        equal(stubTracking.callCount, 1, "'Other' call is tracked");
        ok(stubTracking.calledWith("other"), "'Other' call is tracked correctly");
        equal(stubPrefs.callCount, 1, "Preferences dialog is opened");
    });

    test("Valid zone ID tracks and opens preferences", function () {
        var stubTracking = this.stub(this.checker, "track");
        var stubPrefs = this.stub(this.checker, "openPrefsDialog");

        this.checker.hash = "zone-Australia/Sydney";
        this.checker.checkZoneId();
        equal(stubTracking.callCount, 1, "Zone call is tracked");
        ok(stubTracking.calledWith("prefillzone", {zoneId: "Australia/Sydney"}), "Zone call is tracked correctly");
        equal(stubPrefs.callCount, 1, "Preferences dialog is opened");
    });

    test("Valid zone ID splits into correct parts", function () {
        this.stub(this.checker, "track");
        this.stub(this.checker, "openPrefsDialog");

        this.checker.hash = "zone-America/Argentina/Buenos_Aires";
        this.checker.checkZoneId();
        deepEqual(this.checker.zoneParts, ["America", "Argentina/Buenos_Aires"], "3-part zone IDs are split into region/zone");
    });


    /** openPrefsDialog() **/

    test("Valid zone ID adds preferences dialog callback", function () {
        var stub = this.stub(this.checker, "addDialogCallback");
        this.stub($.fn, "click");

        this.checker.zoneId = "Australia/Sydney";
        this.checker.openPrefsDialog();
        equal(stub.callCount, 1, "Valid zone ID adds preferences dialog callback");
    });

    test("zone keyword only opens dialog", function () {
        var stub = this.stub(this.checker, "addDialogCallback");
        this.stub($.fn, "click");

        this.checker.zoneId = "set";
        this.checker.openPrefsDialog();
        equal(stub.callCount, 0, "Zone keyword doesn't add preferences dialog callback");
    });


    /** addDialogCallback() **/

    test("dialog callback only fires once at the right time", function () {
        var stub = this.stub(this.checker, "setCascadingSelectValues");
        var $doc = $(document);
        var eventName = "dialogContentReady";
        var trigger = "#edit_prefs_lnk";
        var fakeDialog = {
            get$popupContent: function () {
                return $("#does-not-exist");
            },
            options: {trigger: ""}
        };
        this.checker.addDialogCallback();

        fakeDialog.options.trigger = "#something-else";
        $doc.trigger(eventName, fakeDialog);
        equal(stub.callCount, 0, "Irrelevant trigger doesn't fire callback");

        fakeDialog.options.trigger = trigger;
        $doc.trigger(eventName, fakeDialog);
        equal(stub.callCount, 1, "Correct trigger fires callback");

        $doc.trigger(eventName, fakeDialog);
        equal(stub.callCount, 1, "Correct trigger again doesn't re-fire callback");
    });


    /** setCascadingSelectValues() **/

    asyncTest("setting time zone select values", function () {
        var $fixture = $("#qunit-fixture");
        var url = contextPath + "/secure/UpdateUserPreferences!default.jspa?" +
            "username=admin&inline=true&decorator=dialog&os_username=admin&os_password=admin";
        var checker = this.checker;

        $fixture.load(url, function () {
            Events.trigger(Types.NEW_CONTENT_ADDED, [$fixture]);
            start();

            var $parent = $("#timeZoneRegion");
            var $child = $("#defaultUserTimeZone");
            checker.zoneParts = ["America", "Argentina/Buenos_Aires"];
            checker.setCascadingSelectValues();

            equal($parent.val(), "America", "Parent select has correct value");
            equal($child.val(), "America/Argentina/Buenos_Aires", "Child select has correct value");
        });
    });
});

