define('jira-tzdetect/banner', [
    'jira-tzdetect/banner/templates',
    'jira/util/formatter',
    'jira/ajs/ajax/smart-ajax',
    'jira/analytics',
    'jira/flag',
    'wrm/context-path',
    'wrm/data',
    'jira/jquery/deferred',
    'jquery',
    'underscore'
], function(
    Templates,
    formatter,
    SmartAjax,
    analytics,
    flag,
    wrmContextPath,
    wrmData,
    Deferred,
    $,
    _
) {

    function minsToMillis(mins) {
        return (+mins || 0) * 60 * 1000;
    }

    function millisToMins(millis) {
        return Math.floor((+millis || 0) / 1000 / 60);
    }

    function normalLeftClickOnly(handler) {
        return function handleLeftClick(e) {
            if (e.which === 1 && !e.shiftKey && !e.ctrlKey && !e.metaKey) {
                handler.apply(this, arguments);
            }
        };
    }

    var restPath = wrmContextPath() + '/rest/tzdetect/1';

    var banner = {
        $tzFlag: null,
        $link: null,
        $dropdown: null,
        prefs: {},
        detected: {}
    };

    banner.init = function () {
        if (banner.$tzFlag) {
            return;
        }
        banner.detectTimezoneOffsets();
        // only show banner if we found the current timezone information.
        if (banner.detected.janOffset != null && banner.detected.julyOffset != null) {
            // Show the banner if the offsets are different. Dismissals are handled by the flags code
            var zonesDiffer = banner.detected.janOffset != banner.prefs.janOffset || banner.detected.julyOffset != banner.prefs.julyOffset;
            if (zonesDiffer) {
                 banner.create();
            }
        }
    };

    banner.detectTimezoneOffsets = function () {
        banner.prefs = banner.getPreferences();
        if (!banner.prefs.tzid) {
            return;
        }

        // Work out the user's GMT offsets for comparison
        banner.detected.janOffset = banner.getTzOffset(0);
        banner.detected.julyOffset = banner.getTzOffset(6);
    };

    banner.create = function () {
        var link = banner.profileLink();
        var msg = formatter.I18n.getText('tz.no.match', banner.prefs.tzname);

        var updateLinkText = formatter.I18n.getText('tz.update');
        var content = Templates.banner({
            message: msg,
            updateLinkText: updateLinkText,
            updateLink: link + '#zone-set'
        });

        var offsetData = [
            minsToMillis(banner.detected.janOffset),
            minsToMillis(banner.detected.julyOffset)
        ].join(',');

        var tzFlag = flag.showInfoMsg(null, content, { dismissalKey: 'com.atlassian.jira.tzdetect.' + offsetData});
        if (tzFlag) {
            banner.$tzFlag = $(tzFlag);
            // Load list of matching zones from the server in the background
            banner.getPotentialZones();

            banner.$tzFlag.on('aui-flag-close', function () {
                banner.track('clicked.nothanks');
                banner.$tzFlag = undefined; // aui should be handling the actual removal.
            });

            banner.$link = banner.$tzFlag.find('.tz-yes');
            banner.$link.click(normalLeftClickOnly(function (e) {
                e.preventDefault();
                banner.getPotentialZones().then(banner.handleZoneData);
            }));

            banner.track('shown');
        }
    };

    banner.track = function (value, attrs) {
        var data = {
            name: 'tzdetect.banner.' + value
        };
        if (attrs) {
            data.data = attrs;
        }
        analytics.send(data);
    };

    banner.profileLink = function () {
        return wrmContextPath() + '/secure/ViewProfile.jspa';
    };

    banner.redirect = function (url) {
        window.location = url;
    };

    banner.getPreferences = function () {
        var prefix = 'tzdetect.pref.';
        var keys = ['tzid', 'tzname', 'janOffset', 'julyOffset'];
        var prefs = {};
        var i = keys.length;
        var key, val;
        while (i--) {
            key = keys[i];
            val = wrmData.claim(prefix + key);
            // Convert millisecond-based offsets into minutes
            if (key.indexOf('Offset') > -1) {
                val = millisToMins(val);
            }
            prefs[key] = val;
        }
        return prefs;
    };

    banner.handleZoneData = function (data) {
        var zones = banner.filterZoneData(data.zones);
        var len = zones.length;
        // Build a key/value list of regions for later use
        zones.regions = {};
        var i = data.regions.length, region;
        while (i--) {
            region = data.regions[i];
            zones.regions[region.key] = region.displayName;
        }

        // Sort zones by region first, then city
        zones.sort(function (a, b) {
            var ra = a.regionKey;
            var rb = b.regionKey;
            if (ra == rb) {
                return a.city < b.city ? -1 : 1;
            }
            return ra < rb ? -1 : 1;
        });

        banner.track('clicked.update', {matchingZoneCount: len});

        // Somehow we didn't get any matching zones from the server, so redirect
        // to the user profile page where the user can choose something appropriate
        if (!len) {
            banner.redirect(banner.$link.attr('href'));
            return;
        }

        // If there's only one matching zone, just use it without showing a list
        if (len === 1) {
            banner.setUserTimeZone(zones[0].timeZoneId, 'banner');
            return;
        }

        // There's more than one matching zone, so show a list
        if (!banner.$dropdown) {
            // Create an AUI dropdown
            var id = 'timezone-selection-list';
            banner.$link
                .addClass('aui-dropdown2-trigger')
                .attr('aria-owns', id)
                .attr('aria-haspopup', 'true');

            banner.$dropdown =
            $(Templates.dropdown({
                id: id,
                sections: banner.getListSections(zones),
                baseHref: banner.profileLink()
            }))
                .css('z-index', 5000) // Show on top of the aui flag
                .on('click', 'a', normalLeftClickOnly(function (e) {
                    var $this = $(this);
                    var tzid = $this.attr('data-tzid');
                    if (tzid) {
                        e.preventDefault();
                        banner.setUserTimeZone(tzid, 'menu');
                    } else if ($this.attr('data-tzother')) {
                        // Gather the detected offsets for analytics so we can improve the options we give
                        var attrs = {
                            offsets: [banner.detected.janOffset, banner.detected.julyOffset].join(',')
                        };
                        banner.track('menu.other', attrs);
                    }
                }))
                .appendTo('body');
        }

        banner.$dropdown.trigger('aui-button-invoke');
    };

    banner.filterZoneData = function (zones) {
        var ignoreRegions = ['Antarctica', 'Etc'];
        return _.filter(zones, function (zone) {
            return !_.contains(ignoreRegions, zone.regionKey);
        });
    };

    banner.getListSections = function (zones) {
        var sections = [];
        var region, currentRegion, currentSection;
        for (var i = 0, ii = zones.length; i < ii; i++) {
            region = zones[i].regionKey;
            // Start a new section if region key has changed
            if (region !== currentRegion) {
                currentSection = {
                    region: zones.regions[region] || region,
                    zones: []
                };
                sections.push(currentSection);
                currentRegion = region;
            }
            // Add zone data to current section list
            currentSection.zones.push(zones[i]);
        }
        // Add an "other" selection to the end of the list
        sections.push({
            other: formatter.I18n.getText('tz.other')
        });
        return sections;
    };

    banner.getTzOffset = function (month) {
        var year = new Date().getFullYear();
        var date = new Date(year, month, 1, 12, 0, 0);
        // getTimezoneOffset() returns (UTC - zone) in minutes, but we want (zone - UTC) instead
        return -date.getTimezoneOffset();
    };



    banner.getPotentialZones = function () {
        var deferred = new Deferred();
        if (banner.zoneList) {
            deferred.resolve(banner.zoneList);
        } else {
            var data = {
                janOffset: minsToMillis(banner.detected.janOffset),
                julyOffset: minsToMillis(banner.detected.julyOffset)
            };
            SmartAjax.makeRequest({
                url: restPath + '/zones',
                type: 'GET',
                data: data,
                contentType: 'application/json',
                complete: function (xhr, status, smartAjaxResult) {
                    if (status != 'abort' && smartAjaxResult.successful) {
                        banner.zoneList = smartAjaxResult.data;
                        deferred.resolve(banner.zoneList);
                    } else {
                        deferred.reject(smartAjaxResult);
                    }
                }
            });
        }

        return deferred.promise();
    };

    banner.setUserTimeZone = function (tzid, source) {
        banner.track('setzone', {zoneId: tzid, source: source});
        SmartAjax.makeRequest({
            url: restPath + '/update',
            type: 'POST',
            data: tzid,
            contentType: 'application/json',
            complete: function (xhr, status, smartAjaxResult) {
                if (status != 'abort' && smartAjaxResult.successful) {
                    if (banner.$tzFlag) {
                        banner.$tzFlag.find('.aui-icon.icon-close').click();
                    }
                    var data = smartAjaxResult.data;
                    var tzMsg = formatter.I18n.getText('tz.updated', data.gmtOffset + ' ' + data.city);
                    flag.showSuccessMsg('', tzMsg, {close: 'manual'});
                }
            }
        });
    };

    return banner;
});
